package eu.dnetlib.data.mapreduce.hbase.dedup.gt;

import java.io.IOException;
import java.util.Map;

import org.apache.hadoop.hbase.client.Delete;
import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.io.ImmutableBytesWritable;
import org.apache.hadoop.hbase.mapreduce.TableMapper;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.mapreduce.Counter;

import com.google.protobuf.InvalidProtocolBufferException;

import eu.dnetlib.data.mapreduce.util.DedupUtils;
import eu.dnetlib.data.mapreduce.util.OafRowKeyDecoder;
import eu.dnetlib.data.proto.KindProtos.Kind;
import eu.dnetlib.data.proto.OafProtos.Oaf;
import eu.dnetlib.data.proto.OafProtos.OafEntity;
import eu.dnetlib.data.proto.PersonProtos.Person;
import eu.dnetlib.data.proto.TypeProtos.Type;

/**
 * Removes the Non-root rows
 *
 *
 * @author claudio
 *
 */
public class GTCleanerMapper extends TableMapper<ImmutableBytesWritable, Delete> {

	@Override
	protected void map(final ImmutableBytesWritable keyIn, final Result value, final Context context) throws IOException, InterruptedException {

		final OafRowKeyDecoder keyDecoder = OafRowKeyDecoder.decode(keyIn.copyBytes());

		final Type type = keyDecoder.getType();

		if (!type.equals(Type.person)) {
			incrementCounter(context, "wrong entity type", type.toString(), 1);
			return;
		}

		final Map<byte[], byte[]> map = value.getFamilyMap(Bytes.toBytes(type.toString()));
		final byte[] bodyB = map.get(DedupUtils.BODY_B);
		if (bodyB == null) {
			incrementCounter(context, "missing body (map)", type.toString(), 1);
			return;
		}

		final Oaf oaf = decodeProto(context, bodyB);

		if (!isValid(oaf)) {
			incrementCounter(context, "missing body (map)", type.toString(), 1);
			return;
		}

		if (mergedSize(oaf, 0) || mergedSize(oaf, 1)) {
			context.write(keyIn, new Delete(keyIn.copyBytes()));
			incrementCounter(context, Kind.entity.toString(), "deleted", 1);
		}

	}

	private boolean mergedSize(final Oaf oaf, final int size) {
		final OafEntity entity = oaf.getEntity();

		if (entity == null) return false;

		final Person person = entity.getPerson();

		return (person.getMergedpersonList() != null) && (person.getMergedpersonList().size() == size);
	}

	private boolean isValid(final Oaf oaf) {
		return (oaf != null) && oaf.isInitialized();
	}

	private Oaf decodeProto(final Context context, final byte[] body) {
		try {
			return Oaf.parseFrom(body);
		} catch (final InvalidProtocolBufferException e) {
			e.printStackTrace(System.err);
			context.getCounter("decodeProto", e.getClass().getName()).increment(1);
		}
		return null;
	}

	private void incrementCounter(final Context context, final String k, final String t, final int n) {
		getCounter(context, k, t).increment(n);
	}

	private Counter getCounter(final Context context, final String k, final String t) {
		return context.getCounter(k, t);
	}

}
