package eu.dnetlib.data.mapreduce.hbase.broker.mapping;

import java.io.StringReader;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.google.common.base.Function;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import eu.dnetlib.broker.objects.*;
import eu.dnetlib.data.proto.FieldTypeProtos.StructuredProperty;
import eu.dnetlib.data.proto.OafProtos.Oaf;
import eu.dnetlib.data.proto.OafProtos.OafEntity;
import eu.dnetlib.data.proto.ProjectProtos;
import eu.dnetlib.data.proto.ResultProtos;
import eu.dnetlib.data.proto.ResultProtos.Result;
import org.apache.commons.lang.StringUtils;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.io.SAXReader;

import static eu.dnetlib.data.mapreduce.util.OafHbaseUtils.getKey;
import static eu.dnetlib.data.mapreduce.util.OafHbaseUtils.getValue;

/**
 * Created by claudio on 22/07/16.
 */
public abstract class ProtoMapping {

	protected static List<Instance> mapInstances(final List<Result.Instance> in) {
		final Set<Instance> instances = Sets.newHashSet(Iterables.transform(in, new Function<Result.Instance, Instance>() {

			@Override
			public Instance apply(final Result.Instance i) {
				return new Instance()
						.setHostedby(getValue(i.getHostedby()))
						.setInstancetype(getValue(i.getInstancetype()))
						.setLicense(getKey(i.getLicence()))
						.setUrl(Iterables.getFirst(i.getUrlList(), ""));
			}
		}));
		return Lists.newArrayList(instances);
	}

	protected static List<Pid> mapPids(final List<StructuredProperty> sp) {
		return Lists.newArrayList(Iterables.transform(sp, new Function<StructuredProperty, Pid>() {

			@Override
			public Pid apply(final StructuredProperty sp) {
				return new Pid().setType(sp.getQualifier().getClassid()).setValue(sp.getValue());
			}
		}));
	}

	protected static Journal mapJournal(final ResultProtos.Result.Journal j) {
		return new eu.dnetlib.broker.objects.Journal()
				.setName(j.getName())
				.setIssn(j.getIssnPrinted())
				.setEissn(j.getIssnOnline())
				.setLissn(j.getIssnLinking());
	}

	protected static List<ExternalReference> mapExternalRefs(final List<Result.ExternalReference> ext) {
		return Lists.newArrayList(Iterables.transform(ext, new Function<Result.ExternalReference, ExternalReference>() {

			@Override
			public ExternalReference apply(final Result.ExternalReference e) {
				return new ExternalReference()
						.setUrl(e.getUrl())
						.setType(getKey(e.getQualifier()))
						.setRefidentifier(e.getRefidentifier())
						.setSitename(e.getSitename());
			}
		}));
	}

	protected static final List<Project> mapRelatedProjects(final OafEntity entity) {

		final Map<String, Oaf> projectMap = Maps.newHashMap();
		for(Oaf rel : entity.getCachedOafRelList()) {
			final OafEntity p = rel.getRel().getCachedOafTarget().getEntity();
			projectMap.put(p.getId(), Oaf.newBuilder(rel).build());
		}

		return Lists.transform(Lists.newArrayList(projectMap.values()), getProjectMappingFunction());
	}

	protected static final Project mapRelatedProject(final ProjectProtos.Project project) {
		final Project p = new Project();
		final ProjectProtos.Project.Metadata mp = project.getMetadata();

		p.setCode(getValue(mp.getCode()));
		p.setAcronym(getValue(mp.getAcronym()));
		p.setTitle(getValue(mp.getTitle()));

		final String ftree = getValue(mp.getFundingtreeList());
		if (StringUtils.isNotBlank(ftree)) {
			try {
				final Document fdoc = new SAXReader().read(new StringReader(ftree));
				p.setFunder(fdoc.valueOf("/fundingtree/funder/shortname"));
				p.setJurisdiction(fdoc.valueOf("/fundingtree/funder/jurisdiction"));
				p.setFundingProgram(fdoc.valueOf("//funding_level_0/name"));
			} catch (final DocumentException e) {
				throw new RuntimeException(e);
			}
		}
		//System.out.println("ProtoMapping.mapRelatedProjects project = " + project.toJSON());
		return p;
	}

	private static Function<Oaf, Project> getProjectMappingFunction() {
		return new Function<Oaf, Project>() {

			@Override
			public Project apply(final Oaf oafRel) {
				return mapRelatedProject(oafRel.getRel().getCachedOafTarget().getEntity().getProject());
			}
		};
	}

}
