package eu.dnetlib.data.mapreduce.hbase.broker.enrich;

import java.io.IOException;
import java.util.List;
import java.util.Map;

import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import eu.dnetlib.data.mapreduce.hbase.broker.ProjectEventFactory;
import eu.dnetlib.data.proto.OafProtos.Oaf;
import org.apache.commons.lang.StringUtils;
import org.apache.hadoop.hbase.io.ImmutableBytesWritable;
import org.dom4j.DocumentException;

import static eu.dnetlib.data.mapreduce.util.OafHbaseUtils.getKey;

/**
 * Created by claudio on 08/07/16.
 */
public class ProjectEnrichmentReducer extends AbstractEnrichmentReducer {

	@Override
	protected String counterGroup() {
		return "Broker Enrichment projects";
	}

	@Override
	protected void reduce(final ImmutableBytesWritable key, final Iterable<ImmutableBytesWritable> values, final Context context)
			throws IOException, InterruptedException {

		final Map<String, Oaf> projects = Maps.newHashMap();
		final List<Oaf> pubs = Lists.newArrayList();

		for (final Oaf oaf : Iterables.transform(Iterables.limit(values, LIMIT), oafDeserialiser())) {

			switch (oaf.getEntity().getType()) {
			case result:
				pubs.add(oaf);
				break;
			case project:
				projects.put(oaf.getEntity().getId(), oaf);
				break;
			default:
				throw new IllegalArgumentException("invalid type: " + oaf.getEntity().getType());
			}
		}
		if (projects.isEmpty() || pubs.isEmpty()) {
			context.getCounter(counterGroup(), "Events Skipped - Nothing to do").increment(1);
			return;
		}

		try {
			generateEvents(pubs, projects, context);
		} catch (final DocumentException e) {
			throw new RuntimeException(e);
		}
	}

	private void generateEvents(final List<Oaf> pubs, final Map<String, Oaf> projects, final Context context)
			throws IOException, InterruptedException, DocumentException {

		final List<Oaf> enrichedOaf = Lists.newArrayList(Iterables.transform(pubs, new Function<Oaf, Oaf>() {
			@Override
			public Oaf apply(final Oaf pub) {
				return addProjects(pub, projects);
			}
		}));

		final List<Oaf> valid = Lists.newArrayList(Iterables.filter(enrichedOaf, new Predicate<Oaf>() {
			@Override
			public boolean apply(final Oaf result) {
				return !result.getEntity().getCachedOafRelList().isEmpty();
			}
		}));

		if (valid.isEmpty()) {
			context.getCounter(counterGroup(), "Events Skipped - Missing project").increment(1);
			return;
		}

		for (final Oaf current : valid) {

			final String currentId = current.getEntity().getId();

			final String currentDsId = StringUtils.substringAfter(getKey(current.getEntity().getCollectedfromList()), "|");
			final String currentDsType = dsTypeMap.get(currentDsId);

			// System.out.println(String.format("'%s' -> '%s'", currentDsId, currentDsType));

			if (StringUtils.isBlank(currentDsType) && !dsWhitelist.contains(currentDsId)) {
				context.getCounter(counterGroup(), "Events Skipped - Datasource type excluded").increment(1);
			} else if (dsBlacklist.contains(currentDsId)) {
				context.getCounter(counterGroup(), "Events Skipped - Datasource blacklisted").increment(1);
			} else {
				if (valid.size() == 1) {
					ProjectEventFactory.process(context, valid.get(0));
					return;
				}

				for (final Oaf other : valid) {
					final String otherId = other.getEntity().getId();
					if (!currentId.equals(otherId)) {

						final float trust = similarity(current, other);
						ProjectEventFactory.process(context, current, other, trust);
					}
				}
			}
		}

	}

	private Oaf addProjects(final Oaf current, final Map<String, Oaf> projects) {

		final Oaf.Builder oafBuilder = Oaf.newBuilder(current);
		final List<Oaf> cachedRels = Lists.newArrayList();

		for (final Oaf.Builder cachedOafRel : oafBuilder.getEntityBuilder().getCachedOafRelBuilderList()) {
			final String projectId = cachedOafRel.getRel().getTarget();

			if (projects.containsKey(projectId)) {
				final Oaf project = projects.get(projectId);

				cachedOafRel.getRelBuilder().setCachedOafTarget(project);
				cachedRels.add(cachedOafRel.build());
			}
		}

		oafBuilder.getEntityBuilder().clearCachedOafRel().addAllCachedOafRel(cachedRels);

		return oafBuilder.build();
	}

}
