package eu.dnetlib.data.mapreduce.hbase.broker;

import java.io.IOException;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;
import eu.dnetlib.broker.objects.OpenAireEventPayload;
import eu.dnetlib.data.mapreduce.hbase.broker.mapping.HighlightFactory;
import eu.dnetlib.data.mapreduce.hbase.broker.mapping.OpenAireEventPayloadFactory;
import eu.dnetlib.data.mapreduce.hbase.broker.model.EventMessage;
import eu.dnetlib.data.proto.OafProtos.Oaf;
import eu.dnetlib.data.proto.ResultProtos.Result.Instance;
import org.apache.commons.lang.StringUtils;
import org.apache.hadoop.io.Text;
import org.apache.hadoop.mapreduce.Reducer.Context;
import org.dom4j.DocumentException;

import static eu.dnetlib.data.mapreduce.hbase.broker.mapping.EventFactory.asEvent;
import static eu.dnetlib.data.mapreduce.util.OafHbaseUtils.getKey;

/**
 * Created by claudio on 26/07/16.
 */
public class OAVersionEventFactory {

	protected Text tKey = new Text("");

	public static void process(final Context context, final Oaf current, final Oaf other, final float trust, final Set<String> untrustedOaDsList)
			throws IOException, InterruptedException, DocumentException {
		new OAVersionEventFactory().processOAVersion(context, current, other, trust, untrustedOaDsList);
	}

	private void processOAVersion(final Context context, final Oaf current, final Oaf other, final float trust, final Set<String> untrustedOaDsList)
			throws IOException, InterruptedException, DocumentException {
		final String otherDsId = StringUtils.substringAfter(getKey(other.getEntity().getCollectedfromList()), "|");

		// OPEN ACCESS VERSION
		if (!untrustedOaDsList.contains(otherDsId)) {

			if (hasAccess(other, "OPEN", false)) {

				// MORE
				doProcessOAVersion(context, current, other, Topic.ENRICH_MORE_OA_VERSION, trust);

				// MISSING
				if (!hasAccess(current, "OPEN", true)) {
					doProcessOAVersion(context, current, other, Topic.ENRICH_MISSING_OA_VERSION, trust);
				}
			}
		}

	}

	private void doProcessOAVersion(final Context context, final Oaf current, final Oaf other, final Topic topic, final float trust)
			throws IOException, InterruptedException, DocumentException {
		final Oaf.Builder prototype = Oaf.newBuilder(current);

		final List<Instance> instances =
				other.getEntity().getResult().getInstanceList().stream()
						.filter(i ->"OPEN".equalsIgnoreCase(i.getAccessright().getClassid()))
						.collect(Collectors.toList());
		prototype.getEntityBuilder().getResultBuilder().addAllInstance(instances);

		final Oaf oaf = prototype.build();

		final OpenAireEventPayload payload =
				HighlightFactory.highlightEnrichOa(OpenAireEventPayloadFactory.fromOAF(oaf.getEntity(), other.getEntity(), trust), instances);
		final EventMessage event = asEvent(oaf.getEntity(), topic, payload, other.getEntity(), trust);

		context.write(tKey, new Text(event.toString()));
		context.getCounter("event", topic.getValue()).increment(1);
	}

	private boolean hasAccess(final Oaf oaf, final String access, final boolean strict) {
		final Predicate<Instance> p = i -> access.equalsIgnoreCase(i.getAccessright().getClassid());
		final List<Instance> i = oaf.getEntity().getResult().getInstanceList();
		return strict ? Iterables.all(i, p) : Iterables.any(i, p);
	}

}
