package eu.dnetlib.data.mapreduce.hbase.broker.mapping;

import java.io.StringReader;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import com.google.common.collect.Iterables;
import com.google.common.collect.Maps;
import eu.dnetlib.broker.objects.*;
import eu.dnetlib.data.proto.FieldTypeProtos.StructuredProperty;
import eu.dnetlib.data.proto.OafProtos.Oaf;
import eu.dnetlib.data.proto.OafProtos.OafEntity;
import eu.dnetlib.data.proto.ProjectProtos;
import eu.dnetlib.data.proto.ResultProtos;
import eu.dnetlib.data.proto.ResultProtos.Result;
import org.apache.commons.lang.StringUtils;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.io.SAXReader;

import static eu.dnetlib.data.mapreduce.util.OafHbaseUtils.getKey;
import static eu.dnetlib.data.mapreduce.util.OafHbaseUtils.getValue;

/**
 * Created by claudio on 22/07/16.
 */
public abstract class ProtoMapping {

	protected static List<Instance> mapInstances(final Collection<Result.Instance> in) {
		return in.stream()
				.map(i -> new Instance()
						.setHostedby(getValue(i.getHostedby()))
						.setInstancetype(getValue(i.getInstancetype()))
						.setLicense(getKey(i.getAccessright()))
						.setUrl(Iterables.getFirst(i.getUrlList(), "")))
				.collect(Collectors.toList());
	}

	protected static List<Pid> mapPids(final List<StructuredProperty> sp) {
		return sp.stream()
				.map(s -> new Pid().setType(s.getQualifier().getClassid()).setValue(s.getValue()))
				.collect(Collectors.toList());
	}

	protected static Journal mapJournal(final ResultProtos.Result.Journal j) {
		return new eu.dnetlib.broker.objects.Journal()
				.setName(j.getName())
				.setIssn(j.getIssnPrinted())
				.setEissn(j.getIssnOnline())
				.setLissn(j.getIssnLinking());
	}

	protected static List<ExternalReference> mapExternalRefs(final List<Result.ExternalReference> ext) {
		return ext.stream()
				.map(e -> new ExternalReference()
						.setUrl(e.getUrl())
						.setType(getKey(e.getQualifier()))
						.setRefidentifier(e.getRefidentifier())
						.setSitename(e.getSitename()))
				.collect(Collectors.toList());
	}

	protected static final List<Project> mapRelatedProjects(final OafEntity entity) {

		final Map<String, Oaf> projectMap = Maps.newHashMap();
		for(Oaf rel : entity.getCachedOafRelList()) {
			final OafEntity p = rel.getRel().getCachedOafTarget().getEntity();
			projectMap.put(p.getId(), Oaf.newBuilder(rel).build());
		}

		return projectMap.values().stream()
				.map(o -> mapRelatedProject(o.getRel().getCachedOafTarget().getEntity().getProject()))
				.collect(Collectors.toList());
	}

	protected static final Project mapRelatedProject(final ProjectProtos.Project project) {
		final Project p = new Project();
		final ProjectProtos.Project.Metadata mp = project.getMetadata();

		p.setCode(getValue(mp.getCode()));
		p.setAcronym(getValue(mp.getAcronym()));
		p.setTitle(getValue(mp.getTitle()));

		final String ftree = getValue(mp.getFundingtreeList());
		if (StringUtils.isNotBlank(ftree)) {
			try {
				final Document fdoc = new SAXReader().read(new StringReader(ftree));
				p.setFunder(fdoc.valueOf("/fundingtree/funder/shortname"));
				p.setJurisdiction(fdoc.valueOf("/fundingtree/funder/jurisdiction"));
				p.setFundingProgram(fdoc.valueOf("//funding_level_0/name"));
			} catch (final DocumentException e) {
				throw new RuntimeException(e);
			}
		}
		//System.out.println("ProtoMapping.mapRelatedProjects project = " + project.toJSON());
		return p;
	}

	protected static final Software mapRelatedSoftware(final ResultProtos.Result result) {
		final Software s = new Software();
		final ResultProtos.Result.Metadata rp = result.getMetadata();

		s.setName(getValue(rp.getTitleList()));
		s.setDescription(getValue(rp.getDescriptionList()));
		s.setLandingPage(getValue(rp.getCodeRepositoryUrl()));

		if (result.getInstanceList() != null && result.getInstanceList().size() > 0) {
			s.setRepository(getValue(result.getInstanceList().get(0).getUrlList()));
		}

		//System.out.println("ProtoMapping.mapRelatedProjects project = " + project.toJSON());
		return s;
	}
}
