package eu.dnetlib.data.mapreduce.hbase.broker.enrich;

import java.io.IOException;
import java.util.Arrays;
import java.util.Map;

import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import eu.dnetlib.data.mapreduce.util.DedupUtils;
import eu.dnetlib.data.mapreduce.util.OafDecoder;
import eu.dnetlib.data.mapreduce.util.OafRowKeyDecoder;
import eu.dnetlib.data.proto.OafProtos.Oaf;
import eu.dnetlib.data.proto.TypeProtos.Type;
import org.apache.commons.collections.MapUtils;
import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.io.ImmutableBytesWritable;
import org.apache.hadoop.hbase.util.Bytes;

/**
 * Created by claudio on 08/07/16.
 */
public class ProjectEnrichmentMapper extends AbstractEnrichmentMapper {

	@Override
	protected String counterGroup() {
		return "Broker Enrichment projects";
	}

	@Override
	protected void map(final ImmutableBytesWritable key, final Result value, final Context context) throws IOException, InterruptedException {

		final Type type = getEntityType(key);

		final byte[] body = value.getValue(Bytes.toBytes(type.toString()), Bytes.toBytes("body"));

		if (body == null) {
			context.getCounter(counterGroup(), "missing metadata").increment(1);
			return;
		}

		switch (type) {
		case project:
			for (final byte[] resultId : listRelatedIds(value, "resultProject_outcome_produces")) {
				emit(context, resultId, body, "project");
			}
			break;
		case result:

			final Oaf.Builder oafBuilder = Oaf.newBuilder(OafDecoder.decode(body).getOaf());
			for (final String relName : Arrays.asList("resultProject_outcome_isProducedBy")) { // TODO add dataset rels
				for (final Oaf rel : listRelations(value, relName)) {
					oafBuilder.getEntityBuilder().addCachedOafRel(rel);
					context.getCounter(counterGroup(), "rel: " + relName).increment(1);
				}
			}

			final Map<byte[], byte[]> mergedIn = value.getFamilyMap(DedupUtils.getDedupCF_mergedInBytes(Type.result));

			if (MapUtils.isEmpty(mergedIn) & !DedupUtils.isRoot(key)) {
				emit(context, key.copyBytes(), oafBuilder.build().toByteArray(), "result not deduped");
			} else if (DedupUtils.isRoot(key)) {
				emit(context, key.copyBytes(), oafBuilder.build().toByteArray(), "result merges");
			} else {
				emit(context, getRootId(mergedIn), oafBuilder.build().toByteArray(), "result mergedIn");
			}

			break;
		default:
			throw new IllegalArgumentException("invalid type: " + type);
		}
	}

	private Iterable<Oaf> listRelations(final Result value, final String relType) {

		//TODO consider only relationshipt not deletedbyinference

		final Map<byte[], byte[]> map = value.getFamilyMap(Bytes.toBytes(relType));
		if (MapUtils.isNotEmpty(map)) {
			return Iterables.filter(Iterables.transform(map.values(), new Function<byte[], Oaf>() {

				@Override
				public Oaf apply(final byte[] input) {
					return OafDecoder.decode(input).getOaf();
				}
			}), new Predicate<Oaf>() {
				@Override
				public boolean apply(final Oaf rel) {
					return !rel.getRel().getTarget().contains("unidentified");
				}
			});
		} else {
			return Lists.newArrayList();
		}
	}

	private Iterable<byte[]> listRelatedIds(final Result value, final String relType) {

		//TODO consider only relationshipt not deletedbyinference

		final Map<byte[], byte[]> map = value.getFamilyMap(Bytes.toBytes(relType));
		if (MapUtils.isNotEmpty(map)) {
			return map.keySet();
		} else {
			return Lists.newArrayList();
		}
	}

	private Type getEntityType(final ImmutableBytesWritable key) {
		return OafRowKeyDecoder.decode(key.copyBytes()).getType();
	}

	private byte[] getRootId(final Map<byte[], byte[]> mergedIn) {
		return Iterables.getOnlyElement(mergedIn.keySet());
	}

}
