package eu.dnetlib.data.mapreduce.hbase.dataimport;

import java.io.IOException;
import java.lang.reflect.Type;
import java.util.Map;

import com.google.common.collect.Maps;
import com.google.gson.Gson;
import com.google.gson.reflect.TypeToken;
import eu.dnetlib.data.mapreduce.JobParams;
import eu.dnetlib.data.transform.Column;
import eu.dnetlib.data.transform.Row;
import eu.dnetlib.data.transform.XsltRowTransformer;
import eu.dnetlib.data.transform.XsltRowTransformerFactory;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.hbase.client.Durability;
import org.apache.hadoop.hbase.client.Put;
import org.apache.hadoop.hbase.io.ImmutableBytesWritable;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.io.Text;
import org.apache.hadoop.mapreduce.Mapper;

public class ImportRecordsMapper extends Mapper<Text, Text, ImmutableBytesWritable, Put> {

	/**
	 * logger.
	 */
	private static final Log log = LogFactory.getLog(ImportRecordsMapper.class); // NOPMD by marko on 11/24/08 5:02 PM

	private XsltRowTransformer transformer;

	private ImmutableBytesWritable ibw;

	@Override
	protected void setup(final Context context) throws IOException, InterruptedException {
		super.setup(context);

		final String xslt = context.getConfiguration().get(JobParams.HBASE_IMPORT_XSLT).trim();

		if ((xslt == null) || xslt.isEmpty()) throw new IllegalArgumentException("missing xslt");
		final Map<String, Object> xslParams = Maps.newHashMap();

		boolean invisible = context.getConfiguration().getBoolean(JobParams.INVISIBLE, false);
		log.info("invisible flag: " + invisible);
		xslParams.put("invisible", invisible);

		transformer = XsltRowTransformerFactory.newInstance(xslt, xslParams);

		ibw = new ImmutableBytesWritable();

		log.info("got xslt: '" + xslt);
		log.info("using trasformer: '" + transformer.getTransformerClassName() + "'");
	}

	@Override
	protected void map(final Text key, final Text value, final Context context) throws IOException, InterruptedException {
		try {
			for (final Row row : transformer.apply(value.toString())) {

				final byte[] rowKey = Bytes.toBytes(row.getKey());
				final Put put = new Put(rowKey);
				put.setDurability(Durability.SKIP_WAL);

				for (final Column<String, byte[]> col : row) {
					final byte[] family = Bytes.toBytes(row.getColumnFamily());
					final byte[] qualifier = Bytes.toBytes(col.getName());
					put.add(family, qualifier, col.getValue());
				}

				ibw.set(rowKey);
				context.write(ibw, put);
				context.getCounter("mdstore", row.getColumnFamily()).increment(row.getColumns().size());
			}
		} catch (final Throwable e) {
			log.error("error importing the following record on HBase: " + value.toString(), e);
			context.getCounter("error", e.getClass().getName()).increment(1);
			throw new RuntimeException(e);
		}
	}


}
