package eu.dnetlib.data.mapreduce.hbase.dedup.gt;

import java.io.IOException;
import java.util.Map;

import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.io.ImmutableBytesWritable;
import org.apache.hadoop.hbase.mapreduce.TableMapper;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.io.Text;
import org.apache.hadoop.mapreduce.Counter;

import com.google.protobuf.InvalidProtocolBufferException;
import com.googlecode.protobuf.format.JsonFormat;

import eu.dnetlib.data.mapreduce.util.DedupUtils;
import eu.dnetlib.data.mapreduce.util.OafRowKeyDecoder;
import eu.dnetlib.data.proto.KindProtos.Kind;
import eu.dnetlib.data.proto.OafProtos.Oaf;
import eu.dnetlib.data.proto.TypeProtos.Type;

public class RootExportMapper extends TableMapper<Text, Text> {

	private Text outKey;

	private Text outValue;

	@Override
	protected void setup(final Context context) throws IOException, InterruptedException {
		outKey = new Text("");
		outValue = new Text();
	}

	@Override
	protected void map(final ImmutableBytesWritable keyIn, final Result value, final Context context) throws IOException, InterruptedException {

		final OafRowKeyDecoder keyDecoder = OafRowKeyDecoder.decode(keyIn.copyBytes());

		final Type type = keyDecoder.getType();
		if (!DedupUtils.isRoot(keyDecoder.getId())) return;

		final Map<byte[], byte[]> map = value.getFamilyMap(Bytes.toBytes(type.toString()));
		final byte[] bodyB = map.get(DedupUtils.BODY_B);
		if (bodyB == null) {
			incrementCounter(context, "missing body (map)", type.toString(), 1);
			return;
		}

		final Oaf oaf = decodeProto(context, bodyB);

		if (!isValid(oaf)) {
			incrementCounter(context, "missing body (map)", type.toString(), 1);
			return;
		}

		if (oaf.getDataInfo().getDeletedbyinference()) {
			incrementCounter(context, "deleted by inference", type.toString(), 1);
			return;
		}

		emit(new String(keyIn.copyBytes()), context, oaf);
		incrementCounter(context, Kind.entity.toString(), getEntityType(oaf, type), 1);
	}

	private boolean isValid(final Oaf oaf) {
		return (oaf != null) && oaf.isInitialized();
	}

	private Oaf decodeProto(final Context context, final byte[] body) {
		try {
			return Oaf.parseFrom(body);
		} catch (final InvalidProtocolBufferException e) {
			e.printStackTrace(System.err);
			context.getCounter("decodeProto", e.getClass().getName()).increment(1);
		}
		return null;
	}

	private void emit(final String key, final Context context, final Oaf oaf) throws IOException, InterruptedException {
		// outKey.set(key);
		outValue.set(new JsonFormat().printToString(oaf));

		context.write(outKey, outValue);
	}

	private void incrementCounter(final Context context, final String k, final String t, final int n) {
		getCounter(context, k, t).increment(n);
	}

	private Counter getCounter(final Context context, final String k, final String t) {
		return context.getCounter(k, t);
	}

	private String getEntityType(final Oaf oaf, final Type type) {
		switch (type) {
		case result:
			return oaf.getEntity().getResult().getMetadata().getResulttype().getClassid();
		default:
			return type.toString();
		}
	}

}
