package eu.dnetlib.data.mapreduce.hbase.dataimport;

import com.google.gson.Gson;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.googlecode.protobuf.format.JsonFormat;
import eu.dnetlib.actionmanager.actions.ActionFactory;
import eu.dnetlib.actionmanager.actions.AtomicAction;
import eu.dnetlib.actionmanager.common.Agent;
import eu.dnetlib.data.mapreduce.util.StreamUtils;
import eu.dnetlib.data.proto.*;
import eu.dnetlib.data.transform.xml.AbstractDNetXsltFunctions;
import eu.dnetlib.miscutils.collections.Pair;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import java.io.IOException;
import java.io.InputStream;
import java.util.*;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

import static eu.dnetlib.data.mapreduce.hbase.dataimport.DumpToActionsUtility.*;
import static eu.dnetlib.data.proto.ResultOrganizationProtos.ResultOrganization;


public class DOIBoostToActions {

    private static Map<String, Pair<String, String>> datasources =  new HashMap<String, Pair<String, String>>() {{
        put("MAG", new Pair<>("Microsoft Academic Graph", "openaire____::microsoft"));
        put("ORCID", new Pair<>("ORCID", "openaire____::orcid"));
        put("CrossRef", new Pair<>("Crossref", "openaire____::crossref"));
        put("UnpayWall", new Pair<>("UnpayWall", "openaire____::unpaywall"));

    }};

    private static Map<String, FieldTypeProtos.Qualifier> affiliationPIDType =  new HashMap<String, FieldTypeProtos.Qualifier>() {{
        put("MAG", FieldTypeProtos.Qualifier.newBuilder().setClassid("mag_id" ).setClassname("Microsoft Academic Graph Identifier").setSchemename("dnet:pid_types").setSchemeid("dnet:pid_types").build());
        put("grid.ac", getQualifier("grid", "dnet:pid_types"));
        put("wikpedia", getQualifier("urn", "dnet:pid_types"));
    }};

    static Map<String, Map<String, String>> typologiesMapping;

    static {
        try {
            final InputStream is = DOIBoostToActions.class.getResourceAsStream("/eu/dnetlib/data/mapreduce/hbase/dataimport/mapping_typologies.json");
            final String tt =IOUtils.toString(is);
            typologiesMapping = new Gson().fromJson(tt, Map.class);
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    final static String doiBoostNSPREFIX ="doiboost____";


    public static List<AtomicAction> generatePublicationActionsFromDump(final JsonObject rootElement, final ActionFactory factory, final String setName, final Agent agent, boolean invisible) {

        //Create OAF Proto

        final OafProtos.Oaf.Builder oaf = OafProtos.Oaf.newBuilder();
        //Add Data Info
        oaf.setDataInfo(FieldTypeProtos.DataInfo.newBuilder()
                .setInvisible(invisible)
                .setDeletedbyinference(false)
                .setInferred(false)
                .setTrust("0.9")
                .setProvenanceaction(getQualifier("sysimport:actionset", "dnet:provenanceActions"))
                .build());

        //Adding Kind
        oaf.setKind(KindProtos.Kind.entity);

        //creating Result Proto
        final OafProtos.OafEntity.Builder entity = OafProtos.OafEntity.newBuilder().setType(TypeProtos.Type.result);

        entity.setDateofcollection("2018-10-10");

        if (rootElement.has("collectedFrom") && rootElement.get("collectedFrom").isJsonArray()){
            StreamUtils.toStream(rootElement.getAsJsonArray("collectedFrom").iterator())
                    .map(JsonElement::getAsString)
                    .forEach(cf ->
                            {
                                final String id =datasources.get(cf).getValue();
                                final String name =datasources.get(cf).getKey();
                                if (StringUtils.isNotBlank(id) && StringUtils.isNotBlank(name)) {
                                    final FieldTypeProtos.KeyValue collectedFrom = FieldTypeProtos.KeyValue.newBuilder()
                                            .setValue(name)
                                            .setKey("10|openaire____::" + AbstractDNetXsltFunctions.md5(StringUtils.substringAfter(id, "::")))
                                            .build();
                                    entity.addCollectedfrom(collectedFrom);
                                }
                            }
                    );
        }
        //Adding identifier
        final String doi = getStringValue(rootElement, "doi");
        if (doi == null)
            return null;
        final String sourceId = String.format("50|%s::%s", doiBoostNSPREFIX, AbstractDNetXsltFunctions.md5(doi));
        entity.setId(sourceId);

        entity.addPid(FieldTypeProtos.StructuredProperty.newBuilder()
                .setValue(doi)
                .setQualifier(getQualifier("doi", "dnet:pid_types"))
                .build());


        //Create Result Field
        ResultProtos.Result.Builder result = ResultProtos.Result.newBuilder();

        final String type = getStringValue(rootElement,"type");

        if (!typologiesMapping.containsKey(type))
            return null;

        //Adding Instances
        final String typeValue = typologiesMapping.get(type).get("value");
        final String cobjValue = typologiesMapping.get(type).get("cobj");


        getArrayObjects(rootElement, "instances").stream().map(it ->
                {
                    ResultProtos.Result.Instance.Builder instance= ResultProtos.Result.Instance.newBuilder();
                    instance.setInstancetype(FieldTypeProtos.Qualifier.newBuilder()
                            .setClassid(cobjValue)
                            .setClassname(typeValue)
                            .setSchemeid("dnet:publication_resource")
                            .setSchemename("dnet:publication_resource")
                            .build());
                    instance.setHostedby(FieldTypeProtos.KeyValue.newBuilder()
                            .setKey("10|openaire____::55045bd2a65019fd8e6741a755395c8c")
                            .setValue("Unknown Repository")
                            .build());

                    final String acc_class_id =it.get("access-rights").getAsString();
                    String acc_class_value;
                    switch (acc_class_id){
                        case "OPEN": {
                            acc_class_value = "open access";
                            break;
                        }
                        case "CLOSED": {
                            acc_class_value = "closed access";
                            break;
                        }

                        default: {
                            acc_class_value = "not available";
                        }

                    }

                    instance.addUrl(it.get("url").getAsString());
                    instance.setAccessright(FieldTypeProtos.Qualifier.newBuilder()
                            .setClassid(acc_class_id)
                            .setClassname(acc_class_value)
                            .setSchemeid("dnet:access_modes")
                            .setSchemename("dnet:access_modes")
                            .build());

                    final String id =datasources.get(it.get("provenance").getAsString()).getValue();
                    final String name =datasources.get(it.get("provenance").getAsString()).getKey();
                    if (StringUtils.isNotBlank(id) && StringUtils.isNotBlank(name)) {
                        final FieldTypeProtos.KeyValue collectedFrom = FieldTypeProtos.KeyValue.newBuilder()
                                .setValue(name)
                                .setKey("10|openaire____::" + AbstractDNetXsltFunctions.md5(StringUtils.substringAfter(id, "::")))
                                .build();

                        instance.setCollectedfrom(collectedFrom);
                    }

                    return  instance.build();
                }).forEach(result::addInstance);

        //Adding DOI URL as  Instance
        final String doiURL = getStringValue(rootElement, "doi-url");
        if (StringUtils.isNotBlank(doiURL)) {


        final ResultProtos.Result.Instance.Builder instance = ResultProtos.Result.Instance.newBuilder();
        instance.addUrl(doiURL);
            instance.setAccessright(FieldTypeProtos.Qualifier.newBuilder()
                    .setClassid("CLOSED")
                    .setClassname("Closed Access")
                    .setSchemeid("dnet:access_modes")
                    .setSchemename("dnet:access_modes")
                    .build());
            instance.setCollectedfrom(FieldTypeProtos.KeyValue.newBuilder()
                    .setValue("CrossRef")
                    .setKey("10|openaire____::" + AbstractDNetXsltFunctions.md5("crossref"))
                    .build());
            result.addInstance(instance);
        }

        //Create Metadata Proto
        final ResultProtos.Result.Metadata.Builder metadata = ResultProtos.Result.Metadata.newBuilder();


        Pair<List<FieldTypeProtos.Author>, Collection<OafProtos.Oaf>> authorsOrganizations = createAuthorsOrganization(rootElement);

        if (authorsOrganizations.getKey().size() > 0) {
            metadata.addAllAuthor(authorsOrganizations.getKey());
        }
        //adding Language
        metadata.setLanguage(FieldTypeProtos.Qualifier.newBuilder()
                .setClassid("und")
                .setClassname("Undetermined")
                .setSchemeid("dent:languages")
                .setSchemename("dent:languages")
                .build());

        //Adding subjects
        List<String> subjects =getArrayValues(rootElement, "subject");

        subjects.forEach(s-> metadata.addSubject(FieldTypeProtos.StructuredProperty.newBuilder()
                .setValue(s)
                .setQualifier(getQualifier("keyword", "dnet:subject"))
                .build()));

        List<String>titles =getArrayValues(rootElement, "title");
        titles.forEach(t->
                metadata.addTitle(FieldTypeProtos.StructuredProperty.newBuilder()
                        .setValue(t)
                        .setQualifier(getQualifier("main title", "dnet:dataCite_title"))
                        .build()));
        settingRelevantDate(rootElement, metadata, "issued", "issued", true);
        settingRelevantDate(rootElement, metadata, "accepted", "accepted", false);
        settingRelevantDate(rootElement, metadata, "published-online", "published-online", false);
        settingRelevantDate(rootElement, metadata, "published-print", "published-print", false);


        getArrayObjects(rootElement, "abstract").forEach(d -> metadata.addDescription(FieldTypeProtos.StringField.newBuilder().setValue(d.get("value").getAsString()).build()));



        //Adding Journal
        final String publisher = getStringValue(rootElement,"publisher");
        if (StringUtils.isNotBlank(publisher)){

            final ResultProtos.Result.Journal.Builder journal = ResultProtos.Result.Journal.newBuilder().setName(publisher);

            if (hasJSONArrayField(rootElement,"issn" )){
                StreamUtils.toStream(rootElement.getAsJsonArray("issn").iterator())
                        .map(JsonElement::getAsJsonObject)
                        .forEach(it -> {
                            final String issntype = getStringValue(it, "type");
                            final String value = getStringValue(it, "value");
                            if("electronic".equals(issntype)){
                                journal.setIssnOnline(value);
                            }
                            if ("print".equals(issntype))
                                journal.setIssnPrinted(value);
                        });
            }
            metadata.setJournal(journal.build());
        }
        metadata.setResulttype(getQualifier(getDefaultResulttype(cobjValue), "dnet:result_typologies"));
        result.setMetadata(metadata.build());
        entity.setResult(result.build());
        oaf.setEntity(entity.build());
        final List<AtomicAction> actionList = new ArrayList<>();
        actionList.add(factory.createAtomicAction(setName, agent, oaf.getEntity().getId(), "result", "body", oaf.build().toByteArray()));

        if (!authorsOrganizations.getValue().isEmpty()) {

            authorsOrganizations.getValue().forEach(o ->
                    {
                        actionList.add(factory.createAtomicAction(setName, agent, o.getEntity().getId(), "organization", "body", o.toByteArray()));
                        actionList.addAll(createPublicationOrganizationRelation(oaf.build(), o, factory, setName, agent));
                        final String gridOrganization = getSimilarGridOrganization(o.getEntity());
                        if (gridOrganization!= null) {
                            actionList.add(factory.createAtomicAction(setName, agent, o.getEntity().getId(), "organizationOrganization_dedupSimilarity_isSimilarTo", gridOrganization, "".getBytes()));
                            actionList.add(factory.createAtomicAction(setName, agent, gridOrganization, "organizationOrganization_dedupSimilarity_isSimilarTo", o.getEntity().getId(), "".getBytes()));
                        }
                    });
        }

        return actionList;

    }


    private static String getSimilarGridOrganization(final OafProtos.OafEntity organization) {

        final List<FieldTypeProtos.StructuredProperty> pidList = organization.getPidList();
        if (pidList!= null ) {
            for (FieldTypeProtos.StructuredProperty p: pidList) {
                if (p.getQualifier().getClassname().equals("grid")){
                    return "20|grid________::"+AbstractDNetXsltFunctions.md5(p.getValue());
                }
            }
        }
        return null;

    }

    private static List<AtomicAction> createPublicationOrganizationRelation(final OafProtos.Oaf publication, final OafProtos.Oaf organization, final ActionFactory factory, final String setName, final Agent agent) {

        List<AtomicAction> result = new ArrayList<>();

        final OafProtos.Oaf.Builder roaf = OafProtos.Oaf.newBuilder();
        roaf.setKind(KindProtos.Kind.relation);

        roaf.setDataInfo(FieldTypeProtos.DataInfo.newBuilder()
                .setInvisible(false)
                .setDeletedbyinference(false)
                .setInferred(false)
                .setTrust("0.9")
                .setProvenanceaction(getQualifier("sysimport:actionset", "dnet:provenanceActions"))
                .build());


        final OafProtos.OafRel.Builder rel = OafProtos.OafRel.newBuilder();

        rel.setRelType(RelTypeProtos.RelType.resultOrganization);
        rel.setSubRelType(RelTypeProtos.SubRelType.affiliation);

        //Create a relation Result --> Organization
        rel.setSource(publication.getEntity().getId());
        rel.setTarget(organization.getEntity().getId());
        rel.setRelClass(ResultOrganization.Affiliation.RelName.hasAuthorInstitution.toString());

        final ResultOrganization.Builder rel_instance = ResultOrganization.newBuilder();

        final ResultOrganization.Affiliation.Builder affiliationRel = ResultOrganization.Affiliation.newBuilder();
        affiliationRel.setRelMetadata(RelMetadataProtos.RelMetadata.newBuilder()
                .setSemantics(getQualifier("hasAuthorInstitution", "dnet:result_organization_relations"))
                .build());
        rel_instance.setAffiliation(affiliationRel.build());
        rel.setResultOrganization(rel_instance.build());

        rel.addCollectedfrom(FieldTypeProtos.KeyValue.newBuilder()
                .setValue(datasources.get("MAG").getKey())
                .setKey("10|openaire____::" + AbstractDNetXsltFunctions.md5(StringUtils.substringAfter(datasources.get("MAG").getValue(), "::")))
                .build());



        rel.setChild(false);
        roaf.setRel(rel.build());

        result.add(factory.createAtomicAction(setName, agent, publication.getEntity().getId(), "resultOrganization_affiliation_hasAuthorInstitution", organization.getEntity().getId(), roaf.build().toByteArray() ));


        //Create a relation Organization --> Result
        rel.setTarget(publication.getEntity().getId());
        rel.setSource(organization.getEntity().getId());
        rel.setRelClass(ResultOrganization.Affiliation.RelName.isAuthorInstitutionOf.toString());


        affiliationRel.setRelMetadata(RelMetadataProtos.RelMetadata.newBuilder()
                .setSemantics(getQualifier("isAuthorInstitutionOf", "dnet:result_organization_relations"))
                .build());
        rel_instance.setAffiliation(affiliationRel.build());
        rel.setResultOrganization(rel_instance.build());
        roaf.setRel(rel.build());
        result.add(factory.createAtomicAction(setName, agent, organization.getEntity().getId(), "resultOrganization_affiliation_isAuthorInstitutionOf", publication.getEntity().getId(), roaf.build().toByteArray()));

        return result;

    }

    private static boolean hasJSONArrayField(final JsonObject root, final String key) {
        return root.has(key) && root.get(key).isJsonArray();
    }

    private static void settingRelevantDate(JsonObject rootElement, ResultProtos.Result.Metadata.Builder metadata , final String jsonKey, final String dictionaryKey, final boolean addToDateOfAcceptance) {
        //Adding date
        String date = getStringValue(rootElement,jsonKey);
        if (date == null)
            return;
        if (date.length() == 4) {
            date += "-01-01";
        }
        if (isValidDate(date)) {
            if (addToDateOfAcceptance)
                metadata.setDateofacceptance(FieldTypeProtos.StringField.newBuilder().setValue(date).build());
            metadata.addRelevantdate(FieldTypeProtos.StructuredProperty.newBuilder()
                    .setValue(date)
                    .setQualifier(getQualifier(dictionaryKey,"dnet:dataCite_date"))
                    .build());
        }
    }


    public static FieldTypeProtos.KeyValue extractIdentifier(final String value) {
        FieldTypeProtos.KeyValue.Builder pid = FieldTypeProtos.KeyValue.newBuilder();
        if (StringUtils.contains(value, "orcid.org")){
            return pid.setValue(value)
                    .setKey("ORCID").build();
        }
        if (StringUtils.contains(value, "academic.microsoft.com/#/detail")){
            return pid.setValue(value)
                    .setKey("MAG Identifier").build();
        }
        return pid.setValue(value)
                .setKey("URL").build();
    }


    public static OafProtos.Oaf createOrganizationFromJSON(final JsonObject affiliation) {
        final Map<String, FieldTypeProtos.Qualifier> affiliationIdentifiers = new HashMap<>();
        final List<String> magId = new ArrayList<>();
        getArrayObjects(affiliation, "identifiers").forEach(it -> {
            if (StringUtils.contains(it.get("value").getAsString(), "academic.microsoft.com")) {
                affiliationIdentifiers.put(it.get("value").getAsString(), affiliationPIDType.get("MAG"));
                magId.add(it.get("value").getAsString());
            }
            else
                affiliationIdentifiers.put( it.get("value").getAsString(), affiliationPIDType.get(it.get("schema").getAsString()));
        });
        if (magId.size() > 0) {
            final String microsoftID = magId.get(0);
            OafProtos.Oaf.Builder oaf = OafProtos.Oaf.newBuilder();
            oaf.setKind(KindProtos.Kind.entity);
            OafProtos.OafEntity.Builder entity = OafProtos.OafEntity.newBuilder();
            entity.setType(TypeProtos.Type.organization);
            entity.setId("20|microsoft___::"+AbstractDNetXsltFunctions.md5(microsoftID));
            final String id =datasources.get(affiliation.get("provenance").getAsString()).getValue();
            final String name =datasources.get(affiliation.get("provenance").getAsString()).getKey();
            if (StringUtils.isNotBlank(id) && StringUtils.isNotBlank(name)) {
                final FieldTypeProtos.KeyValue collectedFrom = FieldTypeProtos.KeyValue.newBuilder()
                        .setValue(name)
                        .setKey("10|openaire____::" + AbstractDNetXsltFunctions.md5(StringUtils.substringAfter(id, "::")))
                        .build();
                entity.addCollectedfrom(collectedFrom);
            } else {
                return null;
            }
            entity.addOriginalId(microsoftID);

            affiliationIdentifiers.forEach((key, value) -> entity.addPid(
                    FieldTypeProtos.StructuredProperty.newBuilder()
                            .setQualifier(value)
                            .setValue(key)
                            .build()));

            final OrganizationProtos.Organization.Builder organization = OrganizationProtos.Organization.newBuilder();
            organization.setMetadata(OrganizationProtos.Organization.Metadata.newBuilder()
                    .setWebsiteurl(FieldTypeProtos.StringField.newBuilder().setValue(affiliation.get("official-page").getAsString()).build())
                    .setLegalname(FieldTypeProtos.StringField.newBuilder().setValue(affiliation.get("value").getAsString()).build())
                    .build());

            entity.setOrganization(organization);
            oaf.setEntity(entity);
            oaf.setDataInfo(FieldTypeProtos.DataInfo.newBuilder()
                    .setInvisible(false)
                    .setDeletedbyinference(false)
                    .setInferred(false)
                    .setTrust("0.9")
                    .setProvenanceaction(getQualifier("sysimport:actionset", "dnet:provenanceActions"))
                    .build());
            return oaf.build();
        }
        return  null;
    }

    public static Pair<List<FieldTypeProtos.Author>, Collection<OafProtos.Oaf>>  createAuthorsOrganization(final JsonObject root) {

        final Map<String, OafProtos.Oaf> affiliations = new HashMap<>();

        List<JsonObject> authors = getArrayObjects(root, "authors");

        final AtomicInteger counter = new AtomicInteger();

        List<FieldTypeProtos.Author> collect = authors.stream().map(author -> {
            final String given = getStringValue(author, "given");
            final String family = getStringValue(author, "family");
            String fullname = getStringValue(author, "fullname");

            if (StringUtils.isBlank(fullname) && StringUtils.isNotBlank(given) && StringUtils.isNotBlank(family)) {
                fullname = String.format("%s %s", given, family);
            }
            final FieldTypeProtos.Author.Builder abuilder = FieldTypeProtos.Author.newBuilder();

            if (StringUtils.isNotBlank(given))
                abuilder.setName(given);
            if (StringUtils.isNotBlank(family))
            abuilder.setSurname(family);
            if (StringUtils.isNotBlank(fullname))
                abuilder.setFullname(fullname);

            final List<JsonObject> identifiers = getArrayObjects(author, "identifiers");
            final List<JsonObject> authorAffiliation = getArrayObjects(author, "affiliations");

            authorAffiliation.forEach(it ->
            {
                OafProtos.Oaf org = createOrganizationFromJSON(it);
                if (org != null) {
                    affiliations.put(org.getEntity().getId(), org);
                    abuilder.addAffiliation(org.getEntity().getOrganization().getMetadata().getLegalname());
                }
            });
            identifiers.stream().map(id -> {
                final String value = id.get("value").getAsString();
                return extractIdentifier(value);
            }).forEach(abuilder::addPid);
            abuilder.setRank(counter.getAndIncrement());

            return abuilder.build();

        }).collect(Collectors.toList());

        return new Pair<> ( collect,affiliations.values() );
    }






}
