package eu.dnetlib.data.mapreduce.hbase.bulktag;

import com.google.common.base.Functions;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.google.protobuf.InvalidProtocolBufferException;
import com.jayway.jsonpath.DocumentContext;
import com.jayway.jsonpath.JsonPath;
import eu.dnetlib.data.bulktag.CommunityConfiguration;
import eu.dnetlib.data.bulktag.Pair;
import eu.dnetlib.data.proto.FieldTypeProtos;
import eu.dnetlib.data.proto.OafProtos;
import eu.dnetlib.data.proto.ResultProtos;
import eu.dnetlib.data.proto.ResultProtos.Result.Context;
import org.apache.commons.lang3.StringUtils;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.mapreduce.Mapper;

import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * Created by miriam on 02/08/2018.
 */
public class ResultTagger {
    private final static String DATA_INFO_TYPE = "bulktagging";
    private final static String SCHEMA_NAME = "dnet:provenanceActions";
    private final static String CLASS_ID_SUBJECT = "bulktagging:community:subject";
    private final static String CLASS_ID_DATASOURCE = "bulktagging:community:datasource";
    private final static String CLASS_ID_CZENODO = "bulktagging:community:zenodocommunity";
    private final static String SCHEMA_ID = "dnet:provenanceActions";
    private final static String COUNTER_GROUP = "Bulk Tagging";

    private final static String ZENODO_COMMUNITY_INDICATOR = "zenodo.org/communities/";

    private String trust = "0.8";


    private boolean clearContext(ResultProtos.Result.Metadata.Builder mBuilder){
        int tmp = mBuilder.getContextBuilderList().size();
        List<Context.Builder> clist = mBuilder.getContextBuilderList().stream()
                .filter(c -> (!c.getId().contains(ZENODO_COMMUNITY_INDICATOR))).collect(Collectors.toList());
        mBuilder.clearContext();
        clist.forEach(c->mBuilder.addContext(c));
        return (tmp != clist.size());
    }
    public Map<String,List<String>> getParamMap(byte[] body, Map<String,String> params){
        Map<String,List<String>> param = new HashMap<>();
        DocumentContext jsonContext = JsonPath.parse(Bytes.toString(body));

        for(String key:params.keySet()) {
            try {
                param.put(key, jsonContext.read(params.get(key)));
            } catch (Exception e) {
                param.put(key, new ArrayList<>());
            }
        }
        return param;

    }
    public OafProtos.Oaf enrichContextCriteria(byte[] body, final CommunityConfiguration conf, final Mapper.Context context, final Map<String,String> criteria) throws InvalidProtocolBufferException {
        return enrichContextCriteria(OafProtos.Oaf.parseFrom(body),conf,context,getParamMap(body, criteria));

    }

    public OafProtos.Oaf enrichContextCriteria(final OafProtos.Oaf oaf, final CommunityConfiguration conf, final Mapper.Context context,Map<String,List<String>> param){
        final OafProtos.Oaf.Builder builder = OafProtos.Oaf.newBuilder(oaf);
        final ResultProtos.Result.Metadata.Builder mBuilder = builder.getEntityBuilder().getResultBuilder().getMetadataBuilder();

        /*
         * Verify if the entity is deletedbyinference. In case verify if to clean the context list from all the zenodo communities
         * */
        if(oaf.getDataInfo().getDeletedbyinference()){
            context.getCounter(COUNTER_GROUP, "deleted by inference").increment(1);
            if(!clearContext(mBuilder))
                return null;
            context.getCounter(COUNTER_GROUP,"removed zenodo community context in del by inference record").increment(1);
            return builder.build();
        }

        //communities contains all the communities to be added as context for the result
        final Set<String> communities = new HashSet<>();

        /*
         *Tagging for Subject
         * */
        final Set<String> subjects = new HashSet<>();
        oaf.getEntity().getResult().getMetadata().getSubjectList().stream()
                .map(subject -> subject.getValue())
                .filter(StringUtils::isNotBlank)
                .map(String::toLowerCase)
                .map(String::trim)
                .collect(Collectors.toCollection(HashSet::new))
                .forEach(s -> subjects.addAll(conf.getCommunityForSubjectValue(s)));

        //updating counters for subject matching
        subjects.forEach(c->context.getCounter(COUNTER_GROUP, "Matching subject for community " + c).increment(1));
        communities.addAll(subjects);
        context.getCounter(COUNTER_GROUP,"match found for subjects ").increment(subjects.size());

        /*Tagging for datasource*/
        final Set<String> datasources = new HashSet<>();
        final Set<String> tmp = new HashSet<>();
        for(ResultProtos.Result.Instance i : oaf.getEntity().getResult().getInstanceList()){
            tmp.add(StringUtils.substringAfter(i.getCollectedfrom().getKey(),"|"));
            tmp.add(StringUtils.substringAfter(i.getHostedby().getKey(),"|"));
        }

        oaf.getEntity().getResult().getInstanceList()
                .stream()
                .map(i -> new Pair<>(i.getCollectedfrom().getKey(), i.getHostedby().getKey()))
                .flatMap(p -> Stream.of(p.getFst(), p.getSnd()))
                .map(s -> StringUtils.substringAfter(s, "|"))
                .collect(Collectors.toCollection(HashSet::new))
                .forEach(dsId -> datasources.addAll(conf.getCommunityForDatasource(dsId,param)));
        //updating counters for datasource matching
        datasources.forEach(c->context.getCounter(COUNTER_GROUP,"Matching datasource for community " + c).increment(1));
        communities.addAll(datasources);
        context.getCounter(COUNTER_GROUP,"Match found for content providers " ).increment(datasources.size());

        /*Tagging for Zenodo Communities*/
        final Set<String> czenodo = new HashSet<>();
        //final ResultProtos.Result.Metadata.Builder mBuilder = builder.getEntityBuilder().getResultBuilder().getMetadataBuilder();
        mBuilder.getContextBuilderList().stream().filter(cBuilder -> cBuilder.getId().contains(ZENODO_COMMUNITY_INDICATOR))
                .collect(Collectors.toList())
                .forEach(c->czenodo.addAll(conf.getCommunityForZenodoCommunityValue(c.getId().substring(c.getId().lastIndexOf("/")+1).trim())));
        czenodo.forEach(c->context.getCounter(COUNTER_GROUP,"Matching Zenodo community for community " + c).increment(1));
        //updating counters for Zenodo communities matching
        context.getCounter(COUNTER_GROUP,"Match found for Zenodo communities " ).increment(czenodo.size());
        communities.addAll(czenodo);

        boolean removed = clearContext(mBuilder);

        /*Verify if there is something to bulktag*/
        if(communities.isEmpty()){
            context.getCounter(COUNTER_GROUP, "list of communities empty").increment(1);
//            return null;
            if (!removed)
                return null;
            else {
                context.getCounter(COUNTER_GROUP,"removed Zenodo community from result not to be enriched").increment(1);
                return builder.build();
            }
        }else{
            context.getCounter(COUNTER_GROUP, "list of communities has values!").increment(1);
        }

        final Map<String, ResultProtos.Result.Context.Builder> cBuilders = Maps.newHashMap();
        mBuilder.getContextBuilderList().forEach(cBuilder -> {
            cBuilders.put(cBuilder.getId(), cBuilder);
        });

        /*Applies actual tagging*/
        for(String contextId:communities){
            ResultProtos.Result.Context.Builder cBuilder = cBuilders.get(contextId);
            if (cBuilder != null) {
                if (!cBuilder.getDataInfoBuilderList().stream()
                        .map(di -> di.getInferenceprovenance())
                        .anyMatch(s -> DATA_INFO_TYPE.equals(s))) {
                    if (subjects.contains(contextId))
                        cBuilder.addDataInfo(buildDataInfo(CLASS_ID_SUBJECT));
                    if(datasources.contains(contextId))
                        cBuilder.addDataInfo(buildDataInfo(CLASS_ID_DATASOURCE));
                    if(czenodo.contains(contextId))
                        cBuilder.addDataInfo(buildDataInfo(CLASS_ID_CZENODO));
                    context.getCounter(COUNTER_GROUP, "add provenance").increment(1);
                } else {
                    context.getCounter(COUNTER_GROUP, "provenance already bulk tagged").increment(1);
                }
            } else {
                context.getCounter(COUNTER_GROUP, "add context").increment(1);
                cBuilder = Context.newBuilder().setId(contextId);

                if (subjects.contains(contextId))
                    cBuilder.addDataInfo(buildDataInfo(CLASS_ID_SUBJECT));
                if(datasources.contains(contextId))
                    cBuilder.addDataInfo(buildDataInfo(CLASS_ID_DATASOURCE));
                if(czenodo.contains(contextId))
                    cBuilder.addDataInfo(buildDataInfo(CLASS_ID_CZENODO));
                mBuilder.addContext(cBuilder.build());
            }
        }

        return builder.build();
    }

    public OafProtos.Oaf enrichContext(final OafProtos.Oaf oaf, final CommunityConfiguration conf, final Mapper.Context context) {

        final OafProtos.Oaf.Builder builder = OafProtos.Oaf.newBuilder(oaf);
        final ResultProtos.Result.Metadata.Builder mBuilder = builder.getEntityBuilder().getResultBuilder().getMetadataBuilder();

        /*
        * Verify if the entity is deletedbyinference. In case verify if to clean the context list from all the zenodo communities
        * */
        if(oaf.getDataInfo().getDeletedbyinference()){
            context.getCounter(COUNTER_GROUP, "deleted by inference").increment(1);
            if(!clearContext(mBuilder))
                return null;
            context.getCounter(COUNTER_GROUP,"removed zenodo community context in del by inference record").increment(1);
            return builder.build();
        }

        //communities contains all the communities to be added as context for the result
        final Set<String> communities = new HashSet<>();

        /*
        *Tagging for Subject
        * */
        final Set<String> subjects = new HashSet<>();
        oaf.getEntity().getResult().getMetadata().getSubjectList().stream()
                .map(subject -> subject.getValue())
                .filter(StringUtils::isNotBlank)
                .map(String::toLowerCase)
                .map(String::trim)
                .collect(Collectors.toCollection(HashSet::new))
                .forEach(s -> subjects.addAll(conf.getCommunityForSubjectValue(s)));

        //updating counters for subject matching
        subjects.forEach(c->context.getCounter(COUNTER_GROUP, "Matching subject for community " + c).increment(1));
        communities.addAll(subjects);
        context.getCounter(COUNTER_GROUP,"match found for subjects ").increment(subjects.size());

        /*Tagging for datasource*/
        final Set<String> datasources = new HashSet<>();
        final Set<String> tmp = new HashSet<>();
        for(ResultProtos.Result.Instance i : oaf.getEntity().getResult().getInstanceList()){
            tmp.add(StringUtils.substringAfter(i.getCollectedfrom().getKey(),"|"));
            tmp.add(StringUtils.substringAfter(i.getHostedby().getKey(),"|"));
        }

        oaf.getEntity().getResult().getInstanceList()
                .stream()
                .map(i -> new Pair<>(i.getCollectedfrom().getKey(), i.getHostedby().getKey()))
                .flatMap(p -> Stream.of(p.getFst(), p.getSnd()))
                .map(s -> StringUtils.substringAfter(s, "|"))
                .collect(Collectors.toCollection(HashSet::new))
                .forEach(dsId -> datasources.addAll(conf.getCommunityForDatasourceValue(dsId)));
        //updating counters for datasource matching
        datasources.forEach(c->context.getCounter(COUNTER_GROUP,"Matching datasource for community " + c).increment(1));
        communities.addAll(datasources);
        context.getCounter(COUNTER_GROUP,"Match found for content providers " ).increment(datasources.size());

        /*Tagging for Zenodo Communities*/
        final Set<String> czenodo = new HashSet<>();
        //final ResultProtos.Result.Metadata.Builder mBuilder = builder.getEntityBuilder().getResultBuilder().getMetadataBuilder();
        mBuilder.getContextBuilderList().stream().filter(cBuilder -> cBuilder.getId().contains(ZENODO_COMMUNITY_INDICATOR))
                .collect(Collectors.toList())
                .forEach(c->czenodo.addAll(conf.getCommunityForZenodoCommunityValue(c.getId().substring(c.getId().lastIndexOf("/")+1).trim())));
        czenodo.forEach(c->context.getCounter(COUNTER_GROUP,"Matching Zenodo community for community " + c).increment(1));
        //updating counters for Zenodo communities matching
        context.getCounter(COUNTER_GROUP,"Match found for Zenodo communities " ).increment(czenodo.size());
        communities.addAll(czenodo);

        boolean removed = clearContext(mBuilder);

        /*Verify if there is something to bulktag*/
        if(communities.isEmpty()){
            context.getCounter(COUNTER_GROUP, "list of communities empty").increment(1);
//            return null;
            if (!removed)
                return null;
            else {
                context.getCounter(COUNTER_GROUP,"removed Zenodo community from result not to be enriched").increment(1);
                return builder.build();
            }
        }else{
            context.getCounter(COUNTER_GROUP, "list of communities has values!").increment(1);
        }

        final Map<String, ResultProtos.Result.Context.Builder> cBuilders = Maps.newHashMap();
        mBuilder.getContextBuilderList().forEach(cBuilder -> {
            cBuilders.put(cBuilder.getId(), cBuilder);
        });

        /*Applies actual tagging*/
        for(String contextId:communities){
            ResultProtos.Result.Context.Builder cBuilder = cBuilders.get(contextId);
            if (cBuilder != null) {
                if (!cBuilder.getDataInfoBuilderList().stream()
                        .map(di -> di.getInferenceprovenance())
                        .anyMatch(s -> DATA_INFO_TYPE.equals(s))) {
                    if (subjects.contains(contextId))
                        cBuilder.addDataInfo(buildDataInfo(CLASS_ID_SUBJECT));
                    if(datasources.contains(contextId))
                        cBuilder.addDataInfo(buildDataInfo(CLASS_ID_DATASOURCE));
                    if(czenodo.contains(contextId))
                        cBuilder.addDataInfo(buildDataInfo(CLASS_ID_CZENODO));
                    context.getCounter(COUNTER_GROUP, "add provenance").increment(1);
                } else {
                    context.getCounter(COUNTER_GROUP, "provenance already bulk tagged").increment(1);
                }
            } else {
                context.getCounter(COUNTER_GROUP, "add context").increment(1);
                cBuilder = Context.newBuilder().setId(contextId);

                if (subjects.contains(contextId))
                    cBuilder.addDataInfo(buildDataInfo(CLASS_ID_SUBJECT));
                if(datasources.contains(contextId))
                    cBuilder.addDataInfo(buildDataInfo(CLASS_ID_DATASOURCE));
                if(czenodo.contains(contextId))
                    cBuilder.addDataInfo(buildDataInfo(CLASS_ID_CZENODO));
                mBuilder.addContext(cBuilder.build());
            }
        }

        return builder.build();
    }



    private FieldTypeProtos.DataInfo buildDataInfo(String class_id) {
        FieldTypeProtos.DataInfo.Builder builder = FieldTypeProtos.DataInfo.newBuilder()
                .setInferred(true)
                .setProvenanceaction(
                        FieldTypeProtos.Qualifier.newBuilder()
                                .setClassid(class_id)
                                .setClassname("Bulk Tagging for Communities")
                                .setSchemeid(SCHEMA_ID)
                                .setSchemename(SCHEMA_NAME))
                .setInferenceprovenance(DATA_INFO_TYPE)
                .setTrust(trust);
        return builder
                .build();
    }


    public void setTrust(String s) {
        trust = s;
    }
}