package eu.dnetlib.data.mapreduce.hbase.dataexport.linkinganalysis.publicationsoftware;

import com.google.common.collect.Lists;
import com.google.gson.Gson;
import com.google.protobuf.InvalidProtocolBufferException;
import eu.dnetlib.data.mapreduce.hbase.dataexport.linkinganalysis.publicationsoftware.Constants.Type;
import eu.dnetlib.data.proto.OafProtos;
import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.io.ImmutableBytesWritable;
import org.apache.hadoop.hbase.mapreduce.TableMapper;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.io.Text;

import java.io.IOException;
import java.util.*;
import java.util.stream.Collectors;

public class PubSfwLinkMapper extends TableMapper<ImmutableBytesWritable, Text> {

    public static final String COUNTER_GROUP = "Find Link Pub Software";
    private ImmutableBytesWritable keyOut;
    private Text valueOut;

    @Override
    protected void setup(final Context context) throws IOException, InterruptedException {
        super.setup(context);

        keyOut = new ImmutableBytesWritable();
        valueOut = new Text();
    }

    @Override
    protected void map(final ImmutableBytesWritable key, final Result value, final Context context) throws IOException, InterruptedException {

        final Map<byte[], byte[]> resultMap = value.getFamilyMap(Bytes.toBytes("result"));

        final byte[] body = resultMap.get(Bytes.toBytes("body"));

        if (body != null) {
            context.getCounter(COUNTER_GROUP, "not null body ").increment(1);

            final OafProtos.Oaf oaf = OafProtos.Oaf.parseFrom(body);
            if (oaf == null) {
                return;
            }

            if (oaf.getDataInfo().getDeletedbyinference()) {
                context.getCounter(COUNTER_GROUP, "deleted by inference").increment(1);
                return;
            }

            final String resulttype = oaf.getEntity().getResult().getMetadata().getResulttype().getClassid();

            final Map<byte[], byte[]> relationMap = value.getFamilyMap(Bytes.toBytes("resultResult_relationship_isRelatedTo"));

            if (relationMap == null || relationMap.isEmpty()) {
                context.getCounter(COUNTER_GROUP, String.format("missing relation (%s)", resulttype)).increment(1);
                return;
            }

            final IdentifierList pids = oaf.getEntity().getPidList().stream()
                    .map(pid -> new Identifier(pid.getQualifier().getClassid(), pid.getValue()))
                    .collect(Collectors.toCollection(IdentifierList::new));

            context.getCounter(COUNTER_GROUP, resulttype).increment(1);

            switch (resulttype) {
                case "publication":
                    emit(oaf.getEntity().getId(), Value.newInstance(pids.toJson(), Type.publication).toJson(), context);
                    break;

                case "software":
                    List<String> targets = relationMap.values().stream()
                            .map(this::asOaf)
                            .filter(Objects::nonNull)
                            .filter(o -> isValid(o))
                            .filter(o -> !o.getDataInfo().getDeletedbyinference())
                            .map(o -> o.getRel().getTarget())
                            .collect(Collectors.toList());

                    Set<String> urls = oaf.getEntity().getResult().getInstanceList().stream()
                            .map(i -> i.getUrlList())
                            .flatMap(Collection::stream)
                            .collect(Collectors.toCollection(HashSet::new));

                    SoftwareResource sr = new SoftwareResource()
                            .setOpenAireId(new String(key.copyBytes()))
                            .setUrls(Lists.newArrayList(urls))
                            .setSoftwarePIDs(pids);

                    for (String target : targets) {
                        emit(target, Value.newInstance(sr.toJson(), Type.software).toJson(), context);
                    }

                    break;
            }
        }
    }

    private void emit(String target, String value, Context context) throws IOException, InterruptedException {
        keyOut.set(target.getBytes());
        valueOut.set(value);

        context.write(keyOut, valueOut);
    }

    private OafProtos.Oaf asOaf(byte[] r) {
        try {
            return OafProtos.Oaf.parseFrom(r);
        } catch (InvalidProtocolBufferException e) {
            return null;
        }
    }

    private boolean isValid(final OafProtos.Oaf oaf) {
        return (oaf != null) && oaf.isInitialized();
    }

}