package eu.dnetlib.data.mapreduce.hbase.bulktag;

import com.google.common.reflect.TypeToken;
import com.google.gson.Gson;
import com.jayway.jsonpath.DocumentContext;
import com.jayway.jsonpath.JsonPath;
import eu.dnetlib.data.bulktag.CommunityConfiguration;
import eu.dnetlib.data.bulktag.CommunityConfigurationFactory;
import eu.dnetlib.data.proto.FieldTypeProtos;
import eu.dnetlib.data.proto.OafProtos.Oaf;
import org.apache.commons.lang.StringUtils;
import org.apache.hadoop.hbase.client.Put;
import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.io.ImmutableBytesWritable;
import org.apache.hadoop.hbase.mapreduce.TableMapper;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.io.Writable;

import java.io.IOException;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class BulkTaggingMapper extends TableMapper<ImmutableBytesWritable, Writable> {

	private CommunityConfiguration cc;

	private ResultTagger tagger;
	private boolean enabled;

	private Map<String,String> params;

	private Param param;

	@Override
	protected void setup(final Context context) throws IOException, InterruptedException {
		super.setup(context);

		final String conf = context.getConfiguration().get("tagging.conf");
		enabled = context.getConfiguration().getBoolean("tagging.enabled",false);
		if (StringUtils.isBlank(conf)) {
			throw new IllegalArgumentException("missing bulk tagging configuration");
		}
		System.out.println("conf = " + conf);
		cc = CommunityConfigurationFactory.fromJson(conf);

		tagger = new ResultTagger();
		tagger.setTrust(context.getConfiguration().get("bulktagging.trust", "0.85"));

		param = new Gson().fromJson(context.getConfiguration().get("mapping.proto","{}"),Param.class);
		params = param.getP();
	}

	private Map<String,String> loadMapping(String mapping){
		Type type = new TypeToken<Map<String,String>>(){}.getType();
		return new Gson().fromJson(mapping, type);

	}

	@Override
	protected void map(final ImmutableBytesWritable key, final Result value, final Context context) throws IOException, InterruptedException {

		final Map<byte[], byte[]> resultMap = value.getFamilyMap(Bytes.toBytes("result"));

		final byte[] body = resultMap.get(Bytes.toBytes("body"));

		if (body != null) {
			context.getCounter("Bulk Tagging", "not null body ").increment(1);

			//final Oaf oaf = tagger.enrichContext(Oaf.parseFrom(body), cc, context);
			final Oaf oaf = tagger.enrichContextCriteria(body,cc,context,params);// (Oaf.parseFrom(body), cc, context,params);
			if (oaf == null) {
				return;
			}

			long tagged = oaf.getEntity().getResult().getMetadata().getContextList().stream()
					.flatMap(c -> c.getDataInfoList().stream())
					.map(FieldTypeProtos.DataInfo::getInferenceprovenance)
					.filter(infProv -> "bulktagging".equals(infProv))
					.count();
			context.getCounter("Bulk Tagging", " bulktagged ").increment(tagged);

			final Put put = new Put(key.copyBytes()).add(Bytes.toBytes("result"), Bytes.toBytes("body"), oaf.toByteArray());


			if (enabled)
				context.write(key, put);
			context.getCounter("Bulk Tagging", " write op ").increment(1);

		}
		else{
			context.getCounter("Bulk Tagging", " null body ").increment(1);
		}

	}

	private Map<String,List<String>> getParamMap(byte[] body){
		Map<String,List<String>> param = new HashMap<>();
		DocumentContext jsonContext = JsonPath.parse(body);

		for(String key:params.keySet()) {
			try {
				param.put(key, jsonContext.read(params.get(key)));
			} catch (Exception e) {
				param.put(key, new ArrayList<>());
			}
		}
		return param;

	}

}