package eu.dnetlib.data.mapreduce.hbase.dedup.fixrelation;

import com.google.gson.Gson;
import eu.dnetlib.data.mapreduce.JobParams;
import eu.dnetlib.data.mapreduce.util.OafDecoder;
import eu.dnetlib.data.mapreduce.util.OafRelDecoder;
import eu.dnetlib.data.mapreduce.util.RelDescriptor;
import eu.dnetlib.data.proto.FieldTypeProtos;
import eu.dnetlib.data.proto.OafProtos;
import eu.dnetlib.utils.ontologies.Ontologies;
import org.apache.hadoop.hbase.io.ImmutableBytesWritable;
import org.apache.hadoop.hbase.mapreduce.TableReducer;
import org.apache.hadoop.hbase.util.Bytes;

import java.io.IOException;
import java.util.Iterator;

import static eu.dnetlib.data.mapreduce.util.DedupUtils.isRoot;
import static eu.dnetlib.data.mapreduce.util.OafHbaseUtils.asPut;

/**
 * Created by sandro on 2/24/17.
 */
public class DedupFixRelationReducer extends TableReducer<Key, ImmutableBytesWritable, ImmutableBytesWritable> {

    public static final String COUNTER_GROUP = "Fix relations";

    private Ontologies ontologies;

    private boolean simulation;

    @Override
    protected void setup(final Context context) throws IOException, InterruptedException {
        super.setup(context);

        simulation = context.getConfiguration().getBoolean("fixrel.simulation", false);
        ontologies = new Gson().fromJson(context.getConfiguration().get(JobParams.ONTOLOGIES), Ontologies.class);

        System.out.println("ontologies: " + ontologies.toJson(true));
    }

    @Override
    protected void reduce(Key key, Iterable<ImmutableBytesWritable> values, Context context) {
        if (isRoot(key.toString())) {
            System.err.println("aborting DedupFixRelationReducer, found root key: " + key);
            context.getCounter(COUNTER_GROUP, "aborted").increment(1);
            return;
        }

        final Iterator<ImmutableBytesWritable> it = values.iterator();
        final OafProtos.Oaf first = OafDecoder.decode(it.next().copyBytes()).getOaf();

        if (!first.getRel().getRelClass().equals("merges")) {
            context.getCounter(COUNTER_GROUP, "Relation skipped").increment(1);
            return;
        }
        context.getCounter(COUNTER_GROUP, "Item to fix").increment(1);

        final String dedupRoot = first.getRel().getSource();
        it.forEachRemaining(b -> {
            try {
                handleRels(context, OafDecoder.decode(b.copyBytes()).getOaf(), dedupRoot);
            } catch (Exception e) {
                throw new RuntimeException(e);
            }
        });
    }

    private void handleRels(final Context context, final OafProtos.Oaf oaf, final String dedupRoot) throws IOException, InterruptedException {

        final String relType = oaf.getRel().getRelClass();
        if (relType.contains("merges")) {
            return;
        }

        // Set relation deleted by inference from Root to entity that has been merged to another one
        final FieldTypeProtos.DataInfo.Builder dataInfoBuilder = FieldTypeProtos.DataInfo.newBuilder(oaf.getDataInfo());
        dataInfoBuilder.setDeletedbyinference(true);
        OafProtos.Oaf.Builder builder = OafProtos.Oaf.newBuilder(oaf);
        builder.setDataInfo(dataInfoBuilder.build());
        final String sourceKey = oaf.getRel().getSource();
        write(context, sourceKey, builder);

        String relGroup = oaf.getRel().getRelType().toString() + oaf.getRel().getSubRelType().toString();
        context.getCounter(COUNTER_GROUP, String.format("%s - Relation set deleted", relGroup)).increment(1);

        // Create Relation from Root Entity to its deduplicated Entity
        builder = OafProtos.Oaf.newBuilder(oaf);
        OafProtos.OafRel.Builder relBuilder = OafProtos.OafRel.newBuilder(oaf.getRel());
        relBuilder.setTarget(dedupRoot);
        builder.setRel(relBuilder.build());
        write(context, sourceKey, builder);

        // Create Relation from deduplicated Entity to Root Entity
        relBuilder = OafProtos.OafRel.newBuilder(oaf.getRel());
        relBuilder.setTarget(relBuilder.getSource());
        relBuilder.setSource(dedupRoot);

        RelDescriptor rd = OafRelDecoder.decode(oaf.getRel()).getRelDescriptor();

        final String inverseRelation = ontologies.get(rd.getRelType().toString()).inverseOf(rd.getRelClass());
        relBuilder.setRelType(rd.getRelType()).setSubRelType(rd.getSubRelType()).setRelClass(inverseRelation);

        builder.setRel(relBuilder.build());
        write(context, dedupRoot, builder);

        relGroup = oaf.getRel().getRelType().toString() + oaf.getRel().getSubRelType().toString();
        context.getCounter(COUNTER_GROUP, String.format("%s - Relation fixed", relGroup)).increment(2);
    }

    private void write(Context context, String dedupRoot, OafProtos.Oaf.Builder builder) throws IOException, InterruptedException {
        if (!simulation) {
            context.write(new ImmutableBytesWritable(Bytes.toBytes(dedupRoot)), asPut(builder.build()));
        }
    }

}
