package eu.dnetlib.data.mapreduce.util;

import java.io.StringReader;
import java.io.StringWriter;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import javax.xml.transform.*;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import com.google.common.base.Joiner;
import com.google.common.base.Predicate;
import com.google.common.base.Splitter;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.google.protobuf.Descriptors.EnumValueDescriptor;
import com.google.protobuf.Descriptors.FieldDescriptor;
import com.google.protobuf.GeneratedMessage;
import com.mycila.xmltool.XMLDoc;
import com.mycila.xmltool.XMLTag;
import eu.dnetlib.data.mapreduce.hbase.index.config.*;
import eu.dnetlib.data.proto.FieldTypeProtos.*;
import eu.dnetlib.data.proto.OafProtos.OafEntity;
import eu.dnetlib.data.proto.OafProtos.OafRel;
import eu.dnetlib.data.proto.ProjectProtos.Project;
import eu.dnetlib.data.proto.RelMetadataProtos.RelMetadata;
import eu.dnetlib.data.proto.ResultProtos.Result;
import eu.dnetlib.data.proto.ResultProtos.Result.Context;
import eu.dnetlib.data.proto.ResultProtos.Result.ExternalReference;
import eu.dnetlib.data.proto.ResultProtos.Result.Instance;
import eu.dnetlib.data.proto.TypeProtos;
import eu.dnetlib.data.proto.TypeProtos.Type;
import org.apache.commons.lang3.StringUtils;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.Element;
import org.dom4j.Node;
import org.dom4j.io.SAXReader;

import static eu.dnetlib.miscutils.collections.MappedCollection.listMap;

public class XmlRecordFactory {

	// private static final Log log = LogFactory.getLog(XmlRecordFactory.class); // NOPMD by marko on 11/24/08 5:02 PM

	private final Map<String, Integer> relCounters = Maps.newHashMap();
	protected Set<String> specialDatasourceTypes;
	protected TemplateFactory templateFactory = new TemplateFactory();
	protected OafDecoder mainEntity = null;
	protected String key = null;
	protected List<OafDecoder> relations = Lists.newLinkedList();
	protected List<OafDecoder> children = Lists.newLinkedList();
	protected EntityConfigTable entityConfigTable;
	protected ContextMapper contextMapper;
	protected RelClasses relClasses;
	protected String schemaLocation;
	protected boolean entityDefaults;
	protected boolean relDefaults;
	protected boolean childDefaults;
	protected Set<String> contextes = Sets.newHashSet();
	protected List<String> extraInfo = Lists.newArrayList();
	protected Map<String, Integer> counters = Maps.newHashMap();
	protected Transformer transformer;

	protected static Predicate<String> instanceFilter = new Predicate<String>() {
		final Set<String> instanceFieldFilter = Sets.newHashSet("instancetype", "hostedby", "license", "accessright", "collectedfrom", "dateofacceptance", "distributionlocation");
		@Override
		public boolean apply(final String s) {
			return instanceFieldFilter.contains(s);
		}
	};

	public XmlRecordFactory(final EntityConfigTable entityConfigTable, final ContextMapper contextMapper, final RelClasses relClasses,
			final String schemaLocation, final boolean entityDefaults, final boolean relDefaults, final boolean childDefeaults, final Set<String> otherDatasourceTypesUForUI)
			throws TransformerConfigurationException, TransformerFactoryConfigurationError {
		this.entityConfigTable = entityConfigTable;
		this.contextMapper = contextMapper;
		this.relClasses = relClasses;
		this.schemaLocation = schemaLocation;
		this.entityDefaults = entityDefaults;
		this.relDefaults = relDefaults;
		this.childDefaults = childDefeaults;
		this.specialDatasourceTypes = otherDatasourceTypesUForUI;

		transformer = TransformerFactory.newInstance().newTransformer();
		transformer.setOutputProperty(OutputKeys.OMIT_XML_DECLARATION, "yes");
	}

	public static String removePrefix(final String s) {
		if (s.contains("|")) return StringUtils.substringAfter(s, "|");
		return s;
	}

	public static String escapeXml(final String value) {
		return value.replaceAll("&", "&amp;").replaceAll("<", "&lt;").replaceAll(">", "&gt;").replaceAll("\"", "&quot;").replaceAll("'", "&apos;");
	}

	public Map<String, Integer> getRelCounters() {
		return relCounters;
	}

	public RelClasses getRelClasses() {
		return relClasses;
	}

	public String getId() {
		return key;
	}

	public boolean isValid() {
		return mainEntity != null;
	}

	public void setMainEntity(final OafDecoder mainEntity) {
		this.mainEntity = mainEntity;
		this.key = mainEntity.decodeEntity().getId();
	}

	public void addRelation(final Type type, final OafDecoder rel) {
		addRelOrChild(type, relations, rel);
	}

	public void addChild(final Type type, final OafDecoder child) {
		addRelOrChild(type, children, child);
	}

	private void addRelOrChild(final Type type, final List<OafDecoder> list, final OafDecoder decoder) {

		final OafRel oafRel = decoder.getOafRel();
		final String rd = oafRel.getRelType().toString() + "_" + oafRel.getSubRelType() + "_" + relClasses.getInverse(oafRel.getRelClass());
		final LinkDescriptor ld = entityConfigTable.getDescriptor(type, new RelDescriptor(rd));

		if (getRelCounters().get(rd) == null) {
			getRelCounters().put(rd, 0);
		}

		if (ld == null) {
			list.add(decoder);
			return;
		}

		if (ld.getMax() < 0) {
			list.add(decoder);
			return;
		}

		if (getRelCounters().get(rd) < ld.getMax()) {
			getRelCounters().put(rd, getRelCounters().get(rd) + 1);
			list.add(decoder);
		}
	}

	public String build() {
		try {
			final OafEntityDecoder entity = mainEntity.decodeEntity();
			// log.info("building");
			// log.info("main: " + mainEntity);
			// log.info("rel:  " + relations);
			// log.info("chi:  " + children);
			// log.info("=============");

			final Predicate<String> filter = entityConfigTable.getFilter(entity.getType());
			final List<String> metadata = decodeType(entity, filter, entityDefaults, false);

			// rels has to be processed before the contexts because they enrich the contextMap with the funding info.
			final List<String> rels = listRelations();
			metadata.addAll(buildContexts(entity.getType()));
			metadata.add(parseDataInfo(mainEntity));

			final String body = templateFactory.buildBody(entity.getType(), metadata, rels, listChildren(), extraInfo);

			return templateFactory
					.buildRecord(key, entity.getDateOfCollection(), entity.getDateOfTransformation(), schemaLocation, body, countersAsXml());
		} catch (final Throwable e) {
			throw new RuntimeException(String.format("error building record '%s'", this.key), e);
		}
	}

	private String parseDataInfo(final OafDecoder decoder) {
		final DataInfo dataInfo = decoder.getOaf().getDataInfo();

		final StringBuilder sb = new StringBuilder();
		sb.append("<datainfo>");
		sb.append(asXmlElement("inferred", dataInfo.getInferred() + "", null, null));
		sb.append(asXmlElement("deletedbyinference", dataInfo.getDeletedbyinference() + "", null, null));
		sb.append(asXmlElement("trust", dataInfo.getTrust() + "", null, null));
		sb.append(asXmlElement("inferenceprovenance", dataInfo.getInferenceprovenance() + "", null, null));
		sb.append(asXmlElement("provenanceaction", null, dataInfo.getProvenanceaction(), null));
		sb.append("</datainfo>");

		return sb.toString();
	}

	private List<String> decodeType(final OafEntityDecoder decoder, final Predicate<String> filter, final boolean defaults, final boolean expandingRel) {

		final List<String> metadata = Lists.newArrayList();
		metadata.addAll(listFields(decoder.getMetadata(), filter, defaults, expandingRel));
		metadata.addAll(listFields(decoder.getOafEntity(), filter, defaults, expandingRel));

		if ((decoder.getEntity() instanceof Result) && !expandingRel) {
			metadata.add(asXmlElement("bestaccessright", "", getBestAccessright(), null));

			metadata.addAll(listFields(decoder.getEntity(), filter, defaults, expandingRel));
		}
		if ((decoder.getEntity() instanceof Project) && !expandingRel) {
			metadata.addAll(listFields(decoder.getEntity(), filter, defaults, expandingRel));
		}

		return metadata;
	}

	private Qualifier getBestAccessright() {
		Qualifier bestAccessRight = getQualifier("UNKNOWN", "not available", "dnet:access_modes");
		final LicenseComparator lc = new LicenseComparator();
		for (final Instance instance : ((Result) mainEntity.decodeEntity().getEntity()).getInstanceList()) {
			if (lc.compare(bestAccessRight, instance.getAccessright()) > 0) {
				bestAccessRight = instance.getAccessright();
			}
		}
		return bestAccessRight;
	}

	public Qualifier getQualifier(final String classid, final String classname, final String schemename) {
		return Qualifier.newBuilder().setClassid(classid).setClassname(classname).setSchemeid(schemename).setSchemename(schemename).build();
	}

	private List<String> listRelations() {

		final List<String> rels = Lists.newArrayList();

		for (final OafDecoder decoder : this.relations) {

			final OafRel rel = decoder.getOafRel();
			final OafEntity cachedTarget = rel.getCachedTarget();
			final OafRelDecoder relDecoder = OafRelDecoder.decode(rel);

			// if (!relDecoder.getRelType().equals(RelType.personResult) || relDecoder.getRelTargetId().equals(key)) {
			if (relDecoder.getRelSourceId().equals(key) || relDecoder.getRelTargetId().equals(key)) {

				final List<String> metadata = Lists.newArrayList();
				final TypeProtos.Type targetType = relDecoder.getTargetType(mainEntity.getEntity().getType());
				//final Set<String> relFilter = entityConfigTable.getFilter(targetType, relDecoder.getRelDescriptor());
				metadata.addAll(listFields(relDecoder.getSubRel(), entityConfigTable.getIncludeFilter(targetType, relDecoder.getRelDescriptor()), false, true));

				String semanticclass = "";
				String semanticscheme = "";

				final RelDescriptor relDescriptor = relDecoder.getRelDescriptor();

				if ((cachedTarget != null) && cachedTarget.isInitialized()) {

					//final Set<String> filter = entityConfigTable.getFilter(targetType, relDescriptor);
					final OafEntityDecoder d = OafEntityDecoder.decode(cachedTarget);
					metadata.addAll(decodeType(d, entityConfigTable.getIncludeFilter(targetType, relDescriptor), relDefaults, true));
					if (d.getType().equals(Type.result)) {
						for(Instance i : cachedTarget.getResult().getInstanceList()) {
							final List<String> fields = listFields(i, entityConfigTable.getIncludeFilter(targetType, relDecoder.getRelDescriptor()), false, true);
							metadata.addAll(fields);
						}
					}
				}

				final RelMetadata relMetadata = relDecoder.getRelMetadata();
				// debug
				if (relMetadata == null) {
					// System.err.println(this);
					semanticclass = semanticscheme = "UNKNOWN";
				} else {
					semanticclass = relClasses.getInverse(relMetadata.getSemantics().getClassname());
					semanticscheme = relMetadata.getSemantics().getSchemename();
				}

				final String rd = relDescriptor.getSubRelType().toString();
				incrementCounter(rd);

				final DataInfo info = decoder.getOaf().getDataInfo();
				if (info.getInferred()) {
					incrementCounter(rd + "_inferred");
				} else if(StringUtils.startsWith(info.getProvenanceaction().getClassid(), "sysimport:crosswalk")) {
					incrementCounter(rd + "_collected");
				} else if(StringUtils.startsWith(info.getProvenanceaction().getClassid(), "user:")) {
					incrementCounter(rd + "_claimed");
				}

				final LinkDescriptor ld = entityConfigTable.getDescriptor(relDecoder.getTargetType(mainEntity.getEntity().getType()), relDescriptor);

				final String relId = (ld != null) && !ld.isSymmetric() ? relDecoder.getRelTargetId() : relDecoder.getRelSourceId();

				rels.add(templateFactory.getRel(targetType, relId, Sets.newHashSet(metadata), semanticclass, semanticscheme, info.getInferred(), info.getTrust(),
						info.getInferenceprovenance(), info.getProvenanceaction().getClassid()));
			}
		}
		return rels;
	}

	// //////////////////////////////////

	private List<String> listChildren() {

		final List<String> children = Lists.newArrayList();
		for (final OafDecoder decoder : this.children) {
			final OafEntity cachedTarget = decoder.getOafRel().getCachedTarget();
			addChildren(children, cachedTarget, decoder.getRelDescriptor());
		}
		final OafEntityDecoder entity = mainEntity.decodeEntity();
		if (entity.getType().equals(Type.result)) {
			for (final Instance instance : ((Result) entity.getEntity()).getInstanceList()) {
				children.add(templateFactory.getInstance(instance.getHostedby().getKey(), listFields(instance, instanceFilter, false, false),
						listMap(instance.getUrlList(), identifier -> templateFactory.getWebResource(identifier))));
			}
			for (final ExternalReference er : ((Result) entity.getEntity()).getExternalReferenceList()) {
				// Set<String> filters = entityConfigTable.getFilter(Type.result, RelType.resultResult);
				final List<String> fields = listFields(er, null, false, false);
				children.add(templateFactory.getChild("externalreference", null, fields));
			}
		}

		return children;
	}

	private void addChildren(final List<String> children, final OafEntity target, final RelDescriptor relDescriptor) {
		final OafEntityDecoder decoder = OafEntityDecoder.decode(target);
		incrementCounter(relDescriptor.getSubRelType().toString());
		final Predicate<String> filter = entityConfigTable.getIncludeFilter(target.getType(), relDescriptor);
		children.add(templateFactory.getChild(decoder.getType().toString(), decoder.getId(), listFields(decoder.getMetadata(), filter, childDefaults, false)));
	}

	private List<String> listFields(final GeneratedMessage fields, final Predicate<String> filter, final boolean defaults, final boolean expandingRel) {

		final List<String> metadata = Lists.newArrayList();

		if (fields != null) {

			final Set<String> seen = Sets.newHashSet();

			final Map<FieldDescriptor, Object> filtered = filterFields(fields, filter);
			for (final Entry<FieldDescriptor, Object> e : filtered.entrySet()) {

				final String name = e.getKey().getName();
				seen.add(name);
				addFieldValue(metadata, e.getKey(), e.getValue(), expandingRel);
			}

			if (defaults) {
				final Iterable<FieldDescriptor> unseen =
						Iterables.filter(fields.getDescriptorForType().getFields(), fd -> !seen.contains(fd.getName()) && filter.apply(fd.getName()));
				for(FieldDescriptor fd : unseen){
					addFieldValue(metadata, fd, getDefault(fd), expandingRel);
				}
			}
		}
		return metadata;
	}

	private Object getDefault(final FieldDescriptor fd) {
		switch (fd.getType()) {
		case BOOL:
			return false;
		case BYTES:
			return "".getBytes();
		case MESSAGE: {
			if (Qualifier.getDescriptor().equals(fd.getMessageType())) return defaultQualifier();
			if (StructuredProperty.getDescriptor().equals(fd.getMessageType()))
				return StructuredProperty.newBuilder().setValue("").setQualifier(defaultQualifier()).build();
			if (KeyValue.getDescriptor().equals(fd.getMessageType())) return KeyValue.newBuilder().setKey("").setValue("").build();
			if (StringField.getDescriptor().equals(fd.getMessageType())) return StringField.newBuilder().setValue("").build();
			if (BoolField.getDescriptor().equals(fd.getMessageType())) return BoolField.newBuilder().buildPartial();
			return null;
		}
		case SFIXED32:
		case SFIXED64:
		case SINT32:
		case SINT64:
		case INT32:
		case INT64:
		case DOUBLE:
		case FIXED32:
		case FIXED64:
		case FLOAT:
			return 0;
		case STRING:
			return "";
		default:
			return null;
		}
	}

	private Qualifier defaultQualifier() {
		return Qualifier.newBuilder().setClassid("").setClassname("").setSchemeid("").setSchemename("").build();
	}

	@SuppressWarnings("unchecked")
	private void addFieldValue(final List<String> metadata, final FieldDescriptor fd, final Object value, final boolean expandingRel) {
		if ("dateofcollection".equals(fd.getName()) ||
			"dateoftransformation".equals(fd.getName()) ||
			"id".equals(fd.getName()) ||
				(value == null)) return;

		if (fd.getName().equals("datasourcetype")) {
			final String classid = ((Qualifier) value).getClassid();

			final Qualifier.Builder q = Qualifier.newBuilder((Qualifier) value);
			if (specialDatasourceTypes.contains(classid)) {
				q.setClassid("other").setClassname("other");
			}
			metadata.add(asXmlElement("datasourcetypeui", "", q.build(), null));
		}

		if (fd.isRepeated() && (value instanceof List<?>)) {
			for (final Object o : (List<Object>) value) {
				guessType(metadata, fd, o, expandingRel);
			}
		} else {
			guessType(metadata, fd, value, expandingRel);
		}
	}

	private void guessType(final List<String> metadata, final FieldDescriptor fd, final Object o, final boolean expandingRel) {

		if (fd.getType().equals(FieldDescriptor.Type.MESSAGE)) {

			if(Author.getDescriptor().equals(fd.getMessageType())) {

				final Author a = (Author) o;

				final StringBuilder sb = new StringBuilder("<creator rank=\"" + a.getRank() + "\"");
				if (a.hasName()) {
					sb.append(" name=\"" + escapeXml(a.getName()) + "\"");
				}
				if (a.hasSurname()) {
					sb.append(" surname=\"" + escapeXml(a.getSurname()) + "\"");
				}
				if (a.getPidCount() > 0) {
					a.getPidList().stream()
							.filter(kv -> StringUtils.isNotBlank(kv.getKey()) && StringUtils.isNotBlank(kv.getValue()))
							.forEach(kv -> {
								String pidType = escapeXml(kv.getKey())
										.replaceAll("\\W", "");
								String pidValue = escapeXml(kv.getValue());
								sb.append(String.format(" %s=\"%s\"", pidType, pidValue));
							});
				}

				sb.append(">" + escapeXml(a.getFullname()) + "</creator>");

				metadata.add(sb.toString());
			}

			if (Qualifier.getDescriptor().equals(fd.getMessageType())) {
				final Qualifier qualifier = (Qualifier) o;
				metadata.add(asXmlElement(fd.getName(), "", qualifier, null));
			}

			if (StructuredProperty.getDescriptor().equals(fd.getMessageType())) {
				final StructuredProperty sp = (StructuredProperty) o;
				metadata.add(asXmlElement(fd.getName(), sp.getValue(), sp.getQualifier(), sp.hasDataInfo() ? sp.getDataInfo() : null));

				if (!expandingRel && fd.getName().equals("pid")) {
					if (sp.getQualifier().getClassid().equalsIgnoreCase("doi")) {
						incrementCounter("doi");
					}
				}
			}

			if (KeyValue.getDescriptor().equals(fd.getMessageType())) {
				final KeyValue kv = (KeyValue) o;
				metadata.add("<" + fd.getName() + " name=\"" + escapeXml(kv.getValue()) + "\" id=\"" + escapeXml(removePrefix(kv.getKey())) + "\"/>");
			}

			if (StringField.getDescriptor().equals(fd.getMessageType())) {
				final String fieldName = fd.getName();

				if (fieldName.equals("fundingtree")) {
					final String xmlTree = o instanceof StringField ? ((StringField) o).getValue() : o.toString();

					if (expandingRel) {
						metadata.add(getRelFundingTree(xmlTree));
						fillContextMap(xmlTree);
					} else {
						metadata.add(xmlTree);
					}
				} else {
					final StringField sf = (StringField) o;
					final StringBuilder sb = new StringBuilder("<" + fd.getName());
					if (sf.hasDataInfo()) {
						final DataInfo dataInfo = sf.getDataInfo();
						dataInfoAsAttributes(sb, dataInfo);
					}
					sb.append(">" + escapeXml(sf.getValue()) + "</" + fd.getName() + ">");
					metadata.add(sb.toString());
				}
			}

			if (BoolField.getDescriptor().equals(fd.getMessageType())) {
				final BoolField bf = (BoolField) o;
				final StringBuilder sb = new StringBuilder("<" + fd.getName());
				if (bf.hasDataInfo()) {
					final DataInfo dataInfo = bf.getDataInfo();
					dataInfoAsAttributes(sb, dataInfo);
				}

				sb.append(">" + (bf.hasValue() ? bf.getValue() : "") + "</" + fd.getName() + ">");
				metadata.add(sb.toString());
			}

			if (Journal.getDescriptor().equals(fd.getMessageType()) && (o != null)) {
				final Journal j = (Journal) o;
				metadata.add("<journal " + "issn=\"" + escapeXml(j.getIssnPrinted()) + "\" " + "eissn=\"" + escapeXml(j.getIssnOnline()) + "\" " + "lissn=\""
						+ escapeXml(j.getIssnLinking()) + "\" " + "ep=\"" + escapeXml(j.getEp()) + "\" " + "iss=\"" + escapeXml(j.getIss()) + "\" " + "sp=\""
						+ escapeXml(j.getSp()) + "\" " + "vol=\"" + escapeXml(j.getVol()) + "\">" + escapeXml(j.getName()) + "</journal>");
			}

			if (Context.getDescriptor().equals(fd.getMessageType()) && (o != null)) {
				final String contextid = ((Context) o).getId();
				contextes.add(contextid);
				/* FIXME: Workaround for CLARIN mining issue: #3670#note-29 */
				if(contextid.equalsIgnoreCase("dh-ch::subcommunity::2")){
					contextes.add("clarin");
				}

			}

			if (ExtraInfo.getDescriptor().equals(fd.getMessageType()) && (o != null)) {

				final ExtraInfo e = (ExtraInfo) o;
				final StringBuilder sb = new StringBuilder("<" + fd.getName() + " ");

				sb.append("name=\"" + e.getName() + "\" ");
				sb.append("typology=\"" + e.getTypology() + "\" ");
				sb.append("provenance=\"" + e.getProvenance() + "\" ");
				sb.append("trust=\"" + e.getTrust() + "\"");
				sb.append(">");
				sb.append(e.getValue());
				sb.append("</" + fd.getName() + ">");

				extraInfo.add(sb.toString());
			}

		} else if (fd.getType().equals(FieldDescriptor.Type.ENUM)) {
			if (fd.getFullName().equals("eu.dnetlib.data.proto.OafEntity.type")) return;
			metadata.add(asXmlElement(fd.getName(), ((EnumValueDescriptor) o).getName(), null, null));
		} else {
			metadata.add(asXmlElement(fd.getName(), o.toString(), null, null));
		}
	}

	private StringBuilder dataInfoAsAttributes(final StringBuilder sb, final DataInfo dataInfo) {
		sb.append(" inferred=\"" + dataInfo.getInferred() + "\"");
		sb.append(" inferenceprovenance=\"" + dataInfo.getInferenceprovenance() + "\"");
		sb.append(" provenanceaction=\"" + dataInfo.getProvenanceaction().getClassid() + "\"");
		sb.append(" trust=\"" + dataInfo.getTrust() + "\" ");
		return sb;
	}

	private List<String> buildContexts(final Type type) {
		final List<String> res = Lists.newArrayList();

		if ((contextMapper != null) && !contextMapper.isEmpty() && type.equals(Type.result)) {

			XMLTag document = XMLDoc.newDocument(true).addRoot("contextRoot");

			for (final String context : contextes) {

				String id = "";
				for (final String token : Splitter.on("::").split(context)) {
					id += token;

					final ContextDef def = contextMapper.get(id);

					if (def == null) {
						continue;
						// throw new IllegalStateException(String.format("cannot find context for id '%s'", id));
					}

					if (def.getName().equals("context")) {
						final String xpath = "//context/@id='" + def.getId() + "'";
						if (!document.gotoRoot().rawXpathBoolean(xpath, new Object())) {
							document = addContextDef(document.gotoRoot(), def);
						}
					}

					if (def.getName().equals("category")) {
						final String rootId = StringUtils.substringBefore(def.getId(), "::");
						document = addContextDef(document.gotoRoot().gotoTag("//context[./@id='" + rootId + "']", new Object()), def);
					}

					if (def.getName().equals("concept")) {
						document = addContextDef(document, def).gotoParent();
					}
					id += "::";
				}
			}

			for (final org.w3c.dom.Element x : document.gotoRoot().getChildElement()) {
				try {
					res.add(asStringElement(x));
				} catch (final TransformerException e) {
					throw new RuntimeException(e);
				}
			}
		}

		return res;
	}

	private XMLTag addContextDef(final XMLTag tag, final ContextDef def) {
		tag.addTag(def.getName()).addAttribute("id", def.getId()).addAttribute("label", def.getLabel());
		if ((def.getType() != null) && !def.getType().isEmpty()) {
			tag.addAttribute("type", def.getType());
		}
		return tag;
	}

	private String asStringElement(final org.w3c.dom.Element element) throws TransformerException {
		final StringWriter buffer = new StringWriter();
		transformer.transform(new DOMSource(element), new StreamResult(buffer));
		return buffer.toString();
	}

	@SuppressWarnings("unchecked")
	private String getRelFundingTree(final String xmlTree) {
		String funding = "<funding>";
		try {
			final Document ftree = new SAXReader().read(new StringReader(xmlTree));
			funding = "<funding>";
			// String _id = "";

			funding += getFunderElement(ftree);

			for (final Object o : Lists.reverse(ftree.selectNodes("//fundingtree//*[starts-with(local-name(),'funding_level_')]"))) {
				final Element e = (Element) o;
				final String _id = e.valueOf("./id");
				funding += "<" + e.getName() + " name=\"" + escapeXml(e.valueOf("./name")) + "\">" + escapeXml(_id) + "</" + e.getName() + ">";
				// _id += "::";
			}
		} catch (final DocumentException e) {
			throw new IllegalArgumentException("unable to parse funding tree: " + xmlTree + "\n" + e.getMessage());
		} finally {
			funding += "</funding>";
		}
		return funding;
	}

	private String getFunderElement(final Document ftree) {
		final String funderId = ftree.valueOf("//fundingtree/funder/id/text()");
		final String funderShortName = ftree.valueOf("//fundingtree/funder/shortname/text()");
		final String funderName = ftree.valueOf("//fundingtree/funder/name/text()");
		final String funderJurisdiction = ftree.valueOf("//fundingtree/funder/jurisdiction/text()");

		return "<funder id=\"" + escapeXml(funderId) + "\" shortname=\"" + escapeXml(funderShortName) + "\" name=\"" + escapeXml(funderName)
				+ "\" jurisdiction=\"" + escapeXml(funderJurisdiction) + "\" />";
	}

	private void fillContextMap(final String xmlTree) {

		Document fundingPath;
		try {
			fundingPath = new SAXReader().read(new StringReader(xmlTree));
		} catch (final DocumentException e) {
			throw new RuntimeException(e);
		}
		try {
			final Node funder = fundingPath.selectSingleNode("//funder");

			if (funder != null) {

				final String funderShortName = funder.valueOf("./shortname");
				contextes.add(funderShortName);

				contextMapper.put(funderShortName, new ContextDef(funderShortName, funder.valueOf("./name"), "context", "funding"));
				final Node level0 = fundingPath.selectSingleNode("//funding_level_0");
				if (level0 != null) {
					final String level0Id = Joiner.on("::").join(funderShortName, level0.valueOf("./name"));
					contextMapper.put(level0Id, new ContextDef(level0Id, level0.valueOf("./description"), "category", ""));
					final Node level1 = fundingPath.selectSingleNode("//funding_level_1");
					if (level1 == null) {
						contextes.add(level0Id);
					} else {
						final String level1Id = Joiner.on("::").join(level0Id, level1.valueOf("./name"));
						contextMapper.put(level1Id, new ContextDef(level1Id, level1.valueOf("./description"), "concept", ""));
						final Node level2 = fundingPath.selectSingleNode("//funding_level_2");
						if (level2 == null) {
							contextes.add(level1Id);
						} else {
							final String level2Id = Joiner.on("::").join(level1Id, level2.valueOf("./name"));
							contextMapper.put(level2Id, new ContextDef(level2Id, level2.valueOf("./description"), "concept", ""));
							contextes.add(level2Id);
						}
					}
				}
			}
		} catch (final NullPointerException e) {
			throw new IllegalArgumentException("malformed funding path: " + xmlTree, e);
		}
	}

	private String asXmlElement(final String name, final String value, final Qualifier q, final DataInfo dataInfo) {
		StringBuilder sb = new StringBuilder();
		sb.append("<");
		sb.append(name);
		if (q != null) {
			sb.append(getAttributes(q));
		}
		if (dataInfo != null) {
			sb = dataInfoAsAttributes(sb, dataInfo);
		}
		if ((value == null) || value.isEmpty()) {
			sb.append("/>");
			return sb.toString();
			// return "<" + name + getAttributes(q) + "/>";
		}

		sb.append(">");
		// sb.append(escapeXml(Normalizer.normalize(value, Normalizer.Form.NFD)));
		sb.append(escapeXml(value));
		sb.append("</");
		sb.append(name);
		sb.append(">");

		return sb.toString();
		// return "<" + name + getAttributes(q) + ">" + escapeXml(value) + "</" + name + ">";
	}

	private String getAttributes(final Qualifier q) {
		if (q == null) return "";

		final StringBuilder sb = new StringBuilder();
		for (final Entry<FieldDescriptor, Object> e : q.getAllFields().entrySet()) {
			// sb.append(" " + e.getKey().getName() + "=\"" + escapeXml(e.getValue().toString()) + "\"");
			sb.append(" ");
			sb.append(e.getKey().getName());
			sb.append("=\"");
			sb.append(escapeXml(e.getValue().toString()));
			sb.append("\"");
		}
		return sb.toString();
	}


	private Map<FieldDescriptor, Object> filterFields(final GeneratedMessage fields, final Predicate<String> acceptFilter) {
		if(acceptFilter == null) return fields.getAllFields();
		final Map<FieldDescriptor, Object> res = Maps.newHashMap();
		for(Entry<FieldDescriptor, Object> e : fields.getAllFields().entrySet()) {
			if (acceptFilter.apply(e.getKey().getName())) {
				res.put(e.getKey(), e.getValue());
			}
		}
		return res;
	}



	private List<String> countersAsXml() {
		final List<String> out = Lists.newArrayList();
		for (final Entry<String, Integer> e : counters.entrySet()) {
			out.add(String.format("<counter_%s value=\"%s\"/>", e.getKey(), e.getValue()));
		}
		return out;
	}

	private void incrementCounter(final String type) {
		if (!counters.containsKey(type)) {
			counters.put(type, 1);
		} else {
			counters.put(type, counters.get(type) + 1);
		}
	}

	@Override
	public String toString() {
		final StringBuilder sb = new StringBuilder();
		sb.append("################################################\n");
		sb.append("ID: ").append(key).append("\n");
		if (mainEntity != null) {
			sb.append("MAIN ENTITY:\n").append(mainEntity.getEntity().toString() + "\n");
		}
		if (relations != null) {
			sb.append("\nRELATIONS:\n");
			for (final OafDecoder decoder : relations) {
				sb.append(decoder.getOafRel().toString() + "\n");
			}
		}
		if (children != null) {
			sb.append("\nCHILDREN:\n");
			for (final OafDecoder decoder : children) {
				sb.append(decoder.getOafRel().toString() + "\n");
			}
		}
		return sb.toString();
	}

}
