package eu.dnetlib.data.mapreduce.hbase.index;

import eu.dnetlib.data.mapreduce.util.OafDecoder;
import eu.dnetlib.data.mapreduce.util.OafHbaseUtils;
import eu.dnetlib.data.mapreduce.util.OafRowKeyDecoder;
import eu.dnetlib.data.proto.KindProtos;
import eu.dnetlib.data.proto.OafProtos;
import eu.dnetlib.data.proto.TypeProtos;
import org.apache.hadoop.hbase.io.ImmutableBytesWritable;
import org.apache.hadoop.io.NullWritable;
import org.apache.hadoop.io.Text;
import org.apache.hadoop.mapreduce.Counter;
import org.apache.hadoop.mapreduce.Reducer;

public class InfospaceCountsReducer extends Reducer<Text, ImmutableBytesWritable, NullWritable, NullWritable> {

	public static final String ENTITY = KindProtos.Kind.entity.toString();

	@Override
	protected void reduce(final Text key, final Iterable<ImmutableBytesWritable> values, final Context context) {
		try {
			final OafRowKeyDecoder keyDecoder = OafRowKeyDecoder.decode(key.toString());
			for (final ImmutableBytesWritable bytes : values) {
				final OafDecoder decoder = OafHbaseUtils.decode(bytes);
				final TypeProtos.Type type = keyDecoder.getType();

				final OafProtos.Oaf oaf = decoder.getOaf();

				switch (decoder.getKind()) {
					case entity:
						if (deletedByInference(oaf)) {
							if (isInvisible(oaf)) {
								incrementCounter(context, ENTITY, String.format("%s (deleted true / invisible true)", getEntityType(oaf, type)), 1);
							} else {
								incrementCounter(context, ENTITY, String.format("%s (deleted true / invisible false)", getEntityType(oaf, type)), 1);
							}
						} else {

							if (isInvisible(oaf)) {
								incrementCounter(context, ENTITY, String.format("%s (deleted false / invisible true)", getEntityType(oaf, type)), 1);
							} else {
								incrementCounter(context, ENTITY, String.format("%s (deleted false / invisible false)", getEntityType(oaf, type)), 1);
							}
						}
						break;
					case relation:
						if (deletedByInference(oaf)) {
							incrementCounter(context, String.format("%s (deleted true)", ENTITY), decoder.getCFQ(), 1);
						} else {
							incrementCounter(context, String.format("%s (deleted false)", ENTITY), decoder.getCFQ(), 1);
						}
						break;
					default:
						throw new IllegalArgumentException("unknow type: " + decoder.getKind());
				}
			}
		} catch (final Throwable e) {
			context.getCounter("error", e.getClass().getName()).increment(1);
			throw new RuntimeException(e);
		}
	}

	private void incrementCounter(final Reducer.Context context, final String k, final String t, final int n) {
		getCounter(context, k, t).increment(n);
	}

	private Counter getCounter(final Reducer.Context context, final String k, final String t) {
		return context.getCounter(k, t);
	}

	private String getEntityType(final OafProtos.Oaf oaf, final TypeProtos.Type type) {
		switch (type) {
			case result:
				return oaf.getEntity().getResult().getMetadata().getResulttype().getClassid();
			default:
				return type.toString();
		}
	}

	private boolean deletedByInference(final OafProtos.Oaf oaf) {
		return oaf.getDataInfo().getDeletedbyinference();
	}

	private boolean isInvisible(final OafProtos.Oaf oaf) {
		return oaf.getDataInfo().getInvisible();
	}

}
