package eu.dnetlib.data.mapreduce.hbase.broker;

import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import eu.dnetlib.broker.objects.OpenAireEventPayload;
import eu.dnetlib.data.mapreduce.hbase.broker.mapping.HighlightFactory;
import eu.dnetlib.data.mapreduce.hbase.broker.mapping.OpenAireEventPayloadFactory;
import eu.dnetlib.data.mapreduce.hbase.broker.model.EventWrapper;
import eu.dnetlib.data.proto.OafProtos.Oaf;
import eu.dnetlib.data.proto.ResultProtos.Result.Instance;
import org.apache.commons.lang3.StringUtils;
import org.dom4j.DocumentException;

import static eu.dnetlib.data.mapreduce.hbase.broker.mapping.EventFactory.asEvent;
import static eu.dnetlib.data.mapreduce.util.OafHbaseUtils.getKey;

/**
 * Created by claudio on 26/07/16.
 */
public class OAVersionEventFactory {

	public static final String OPEN = "OPEN";

	public static List<EventWrapper> process(final Oaf current, final Oaf other, final float trust, final Set<String> untrustedOaDsList)
			throws IOException, InterruptedException, DocumentException {
		return new OAVersionEventFactory().processOAVersion(current, other, trust, untrustedOaDsList);
	}

	private List<EventWrapper> processOAVersion(final Oaf current, final Oaf other, final float trust, final Set<String> untrustedOaDsList)
			throws IOException, InterruptedException, DocumentException {

		final List<EventWrapper> events = Lists.newArrayList();
		final String otherDsId = StringUtils.substringAfter(getKey(other.getEntity().getCollectedfromList()), "|");

		// OPEN ACCESS VERSION
		if (!untrustedOaDsList.contains(otherDsId)) {

			if (hasAccess(other, OPEN, false)) {

				// MORE
				events.add(doProcessOAVersion(current, other, Topic.ENRICH_MORE_OA_VERSION, trust));

				// MISSING
				if (!hasAccess(current, OPEN, true)) {
					events.add(doProcessOAVersion(current, other, Topic.ENRICH_MISSING_OA_VERSION, trust));
				}
			}
		}
		return events;
	}

	private EventWrapper doProcessOAVersion(final Oaf current, final Oaf other, final Topic topic, final float trust)
			throws IOException, InterruptedException, DocumentException {
		final Oaf.Builder prototype = Oaf.newBuilder(current);

		final Map<String, Instance> instances =
				other.getEntity().getResult().getInstanceList().stream()
						.filter(i -> OPEN.equalsIgnoreCase(i.getAccessright().getClassid()))
						.filter(i -> i.getUrlList() != null && !i.getUrlList().isEmpty())
						.collect(Collectors.toMap(
								i -> {
									final List<String> url = i.getUrlList();
									return StringUtils.lowerCase(url.get(0));
								},
								Function.identity(),
								(i1, i2) -> i1));

		prototype.getEntityBuilder().getResultBuilder().addAllInstance(instances.values());

		final Oaf oaf = prototype.build();

		final OpenAireEventPayload payload =
				HighlightFactory.highlightEnrichOa(OpenAireEventPayloadFactory.fromOAF(oaf.getEntity(), other.getEntity(), trust), instances.values());

		return EventWrapper.newInstance(
				asEvent(oaf.getEntity(), topic, payload, other.getEntity(), trust),
				payload.getHighlight().getInstances().stream().filter(Objects::nonNull).map(i -> i.getLicense() + ":" + i.getUrl()).sorted()
						.collect(Collectors.joining(", ")),
				topic.getValue());
	}

	private boolean hasAccess(final Oaf oaf, final String access, final boolean strict) {
		final Predicate<Instance> p = i -> access.equalsIgnoreCase(i.getAccessright().getClassid());
		final List<Instance> i = oaf.getEntity().getResult().getInstanceList();
		return strict ? Iterables.all(i, p) : Iterables.any(i, p);
	}

}
