package eu.dnetlib.data.mapreduce.hbase.propagation.communitythroughorganization;

import com.google.gson.Gson;
import eu.dnetlib.data.mapreduce.hbase.propagation.Value;
import eu.dnetlib.data.mapreduce.util.OafRowKeyDecoder;
import eu.dnetlib.data.proto.OafProtos;
import eu.dnetlib.data.proto.TypeProtos;
import org.apache.commons.lang3.StringUtils;
import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.io.ImmutableBytesWritable;
import org.apache.hadoop.hbase.mapreduce.TableMapper;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.io.Text;

import java.io.IOException;
import java.util.Set;

import static eu.dnetlib.data.mapreduce.hbase.propagation.PropagationConstants.*;
import static eu.dnetlib.data.mapreduce.hbase.propagation.Utils.getEntity;
import static eu.dnetlib.data.mapreduce.hbase.propagation.Utils.getRelationTarget;

public class PropagationCommunityThroughOrganizationMapper extends TableMapper<ImmutableBytesWritable, Text> {
    private Text valueOut;
    private ImmutableBytesWritable keyOut;
    private OrganizationMap organizationMap;

    //seleziono il tipo della entry:
    //Result:
    //se non e' deleted by inference ed ha organizzazioni a cui e' associato,
    // //emetto id della relazione ed id del risultato per ogni organizzazione a cui e' associato
    //ORGANIZATION:
    //se non e' deleted by inference e non e' deduplicata emetto l'id della organizzazione
    //se non e' deleted by inference ed e' deduplicata, emetto id della organizzazione ed id del deduplicato e lista delle organizzazioni mergiate
    @Override
    protected void setup(final Context context) throws IOException, InterruptedException {
        super.setup(context);
        valueOut = new Text();
        keyOut = new ImmutableBytesWritable();
        organizationMap = new Gson().fromJson(context.getConfiguration().get("organizationtoresult.community.map"), OrganizationMap.class);
        System.out.println("got organizationtoresult map: " + new Gson().toJson(organizationMap)) ;
    }

    @Override
    protected void map(final ImmutableBytesWritable keyIn, final Result value, final Context context) throws IOException, InterruptedException {
        final TypeProtos.Type type = OafRowKeyDecoder.decode(keyIn.copyBytes()).getType();
        final OafProtos.OafEntity entity = getEntity(value, type);//getEntity already verified that it is not delByInference
        if (entity != null) {
            switch (type) {
                case organization:
                    DedupedList communityList = getCommunityList(Bytes.toString(keyIn.get()),
                            getRelationTarget(value, DEDUP_RELATION_ORGANIZATION + REL_DEDUP_REPRESENTATIVE_RESULT, context, COUNTER_PROPAGATION), context);
                    if (communityList.size() > 0){
                        valueOut.set(Value.newInstance(
                                new Gson().toJson(
                                        communityList, //search for organizationtoresult it merges
                                        DedupedList.class),
                                ORGANIZATION_COMMUNITY_TRUST,
                                Type.fromorganization).toJson());
                        context.write(keyIn, valueOut);
                        context.getCounter(COUNTER_PROPAGATION, "emit for organizationtoresult ").increment(1);
                    }else{
                        context.getCounter(COUNTER_PROPAGATION, "community list size = 0 ").increment(1);
                    }

                    break;
                case result:
                    Set<String> result_organization = getRelationTarget(value, RELATION_ORGANIZATION + REL_RESULT_ORGANIZATION, context, COUNTER_PROPAGATION);
                    for(String org: result_organization)
                        emit(org, Bytes.toString(keyIn.get()), context);
                    break;
            }
        }
    }

    private DedupedList getCommunityList(String organizationId, Set<String> relationTarget, Context context) {
        DedupedList communityList = new DedupedList();
        relationTarget.stream().forEach(org -> communityList.addAll(organizationMap.get(StringUtils.substringAfter(org, "|"))));
        communityList.addAll(organizationMap.get(StringUtils.substringAfter(organizationId,"|")));
        communityList.stream().forEach(c->context.getCounter(COUNTER_PROPAGATION,"found organizationtoresult for " + c).increment(1));
        return communityList;
    }

    private void emit(String org, String resId, Context context) throws IOException, InterruptedException {
        keyOut.set(Bytes.toBytes(org));
        valueOut.set(Value.newInstance(resId, ORGANIZATION_COMMUNITY_TRUST, Type.fromresult).toJson());
        context.write(keyOut,valueOut);
        context.getCounter(COUNTER_PROPAGATION, "emit for result").increment(1);
    }

}
