package eu.dnetlib.data.mapreduce.hbase.oai.config;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import javax.annotation.Nullable;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Collections2;
import com.google.common.collect.Multimap;
import com.google.common.collect.Multimaps;
import com.google.common.collect.Sets;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import eu.dnetlib.data.mapreduce.hbase.oai.utils.MDFInfo;
import eu.dnetlib.data.mapreduce.hbase.oai.utils.PublisherField;
import eu.dnetlib.data.mapreduce.hbase.oai.utils.SetInfo;

/**
 * An instance of this class represents an OAI Configuration profile.
 * 
 * @author alessia
 * 
 */
public class OAIConfiguration {

	/**
	 * Map for OAI sets. Keys: setSpec, values: SetInfo instances.
	 */
	private Map<String, SetInfo> setsMap;

	/**
	 * Map for metadata formats. Keys: metadata prefix, values: MDFInfo instances.
	 */
	private Map<String, MDFInfo> mdFormatsMap;

	private List<PublisherField> fields;

	private List<String> fieldNames;

	/**
	 * Used to generate the OAI identifiers compliant to the protocol. See
	 * http://www.openarchives.org/OAI/openarchivesprotocol.html#UniqueIdentifier. See property services.oai.publisher.id.scheme.
	 */
	private String idScheme;
	/**
	 * Used to generate the OAI identifiers compliant to the protocol. See
	 * http://www.openarchives.org/OAI/openarchivesprotocol.html#UniqueIdentifier. See property services.oai.publisher.id.namespace.
	 */
	private String idNamespace;

	private Set<MDFInfo> sourcesMDF = Sets.newHashSet();

	private Multimap<String, String> enrichmentXPaths;

	public Collection<PublisherField> getFieldsFor(final String format, final String layout, final String interpretation) {
		final String mdRef = Joiner.on("-").join(format, layout, interpretation);
		Collection<PublisherField> filtered = Collections2.filter(this.getFields(), theField -> {
			if (theField.getSources() == null) return false;
			return theField.getSources().containsKey(mdRef);
		});
		return Collections2.transform(filtered, new Function<PublisherField, PublisherField>() {
			@Nullable
			@Override
			public PublisherField apply(@Nullable final PublisherField publisherField) {
				Multimap<String, String> filtered = Multimaps.filterKeys(publisherField.getSources(), k -> k.equalsIgnoreCase(mdRef));
				return new PublisherField(publisherField.getFieldName(), publisherField.isRepeatable(), filtered);
			}
		});
	}

	public Collection<String> getEnrichmentXPathsFor(final String format, final String layout, final String interpretation) {
		final String mdRef = Joiner.on("-").join(format, layout, interpretation);
		return enrichmentXPaths.get(mdRef);
	}

	public Map<String, SetInfo> getSetsMap() {
		return setsMap;
	}

	public void setSetsMap(final Map<String, SetInfo> setsMap) {
		this.setsMap = setsMap;
	}

	public Map<String, MDFInfo> getMdFormatsMap() {
		return mdFormatsMap;
	}

	public void setMdFormatsMap(final Map<String, MDFInfo> mdFormatsMap) {
		this.mdFormatsMap = mdFormatsMap;
	}

	public List<PublisherField> getFields() {
		return fields;
	}

	public void setFields(final List<PublisherField> fields) {
		this.fields = fields;
	}

	public List<String> getFieldNames() {
		return fieldNames;
	}

	public void setFieldNames(final List<String> fieldNames) {
		this.fieldNames = fieldNames;
	}

	public Set<MDFInfo> getSourcesMDF() {
		return sourcesMDF;
	}

	public void setSourcesMDF(final Set<MDFInfo> sourcesMDF) {
		this.sourcesMDF = sourcesMDF;
	}

	public String getIdScheme() {
		return idScheme;
	}

	public void setIdScheme(final String idScheme) {
		this.idScheme = idScheme;
	}

	public String getIdNamespace() {
		return idNamespace;
	}

	public void setIdNamespace(final String idNamespace) {
		this.idNamespace = idNamespace;
	}

	public Multimap<String, String> getEnrichmentXPaths() {
		return enrichmentXPaths;
	}

	public void setEnrichmentXPaths(final Multimap<String, String> enrichmentXPaths) {
		this.enrichmentXPaths = enrichmentXPaths;
	}

	@Override
	public String toString() {
		Gson gson = new GsonBuilder().setPrettyPrinting().create();
		return gson.toJson(this);
	}

}
