package eu.dnetlib.data.mapreduce.hbase.broker.model;

import java.util.Date;
import java.util.stream.Collectors;

import org.apache.commons.codec.digest.DigestUtils;

public class EventWrapper {

	private final EventMessage event;

	private final String counterName;

	private final String highlightValue;

	public static EventWrapper newInstance(final EventMessage event, final String highlightValue, final String counterName) {
		return new EventWrapper(event, highlightValue, counterName);
	}

	public EventWrapper(final EventMessage event, final String highlightValue, final String counterName) {
		this.event = event;
		this.counterName = counterName;
		this.highlightValue = highlightValue;
	}

	public Event asBrokerEvent() {

		final long now = new Date().getTime();

		final Event res = new Event();

		final String eventId =
			calculateEventId(event.getTopic(), event.getMap().get("target_publication_id").toString(), highlightValue);

		res.setEventId(eventId);
		res.setProducerId(event.getProducerId());
		res.setPayload(event.getPayload());
		res.setMap(event.getMap()
			.entrySet()
			.stream()
			.filter(e -> e.getValue().asObject() != null)
			.collect(Collectors.toMap(e -> e.getKey(), e -> e.getValue().asObject())));
		res.setTopic(event.getTopic());
		res.setCreationDate(now);
		res.setExpiryDate(calculateExpiryDate(now, event.getTthDays()));
		res.setInstantMessage(event.getTthDays() == 0);
		return res;
	}

	private String calculateEventId(final String topic, final String publicationId, final String value) {
		return "event-"
			+ DigestUtils.md5Hex(topic).substring(0, 6) + "-"
			+ DigestUtils.md5Hex(publicationId).substring(0, 8) + "-"
			+ DigestUtils.md5Hex(value).substring(0, 8);
	}

	private Long calculateExpiryDate(final long now, final int ttlDays) {
		if (ttlDays < 0) {
			return null;
		} else if (ttlDays == 0) {
			return now;
		} else {
			return now + ttlDays * 24 * 60 * 60 * 1000;
		}
	}

	public EventMessage getEvent() {
		return event;
	}

	public String getCounterName() {
		return counterName;
	}
}
