package eu.dnetlib.data.mapreduce.hbase.misc;

import java.io.IOException;

import com.google.protobuf.InvalidProtocolBufferException;
import eu.dnetlib.data.proto.OafProtos.Oaf;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.hbase.io.ImmutableBytesWritable;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.io.Text;
import org.apache.hadoop.mapred.JobTracker.IllegalStateException;
import org.apache.hadoop.mapreduce.Reducer;

public class PredatoryJournalsReducer extends Reducer<Text, ImmutableBytesWritable, Text, Text> {

	private static final Log log = LogFactory.getLog(PredatoryJournalsReducer.class); // NOPMD by marko on 11/24/08 5:02 PM

	private static final String PREDATORY_JOURNALS = "Predatory Journal Counters";
	private static final String PUB_CHECKS = "Publication Counters";


	private Text keyOut = new Text();
	private Text valueOut = new Text();

	@Override
	protected void setup(final Context context) throws IOException, InterruptedException {
		super.setup(context);
	}

	@Override
	protected void reduce(final Text key, final Iterable<ImmutableBytesWritable> values, final Context context) throws IOException, InterruptedException {

		try {
			int count = 0;
			String dsName = "";
			Oaf oaf = null;
			for(ImmutableBytesWritable ibw : values) {
				final byte[] data = ibw.copyBytes();
				oaf = parseOaf(data);

				if (oaf != null) {
					dsName = oaf.getEntity().getDatasource().getMetadata().getOfficialname().getValue();
				} else {
					final int one = Bytes.toInt(data);
					if (one != 1) {
						throw new IllegalStateException("woops unexpected number, got " + one);
					}
					count += one;
				}
			}

			if (count > 0 && oaf != null) {
				valueOut.set(String.format("\"%s\", %s", dsName.replaceAll(",", " "), count));
				context.write(keyOut, valueOut);
				context.getCounter(PREDATORY_JOURNALS, "journals").increment(count);
			}

			if (oaf != null && count == 0) {
				context.getCounter(PUB_CHECKS, "pubs from journals not linked to projects").increment(1);
			}

			if (oaf == null && count > 0) {
				context.getCounter(PUB_CHECKS, "pubs from non-journal and linked to projects").increment(count);
			}

		} catch (final Exception e) {
			context.getCounter("error", e.getClass().getName()).increment(1);
			throw new RuntimeException(e);
		}
	}

	private Oaf parseOaf(final byte[] data) {
		try {
			return Oaf.parseFrom(data);
		} catch (InvalidProtocolBufferException e) {
			return null;
		}
	}

}
