package eu.dnetlib.data.mapreduce.hbase.propagation.organizationtoresult.datasource;

import com.google.common.base.Splitter;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.google.gson.Gson;
import eu.dnetlib.data.mapreduce.hbase.propagation.Value;
import eu.dnetlib.data.mapreduce.hbase.propagation.compositekeys.InstOrgKey;
import eu.dnetlib.data.mapreduce.hbase.propagation.organizationtoresult.Emit;
import eu.dnetlib.data.mapreduce.util.OafRowKeyDecoder;
import eu.dnetlib.data.proto.*;
import org.apache.commons.collections.MapUtils;
import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.io.ImmutableBytesWritable;
import org.apache.hadoop.hbase.mapreduce.TableMapper;
import org.apache.hadoop.io.Text;

import java.io.IOException;
import java.io.InterruptedIOException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import static eu.dnetlib.data.mapreduce.hbase.propagation.PropagationConstants.*;
import static eu.dnetlib.data.mapreduce.hbase.propagation.Utils.getEntity;
import static eu.dnetlib.data.mapreduce.hbase.propagation.Utils.getRelationTarget;


/**
 * Propagates products to organizationtoresult
 * One product linked to one organizationtoresult and to another product through strong semantic relation
 * (isSupplementedBy, is SupplementTo)
 * The linked result is linked also to the organizationtoresult.
 * First implementation for institutional repositories
 */
//public class PropagationProductToOrganizationMapper extends TableMapper<ImmutableBytesWritable, Text> {
public class PropagationProductToOrganizationMapper extends TableMapper<InstOrgKey, Text> {
    private Set<String> datasourceTypes = Sets.newHashSet("pubsrepository::institutional");
    private Set<String> whiteList = Sets.newHashSet("10|opendoar____::300891a62162b960cf02ce3827bb363c");//PUMA
    private Text valueOut;
   // private InstOrgKey keyOut;
    private String[] sem_rels;
//    private String propagation_type;

    @Override
    protected void setup(final Context context) throws IOException, InterruptedException {
        super.setup(context);

        valueOut = new Text();
       // keyOut = new ImmutableBytesWritable();

        datasourceTypes.addAll(getParam(context, "datasource.types"));
        whiteList.addAll(getParam(context, "datasource.whitelist"));

    }


    @Override
    protected void map(final ImmutableBytesWritable keyIn, final Result value, final Context context) throws IOException, InterruptedException {

        final TypeProtos.Type type = OafRowKeyDecoder.decode(keyIn.copyBytes()).getType();
        final OafProtos.OafEntity entity = getEntity(value, type);
        if (entity != null) {
            final String id = entity.getId();
            Emit e;
            switch (type) {
                case datasource:
                    final DatasourceProtos.Datasource datasource = entity.getDatasource();

                    if (datasource == null) {
                        throw new RuntimeException("oaf type is datasource, but datasource proto is not found in oafproto");
                    }

                    String dsType = datasource.getMetadata().getDatasourcetype().getClassid();


                        if (datasourceTypes.contains(dsType) || whiteList.contains(id)) {
                            valueOut.set(Value.newInstance(ONE, PRODUCT_TO_ORGANIZATION, Type.valid).toJson());
                            //valueOut.set(Value.newInstance(new Gson().toJson(e, Emit.class), PRODUCT_TO_ORGANIZATION, PropagationConstants.Type.valid).toJson());
                            context.getCounter(COUNTER_PROPAGATION, String.format("%s in propagation allowed list or id whitelisted", dsType)).increment(1);
                        } else {
                            //valueOut.set(Value.newInstance(new Gson().toJson(e, Emit.class), PRODUCT_TO_ORGANIZATION, Type.notvalid).toJson());
                            valueOut.set(Value.newInstance(ZERO, PRODUCT_TO_ORGANIZATION, Type.notvalid).toJson());
                            context.getCounter(COUNTER_PROPAGATION, String.format("%s in propagation not allowed list", dsType)).increment(1);
                        }
                        context.write(InstOrgKey.datasource(id), valueOut);



                    break;
                case organization:
                    OrganizationProtos.Organization organization = entity.getOrganization();

                    if (organization == null) {
                        throw new RuntimeException("oaf type is organizationtoresult, but organizationtoresult proto is not found in oafproto");
                    }

                    final Map<byte[], byte[]> ds_org = value.getFamilyMap((DATASOURCE_RELATION_ORGANIZATION + REL_ORGANIZATION_DATASOURCE).getBytes());
                    if (MapUtils.isNotEmpty(ds_org)) {

                        for(String dsId: ds_org.keySet().stream().map(String::new).collect(Collectors.toList())){
                            valueOut.set(Value.newInstance(id, PRODUCT_TO_ORGANIZATION, Type.fromorganization).toJson());
                            context.write(InstOrgKey.organization(dsId), valueOut);
                            context.getCounter(COUNTER_PROPAGATION, "emit for organization ").increment(1);
                        }

                    }

                    break;
                case result:
                    ResultProtos.Result result = entity.getResult();


                    Set<String> institution = getRelationTarget(value, RELATION_ORGANIZATION + REL_RESULT_ORGANIZATION, context, COUNTER_PROPAGATION);


                    e = new Emit();
                    e.setId(id);
                    e.setResult_list(institution.stream().collect(Collectors.toList()));
                    valueOut.set(Value.newInstance(new Gson().toJson(e, Emit.class), PRODUCT_TO_ORGANIZATION, Type.fromresult).toJson());
                    for (ResultProtos.Result.Instance instance : result.getInstanceList()) {
                        //todo add check if key is not empty and field is not null
                        String hostedBy = instance.getHostedby().getKey();
                        context.write( InstOrgKey.publication(hostedBy),valueOut);
                        context.getCounter(COUNTER_PROPAGATION, "emit hostedby | collectedfrom for result ").increment(1);
                        String collectedFrom = instance.getCollectedfrom().getKey();
                        if (!hostedBy.equals(collectedFrom)) {
                            context.write(InstOrgKey.publication(collectedFrom), valueOut);
                            context.getCounter(COUNTER_PROPAGATION, "emit hostedby | collectedfrom for result ").increment(1);
                        }
                    }

                    break;

            }
        }
    }

    private ArrayList<String> getParam(Context context, String s) {
        return Lists.newArrayList(Splitter.on(",").omitEmptyStrings().split(context.getConfiguration().get(s, "")));
    }
}
