package eu.dnetlib.data.mapreduce.hbase.broker.enrich;

import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Streams;
import eu.dnetlib.data.mapreduce.hbase.broker.ProjectEventFactory;
import eu.dnetlib.data.mapreduce.hbase.broker.model.EventWrapper;
import eu.dnetlib.data.proto.OafProtos.Oaf;
import org.apache.commons.lang.StringUtils;
import org.apache.hadoop.hbase.io.ImmutableBytesWritable;
import org.dom4j.DocumentException;

import static eu.dnetlib.data.mapreduce.util.OafHbaseUtils.getKey;

/**
 * Created by claudio on 08/07/16.
 */
public class ProjectEnrichmentReducer extends AbstractEnrichmentReducer {

	@Override
	protected String counterGroup() {
		return "Broker Enrichment projects";
	}

	@Override
	protected void reduce(final ImmutableBytesWritable key, final Iterable<ImmutableBytesWritable> values, final Context context)
			throws IOException, InterruptedException {

		final Map<String, Oaf> projects = Maps.newHashMap();
		final List<Oaf> pubs = Lists.newArrayList();

		Streams.stream(values).limit(LIMIT).map(EnrichmentReducer::toOaf).forEach(oaf -> {
			switch (oaf.getEntity().getType()) {
			case result:
				pubs.add(oaf);
				break;
			case project:
				projects.put(oaf.getEntity().getId(), oaf);
				break;
			default:
				throw new IllegalArgumentException("invalid type: " + oaf.getEntity().getType());
			}
		});

		if (projects.isEmpty() || pubs.isEmpty()) {
			context.getCounter(counterGroup(), "Events Skipped - Nothing to do").increment(1);
			return;
		}

		try {
			generateEvents(pubs, projects, context);
		} catch (final DocumentException e) {
			throw new RuntimeException(e);
		}
	}

	private void generateEvents(final List<Oaf> pubs, final Map<String, Oaf> projects, final Context context)
			throws IOException, InterruptedException, DocumentException {

		final List<Oaf> valid = pubs.stream()
				.map(p -> addProjects(p, projects))
				.filter(r -> !r.getEntity().getCachedOafRelList().isEmpty())
				.collect(Collectors.toList());

		if (valid.isEmpty()) {
			context.getCounter(counterGroup(), "Events Skipped - Missing project").increment(1);
			return;
		}

		for (final Oaf current : valid) {

			final String currentId = current.getEntity().getId();

			final String currentDsId = StringUtils.substringAfter(getKey(current.getEntity().getCollectedfromList()), "|");
			final String currentDsType = dsTypeMap.get(currentDsId);

			// System.out.println(String.format("'%s' -> '%s'", currentDsId, currentDsType));

			if (StringUtils.isBlank(currentDsType) && !dsWhitelist.contains(currentDsId)) {
				context.getCounter(counterGroup(), "Events Skipped - Datasource type excluded").increment(1);
			} else if (dsBlacklist.contains(currentDsId)) {
				context.getCounter(counterGroup(), "Events Skipped - Datasource blacklisted").increment(1);
			} else {
				final List<EventWrapper> events = Lists.newArrayList();

				if (valid.size() == 1) {
					events.addAll(ProjectEventFactory.process(context, valid.get(0), baseUrlMap));
				} else {

					for (final Oaf other : valid) {
						final String otherId = other.getEntity().getId();
						if (!currentId.equals(otherId)) {

							final double similarity = similarity(current, other);

							if (similarity >= dedupConf.getWf().getThreshold()) {

								final float trust = scale(similarity);

								events.addAll(ProjectEventFactory.process(context, current, other, trust, baseUrlMap));
							} else {
								context.getCounter(counterGroup(), "d < " + dedupConf.getWf().getThreshold()).increment(1);
							}

						}
					}
				}
				emit(events, context);
			}
		}

	}

	private Oaf addProjects(final Oaf current, final Map<String, Oaf> projects) {

		final Oaf.Builder oafBuilder = Oaf.newBuilder(current);
		final List<Oaf> cachedRels = Lists.newArrayList();

		for (final Oaf.Builder cachedOafRel : oafBuilder.getEntityBuilder().getCachedOafRelBuilderList()) {
			final String projectId = cachedOafRel.getRel().getTarget();

			if (projects.containsKey(projectId)) {
				final Oaf project = projects.get(projectId);

				cachedOafRel.getRelBuilder().setCachedOafTarget(project);
				cachedRels.add(cachedOafRel.build());
			}
		}

		oafBuilder.getEntityBuilder().clearCachedOafRel().addAllCachedOafRel(cachedRels);

		return oafBuilder.build();
	}

}
