package eu.dnetlib.functionality.index;

import java.io.StringReader;
import java.util.List;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;

import javax.annotation.Resource;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.Element;
import org.dom4j.Node;
import org.dom4j.io.SAXReader;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Required;

import eu.dnetlib.enabling.actions.AbstractSubscriptionAction;
import eu.dnetlib.functionality.index.utils.MetadataReference;
import eu.dnetlib.functionality.index.utils.MetadataReferenceFactory;
import eu.dnetlib.functionality.index.utils.ServiceTools;
import eu.dnetlib.miscutils.dom4j.XPathHelper;

/**
 * MdFormatNotificationHandler updates index schema in response to metadata formats modifications.
 * 
 * @author claudio
 * 
 */
public class MdFormatNotificationHandler extends AbstractSubscriptionAction {

	/**
	 * Logger.
	 */
	private static final Log log = LogFactory.getLog(MdFormatNotificationHandler.class); // NOPMD by marko on 11/24/08 5:02 PM

	@Autowired
	private IndexServerDAOMap indexServerDAOMap;

	@Autowired
	private ServiceTools serviceTools;

	private Executor executor = Executors.newSingleThreadExecutor();

	private final transient ThreadLocal<SAXReader> domFactory = new ThreadLocal<SAXReader>() {

		@Override
		protected SAXReader initialValue() {
			return new SAXReader();
		}
	};

	/**
	 * {@inheritDoc}
	 * 
	 * @see eu.dnetlib.enabling.tools.Enableable
	 */
	private boolean enabled;

	@Override
	public void notified(final String subscrId, final String topic, final String rsId, final String profile) {
		if (!topic.startsWith(getTopicPrefix()) || !isEnabled()) return;
		executor.execute(new Runnable() {

			@Override
			public void run() {
				try {
					final Document doc = parse(profile);
					final Node formatNode = doc.selectSingleNode("//CONFIGURATION/NAME/text()");

					if ((formatNode != null) && !formatNode.asXML().isEmpty()) {

						final String format = formatNode.asXML();
						final Iterable<Element> layouts = XPathHelper.selectElements(doc, "//STATUS/LAYOUTS/LAYOUT");
						final Node interpretationNode = doc.selectSingleNode("//CONFIGURATION/INTERPRETATION/text()");
						final String interpretation = interpretationNode.asXML();

						log.info("found a change in mdFormat: " + format);
						for (Element element : layouts) {
							final String layout = element.attributeValue("name");

							log.info(String.format("processing format '%s', layout '%s', interpretation '%s'", format, layout, interpretation));

							// updates index schema based on mdFormat and layout
							final MetadataReference mdRef = MetadataReferenceFactory.getMetadata(format, layout, interpretation);

							log.info("got metadata reference " + mdRef.toString());

							final Document fields = parse(element.selectSingleNode("./FIELDS").asXML());

							log.debug("fields " + fields.asXML());

							List<String> backends = serviceTools.getBackendIds(mdRef);

							if ((backends == null) || backends.isEmpty()) {
								log.warn("There is no backendId in profiles for mdref " + mdRef);
							}

							log.info("found backends " + backends.toString());

							for (String backendId : backends) {

								log.info("processing backend " + backendId);

								IndexServerDAO idxDao = indexServerDAOMap.getIndexServerDAO(backendId);

								if (idxDao == null) {
									log.error("No index found for the mdformat " + mdRef);
									throw new RuntimeException("No index found for the mdformat " + mdRef);
								}

								log.info(String.format("Found index DAO which serve %s, backendId %s", mdRef.toString(), backendId));

								idxDao.updateIndexCollection(mdRef, fields);
							}
						}
					}
					log.info("Upload schema completed");
				} catch (Exception e) {
					throw new RuntimeException(e); // NOPMD
				}
			}
		});
	}

	@Override
	@Required
	public void setEnabled(final boolean enabled) {
		this.enabled = enabled;
	}

	/**
	 * Helper method, parses an xml string.
	 * 
	 * @param source
	 *            the xml.
	 * @return the parsed xml.
	 * @throws DocumentException
	 *             could happen
	 */
	private Document parse(final String source) throws DocumentException {
		return domFactory.get().read(new StringReader(source));
	}

	@Override
	public boolean isEnabled() {
		return enabled;
	}

}
