package eu.dnetlib.functionality.modular.ui.vocabularies.controllers;

import java.util.List;

import javax.annotation.Resource;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import eu.dnetlib.functionality.modular.ui.vocabularies.model.Term;
import eu.dnetlib.functionality.modular.ui.vocabularies.model.Vocabulary;
import eu.dnetlib.functionality.modular.ui.vocabularies.persistence.VocabularyDAO;
import eu.dnetlib.functionality.modular.ui.vocabularies.persistence.VocabularyException;

/**
 * Web controller for the UI
 *
 * @author Andrea Mannocci
 */
@Controller
public class VocabulariesController extends VocabularyDAO {

	private static final Log log = LogFactory.getLog(VocabulariesController.class);

	@Resource(name = "vocabularyDao")
	private VocabularyDAO dao;

	/**
	 * Returns all vocabularies serialized in json. Invoked by angularJS
	 */
	@Override
	@RequestMapping("/ui/vocabularies.json")
	public @ResponseBody List<Vocabulary> getVocabularies() throws VocabularyException {
		log.info("vocabularies.json");
		return dao.getVocabularies();
	}

	/**
	 * Returns all terms and their synonyms belonging to a vocabulary. Invoked by AngularJS
	 */
	@Override
	@RequestMapping("/ui/terms.json")
	public @ResponseBody List<Term> getTerms(@RequestParam(value = "vocabularyId", required = true) final String vocabularyId) throws VocabularyException {
		log.info("terms.json?vocabularyId=" + vocabularyId);
		return dao.getTerms(vocabularyId);
	}

	@Override
	@RequestMapping(value = "/ui/commitVocabulary", method = RequestMethod.POST)
	public @ResponseBody void commitTerms(@RequestBody(required = true) final List<Term> terms,
			@RequestParam(value = "vocabularyId", required = true) final String vocabularyId) throws VocabularyException {
		log.info("committing vocabulary id = " + vocabularyId);
		dao.commitTerms(terms, vocabularyId);
	}

	@Override
	@RequestMapping(value = "/ui/commitVocabularyInfo", method = RequestMethod.POST)
	public @ResponseBody void commitVocabularyInfo(@RequestBody(required = true) final Vocabulary voc,
			@RequestParam(value = "vocabularyId", required = true) final String vocabularyId) throws VocabularyException {
		log.info("committing vocabulary info id = " + vocabularyId);
		dao.commitVocabularyInfo(voc, vocabularyId);
	}

	@Override
	@RequestMapping(value = "/ui/createVocabulary", method = RequestMethod.POST)
	public @ResponseBody String createVocabulary(@RequestBody(required = true) final Vocabulary voc) throws VocabularyException {
		log.info("create vocabulary");
		String vocId = dao.createVocabulary(voc);
		return vocId;
	}

	@Override
	@RequestMapping(value = "/ui/dropVocabulary", method = RequestMethod.GET)
	public @ResponseBody void dropVocabulary(@RequestParam(value = "vocabularyId", required = true) final String vocabularyId) throws VocabularyException {
		log.info("delete vocabulary id=" + vocabularyId);
		dao.dropVocabulary(vocabularyId);
	}
}
