package eu.dnetlib.msro.workflows.nodes.repohi;

import org.springframework.beans.factory.annotation.Autowired;

import com.google.common.base.Splitter;
import com.googlecode.sarasvati.Arc;
import com.googlecode.sarasvati.NodeToken;

import eu.dnetlib.enabling.datasources.common.Api;
import eu.dnetlib.enabling.datasources.common.Datasource;
import eu.dnetlib.enabling.datasources.common.LocalDatasourceManager;
import eu.dnetlib.msro.rmi.MSROException;
import eu.dnetlib.msro.workflows.nodes.SimpleJobNode;
import eu.dnetlib.msro.workflows.util.WorkflowsConstants;

public class VerifyDatasourceJobNode extends SimpleJobNode {

	@Autowired
	private LocalDatasourceManager<Datasource<?, ?, ?>, Api<?>> dsManager;

	private String expectedDatasourceTypes;

	private String expectedCompatibilityLevels;

	private Boolean expectedConsentTermsOfUse;

	private Boolean expectedFullTextDownload;

	@Override
	protected String execute(final NodeToken token) throws Exception {
		final String dsId = token.getFullEnv().getAttribute(WorkflowsConstants.DATAPROVIDER_ID);
		final String ifaceId = token.getFullEnv().getAttribute(WorkflowsConstants.DATAPROVIDER_INTERFACE);

		final String type = dsManager.getDs(dsId).getEoscDatasourceType();

		verifyValue(type, expectedDatasourceTypes);

		final Api<?> api = dsManager.getApis(dsId)
			.stream()
			.filter(a -> a.getId().equals(ifaceId))
			.findFirst()
			.orElseThrow(() -> new MSROException("Intervace not found: " + ifaceId));;

		verifyValue(api.getCompatibility(), expectedCompatibilityLevels);

		if (expectedConsentTermsOfUse != null || expectedFullTextDownload != null) {
			final Datasource<?, ?, ?> ds = dsManager.getDs(dsId);
			verifyValue(ds.getConsentTermsOfUse(), expectedConsentTermsOfUse);
			verifyValue(ds.getFullTextDownload(), expectedFullTextDownload);
		}

		token.getFullEnv().setAttribute(WorkflowsConstants.DATAPROVIDER_INTERFACE_COMPLIANCE, api.getCompatibility());

		return Arc.DEFAULT_ARC;
	}

	private void verifyValue(final Boolean value, final Boolean expected) throws MSROException {
		if (expected != null && value != expected) { throw new MSROException("Invalid boolean value: " + value + ", expected value was " + expected); }
	}

	private void verifyValue(final String value, final String expected) throws Exception {
		if (value != null && expected != null && !expected.isEmpty()) {
			for (final String s : Splitter.on(",").omitEmptyStrings().trimResults().split(expected)) {
				if (value.equalsIgnoreCase(s)) { return; }
			}
			throw new MSROException("Invalid value: " + value + ", Valid term prefixes are: [" + expected + "]");
		}
	}

	public String getExpectedDatasourceTypes() {
		return expectedDatasourceTypes;
	}

	public void setExpectedDatasourceTypes(final String expectedDatasourceTypes) {
		this.expectedDatasourceTypes = expectedDatasourceTypes;
	}

	public Boolean getExpectedConsentTermsOfUse() {
		return expectedConsentTermsOfUse;
	}

	public void setExpectedConsentTermsOfUse(final Boolean expectedConsentTermsOfUse) {
		this.expectedConsentTermsOfUse = expectedConsentTermsOfUse;
	}

	public Boolean getExpectedFullTextDownload() {
		return expectedFullTextDownload;
	}

	public void setExpectedFullTextDownload(final Boolean expectedFullTextDownload) {
		this.expectedFullTextDownload = expectedFullTextDownload;
	}

	public String getExpectedCompatibilityLevels() {
		return expectedCompatibilityLevels;
	}

	public void setExpectedCompatibilityLevels(final String expectedCompatibilityLevels) {
		this.expectedCompatibilityLevels = expectedCompatibilityLevels;
	}

}
