package eu.dnetlib.msro.workflows.nodes.transform;

import java.io.IOException;
import java.net.URLEncoder;

import javax.xml.ws.wsaddressing.W3CEndpointReference;

import org.antlr.stringtemplate.StringTemplate;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.core.io.Resource;

import com.googlecode.sarasvati.Arc;
import com.googlecode.sarasvati.NodeToken;

import eu.dnetlib.enabling.datasources.common.Api;
import eu.dnetlib.enabling.datasources.common.Datasource;
import eu.dnetlib.enabling.datasources.common.LocalDatasourceManager;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpException;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpService;
import eu.dnetlib.enabling.locators.UniqueServiceLocator;
import eu.dnetlib.enabling.resultset.XSLTMappedResultSetFactory;
import eu.dnetlib.enabling.resultset.client.utils.EPRUtils;
import eu.dnetlib.msro.rmi.MSROException;
import eu.dnetlib.msro.workflows.nodes.SimpleJobNode;

public class MdBuilderJobNode extends SimpleJobNode {

	private static final Log log = LogFactory.getLog(MdBuilderJobNode.class);

	private Resource mdBuilderTemplateXslt;

	private String inputEprParam;
	private String outputEprParam;
	private String datasourceId;
	private String datasourceInterface;

	private XSLTMappedResultSetFactory xsltMappedResultSetFactory;

	@Autowired
	private LocalDatasourceManager<Datasource<?, ?, ?>, Api<?>> dsManager;

	@Autowired
	private UniqueServiceLocator serviceLocator;

	@Override
	protected String execute(final NodeToken token) throws MSROException {
		final String inputEpr = token.getEnv().getAttribute(inputEprParam);

		if (inputEpr == null || inputEpr.isEmpty()) { throw new MSROException("InputEprParam (" + inputEprParam + ") not found in ENV"); }
		StringTemplate st = null;
		try {
			final Datasource<?, ?, ?> ds = dsManager.getDs(datasourceId);
			final Api<?> api = dsManager.getApis(datasourceId)
				.stream()
				.filter(a -> a.getId().equals(datasourceInterface))
				.findFirst()
				.orElseThrow(() -> new Exception("Api not found"));

			st = new StringTemplate(IOUtils.toString(getMdBuilderTemplateXslt().getInputStream()));
			st.setAttribute("datasourceId", datasourceId);
			st.setAttribute("xpath", api.getMetadataIdentifierPath().replace("\"", "'"));
			st.setAttribute("baseurl", URLEncoder.encode(api.getBaseurl(), "UTF-8"));
			st.setAttribute("metadatanamespace", getMetadataNamespace(api.getApiParams()
				.stream()
				.filter(a -> a.getParam().equalsIgnoreCase("format"))
				.map(a -> a.getValue())
				.findFirst()
				.orElse(null)));

			if (token.getEnv().hasAttribute("namespacePrefix")) {
				st.setAttribute("namespacePrefix", token.getEnv().getAttribute("namespacePrefix"));
			} else if (StringUtils.isNotBlank(ds.getNamespaceprefix())) {
				st.setAttribute("namespacePrefix", ds.getNamespaceprefix());
			} else {
				st.setAttribute("namespacePrefix", datasourceId);
			}

			final W3CEndpointReference epr = xsltMappedResultSetFactory.createMappedResultSet(new EPRUtils().getEpr(inputEpr), st.toString());

			token.getEnv().setAttribute(outputEprParam, epr.toString());

			return Arc.DEFAULT_ARC;
		} catch (final ISLookUpException e) {
			throw new MSROException("Error while initializing mdBuilder template (" + getMdBuilderTemplateXslt().getFilename() + ") for datasource "
				+ datasourceId, e);
		} catch (final IOException e) {
			throw new MSROException("Error parsing template: " + getMdBuilderTemplateXslt().getFilename(), e);
		} catch (final Exception e) {
			log.error(st.toString());
			throw new MSROException(e);
		}
	}

	private String getMetadataNamespace(final String mdFormat) {
		if (StringUtils.isBlank(mdFormat)) { return ""; }
		try {
			String xQuery = "/*[.//RESOURCE_TYPE/@value='MetadataFormatDSResourceType']//METADATAFORMAT[@Prefix='{format}']/@NameSpace/string()";
			xQuery = xQuery.replace("{format}", mdFormat);
			return serviceLocator.getService(ISLookUpService.class).getResourceProfileByQuery(xQuery);
		} catch (final ISLookUpException e) {
			log.error("The interface is not OAI or the format is not found in the MetadataFormatDSResourceType, thus metadata format in the <about> section "
				+ "cannot managed here and it will be leaved empty (for the time being)");
			return "";
		}
	}

	public String getInputEprParam() {
		return inputEprParam;
	}

	public void setInputEprParam(final String inputEprParam) {
		this.inputEprParam = inputEprParam;
	}

	public String getOutputEprParam() {
		return outputEprParam;
	}

	public void setOutputEprParam(final String outputEprParam) {
		this.outputEprParam = outputEprParam;
	}

	public XSLTMappedResultSetFactory getXsltMappedResultSetFactory() {
		return xsltMappedResultSetFactory;
	}

	@Required
	public void setXsltMappedResultSetFactory(final XSLTMappedResultSetFactory xsltMappedResultSetFactory) {
		this.xsltMappedResultSetFactory = xsltMappedResultSetFactory;
	}

	public String getDatasourceId() {
		return datasourceId;
	}

	public void setDatasourceId(final String datasourceId) {
		this.datasourceId = datasourceId;
	}

	public String getDatasourceInterface() {
		return datasourceInterface;
	}

	public void setDatasourceInterface(final String datasourceInterface) {
		this.datasourceInterface = datasourceInterface;
	}

	public Resource getMdBuilderTemplateXslt() {
		return mdBuilderTemplateXslt;
	}

	@Required
	public void setMdBuilderTemplateXslt(final Resource mdBuilderTemplateXslt) {
		this.mdBuilderTemplateXslt = mdBuilderTemplateXslt;
	}

}
