package eu.dnetlib.enabling.datasources;

import java.io.IOException;
import java.sql.Array;
import java.sql.Date;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.*;
import java.util.stream.Collectors;

import eu.dnetlib.enabling.datasources.common.*;
import eu.dnetlib.miscutils.datetime.DateUtils;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.Element;
import org.dom4j.io.SAXReader;
import org.springframework.core.io.ClassPathResource;
import org.springframework.core.io.Resource;

public class DatasourceFunctions {

    private static final Log log = LogFactory.getLog(DatasourceFunctions.class);

    private static final Resource baseDsProfile = new ClassPathResource("/eu/dnetlib/enabling/datasources/templates/datasource_base.xml");


    public static SimpleDatasource mapToSimpleDs(final Map<String, Object> map) {
        final SimpleDatasource ds = new SimpleDatasource();
        ds.setId(castObject(map.get("id"), String.class));
        ds.setName(castObject(map.get("name"), String.class));
        ds.setOrigId(castObject(map.get("id"), String.class));
        ds.setTypology(castObject(map.get("typology"), String.class));
        ds.setValid(true);
        try {
            final Array arr = castObject(map.get("apis"), Array.class);

            if (arr != null && ((Object[]) arr.getArray()).length > 0) {
                ds.setApis(Arrays.stream((Object[]) arr.getArray())
                        .filter(Objects::nonNull)
                        .map(Object::toString)
                        .collect(Collectors.toSet()));
            }
        } catch (final SQLException e) {
            log.error("Error parsing array (apis)", e);
            throw new RuntimeException("Error parsing array (apis)", e);
        }

        return ds;
    }


    public static Datasource<Organization<?>, Identity> mapToDatasource(final Map<String, Object> map) {
        final Datasource<Organization<?>, Identity> ds = new Datasource<>();
        ds.setId(castObject(map.get("id"), String.class));
        ds.setOfficialname(castObject(map.get("officialname"), String.class));
        ds.setEnglishname(castObject(map.get("englishname"), String.class));
        ds.setWebsiteurl(castObject(map.get("websiteurl"), String.class));
        ds.setLogourl(castObject(map.get("logourl"), String.class));
        ds.setContactemail(castObject(map.get("contactemail"), String.class));
        ds.setLatitude(castObject(map.get("latitude"), Double.class));
        ds.setLongitude(castObject(map.get("longitude"), Double.class));
        ds.setTimezone(castObject(map.get("timezone"), String.class));
        ds.setNamespaceprefix(castObject(map.get("namespaceprefix"), String.class));
        ds.setLanguages(castObject(map.get("languages"), String.class));
        ds.setOd_contenttypes(castObject(map.get("od_contenttypes"), String.class));
        ds.setCollectedfrom(castObject(map.get("collectedfrom"), String.class));
        ds.setDateofvalidation(castObject(map.get("dateofvalidation"), Date.class));
        ds.setTypology(castObject(map.get("typology"), String.class));
        ds.setProvenanceaction(castObject(map.get("provenanceaction"), String.class));
        ds.setDateofcollection(castObject(map.get("dateofcollection"), Date.class));
        ds.setPlatform(castObject(map.get("platform"), String.class));
        ds.setActivationId(castObject(map.get("activationId"), String.class));
        ds.setDescription(castObject(map.get("description"), String.class));
        ds.setReleasestartdate(castObject(map.get("releasestartdate"), Date.class));
        ds.setReleaseenddate(castObject(map.get("releaseenddate"), Date.class));
        ds.setMissionstatementurl(castObject(map.get("missionstatementurl"), String.class));
        ds.setDataprovider(castObject(map.get("dataprovider"), Boolean.class));
        ds.setServiceprovider(castObject(map.get("serviceprovider"), Boolean.class));
        ds.setDatabaseaccesstype(castObject(map.get("databaseaccesstype"), String.class));
        ds.setDatauploadtype(castObject(map.get("datauploadtype"), String.class));
        ds.setDatabaseaccessrestriction(castObject(map.get("databaseaccessrestriction"), String.class));
        ds.setDatauploadrestriction(castObject(map.get("datauploadrestriction"), String.class));
        ds.setVersioning(castObject(map.get("versioning"), Boolean.class));
        ds.setCitationguidelineurl(castObject(map.get("citationguidelineurl"), String.class));
        ds.setQualitymanagementkind(castObject(map.get("qualitymanagementkind"), String.class));
        ds.setPidsystems(castObject(map.get("pidsystems"), String.class));
        ds.setCertificates(castObject(map.get("certificates"), String.class));
        ds.setAggregator(castObject(map.get("aggregator"), String.class));
        ds.setIssn(castObject(map.get("issn"), String.class));
        ds.setEissn(castObject(map.get("eissn"), String.class));
        ds.setLissn(castObject(map.get("lissn"), String.class));
        ds.setRegisteredby(castObject(map.get("registeredby"), String.class));
        ds.setSubjects(castObject(map.get("subjects"), String.class));
        ds.setManaged(castObject(map.get("managed"), Boolean.class));
        return ds;
    }

    public static Identity mapToDsIdentity(final Map<String, Object> map) {
        final Identity id = new Identity();
        id.setPid(castObject(map.get("pid"), String.class));
        id.setIssuertype(castObject(map.get("issuertype"), String.class));
        return id;
    }

    public static Organization<Datasource<?, ?>> mapToDsOrganization(final Map<String, Object> map) {
        final Organization<Datasource<?, ?>> org = new Organization<>();
        org.setId(castObject(map.get("id"), String.class));
        org.setLegalshortname(castObject(map.get("legalshortname"), String.class));
        org.setLegalname(castObject(map.get("legalname"), String.class));
        org.setWebsiteurl(castObject(map.get("websiteurl"), String.class));
        org.setLogourl(castObject(map.get("logourl"), String.class));
        org.setCountry(castObject(map.get("country"), String.class));
        org.setCollectedfrom(castObject(map.get("collectedfrom"), String.class));
        org.setDateofcollection(castObject(map.get("dateofcollection"), Date.class));
        org.setProvenanceaction(castObject(map.get("provenanceaction"), String.class));
        return org;
    }

    public static SearchApisEntry mapToSearchApisEntry(final Map<String, Object> map) {
        final SearchApisEntry a = new SearchApisEntry();
        a.setId(castObject(map.get("id"), String.class));
        a.setCompliance(castObject(map.get("compliance"), String.class));
        a.setActive(castObject(map.get("active"), Boolean.class));
        a.setRepoId(castObject(map.get("dsId"), String.class));
        a.setRepoName(castObject(map.get("name"), String.class));
        a.setRepoCountry(castObject(map.get("country"), String.class));
        a.setRepoPrefix(castObject(map.get("prefix"), String.class));
        a.setAggrDate(castObject(map.get("aggrDate"), String.class));
        a.setAggrTotal(castObject(map.get("aggrTotal"), Integer.class));
        a.setProtocol(castObject(map.get("protocol"), String.class));
        a.setAlternativeName(castObject(map.get("alternativeName"), String.class));
        a.setRepoOrganization(castObject(map.get("organization"), String.class));
        return a;
    }

    public static Api<ApiParam> mapToApi(final Map<String, Object> map) {
        final Api<ApiParam> a = new Api<>();
        a.setId(castObject(map.get("id"), String.class));
        a.setProtocol(castObject(map.get("protocol"), String.class));
        a.setDatasource(castObject(map.get("datasource"), String.class));
        a.setContentdescription(castObject(map.get("contentdescription"), String.class));
        a.setActive(castObject(map.get("active"), Boolean.class));
        a.setRemovable(castObject(map.get("removable"), Boolean.class));
        a.setTypology(castObject(map.get("typology"), String.class));
        a.setCompatibility(castObject(map.get("compatibility"), String.class));
        a.setCompatibilityOverrided(castObject(map.get("isCompatibilityOverrided"), Boolean.class));
        a.setMetadataIdentifierPath(castObject(map.get("metadataIdentifierPath"), String.class));
        a.setLastCollectionTotal(castObject(map.get("lastCollectionTotal"), Integer.class));
        a.setLastCollectionDate(castObject(map.get("lastCollectionDate"), Timestamp.class));
        a.setLastCollectionMdid(castObject(map.get("lastCollectionMdid"), String.class));
        a.setLastAggregationTotal(castObject(map.get("lastAggregationTotal"), Integer.class));
        a.setLastAggregationDate(castObject(map.get("lastAggregationDate"), Timestamp.class));
        a.setLastAggregationMdid(castObject(map.get("lastAggregationMdid"), String.class));
        a.setLastDownloadTotal(castObject(map.get("lastDownloadTotal"), Integer.class));
        a.setLastDownloadDate(castObject(map.get("lastDownloadDate"), Timestamp.class));
        a.setLastDownloadObjid(castObject(map.get("lastDownloadObjid"), String.class));
        a.setLastValidationJob(castObject(map.get("lastValidationJob"), String.class));
        a.setBaseurl(castObject(map.get("baseUrl"), String.class));
        try {
            final Array arr = castObject(map.get("params"), Array.class);
            if (arr != null) {
                a.setApiParams(Arrays.stream((Object[]) arr.getArray())
                        .filter(Objects::nonNull)
                        .map(Object::toString)
                        .map(s -> {
                            final ApiParam p = new ApiParamImpl();
                            p.setParam(StringUtils.substringBefore(s, "="));
                            p.setValue(StringUtils.substringAfter(s, "="));
                            return p;
                        })
                        .collect(Collectors.toSet()));
            }
        } catch (final SQLException e) {
            log.error("Error parsing array params", e);
            throw new RuntimeException("Error parsing array params", e);
        }

        return a;
    }

    public static String dsToProfile(final Datasource<Organization<?>, Identity> ds, final List<Api<ApiParam>> apis, final String profId)
            throws DocumentException, IOException {
        final Document doc = new SAXReader().read(baseDsProfile.getInputStream());

        setValue(doc, "//DATASOURCE_TYPE", ds.getTypology());
        setValue(doc, "//DATASOURCE_ORIGINAL_ID", ds.getId());
        setValue(doc, "//TYPOLOGY", ds.getPlatform());
        setValue(doc, "//OFFICIAL_NAME", ds.getOfficialname());
        setValue(doc, "//ENGLISH_NAME", ds.getEnglishname());
        setValue(doc, "//ICON_URI", ds.getLogourl());
        setValue(doc, "//COUNTRY", ds.getOrganizations().stream().map(Organization::getCountry).findFirst().orElse(""));
        setValue(doc, "//LONGITUDE", ds.getLongitude());
        setValue(doc, "//LATITUDE", ds.getLatitude());
        setValue(doc, "//TIMEZONE", ds.getTimezone());
        setValue(doc, "//REPOSITORY_WEBPAGE", ds.getWebsiteurl());
        setValue(doc, "//REPOSITORY_INSTITUTION", ds.getOrganizations().stream().map(Organization::getLegalname).findFirst().orElse(""));
        setValue(doc, "//ADMIN_INFO", ds.getContactemail());
        setValue(doc, "//REGISTERED_BY", ds.getRegisteredby());
        setValue(doc, "//LAST_UPDATE/@value", DateUtils.now_ISO8601());

        final Element ef = (Element) doc.selectSingleNode("//EXTRA_FIELDS");
        addExtraField(ef, "OpenAireDataSourceId", ds.getId());
        addExtraField(ef, "NamespacePrefix", ds.getNamespaceprefix());
        addExtraField(ef, "VERIFIED", "NO");
        addExtraField(ef, "aggregatorName", ds.getAggregator());
        addExtraField(ef, "dateOfValidation", ds.getDateofvalidation());
        addExtraField(ef, "dateOfCollection", ds.getDateofcollection());
        addExtraField(ef, "mergeHomonyms", "NO");
        addExtraField(ef, "ACTID", ds.getActivationId());

        if (apis != null) {
            final Element ifaces = (Element) doc.selectSingleNode("//INTERFACES");
            apis.forEach(api -> addInterface(ifaces, api));
        }
        return doc.asXML();
    }

    private static void setValue(final Document doc, final String xpath, final Object value) {
        if (value != null) {
            doc.selectSingleNode(xpath).setText(value.toString());
        }
    }

    private static void addInterface(final Element ifaces, final Api<?> api) {

        final Element ifc = ifaces.addElement("INTERFACE");
        ifc.addAttribute("id", api.getId());
        ifc.addAttribute("label", String.format("%s (%s)",
                StringUtils.defaultIfBlank(api.getTypology(), "-"),
                StringUtils.defaultIfBlank(api.getCompatibility(), "-")));
        ifc.addAttribute("typology", StringUtils.defaultIfBlank(api.getTypology(), ""));
        ifc.addAttribute("active", "" + BooleanUtils.toBooleanDefaultIfNull(api.getActive(), false));
        ifc.addAttribute("compliance", StringUtils.defaultIfBlank(api.getCompatibility(), ""));
        ifc.addAttribute("contentDescription", StringUtils.defaultIfBlank(api.getContentdescription(), ""));
        ifc.addAttribute("removable", "" + BooleanUtils.toBooleanDefaultIfNull(api.getRemovable(), false));

        final Element ap = ifc.addElement("ACCESS_PROTOCOL");
        api.getApiParams().forEach(p -> ap.addAttribute(p.getParam(), p.getValue()));
        ap.setText(StringUtils.defaultIfBlank(api.getProtocol(), ""));
        ifc.addElement("BASE_URL").setText(StringUtils.defaultIfBlank(api.getBaseurl(), ""));

        if (api.isCompatibilityOverrided()) {
            addInterfaceExtraField(ifc, "overriding_compliance", true);
        }

        addInterfaceExtraField(ifc, "metadata_identifier_path", api.getMetadataIdentifierPath());
        addInterfaceExtraField(ifc, "last_collection_date", api.getLastCollectionDate());
        addInterfaceExtraField(ifc, "last_collection_mdId", api.getLastCollectionMdid());
        addInterfaceExtraField(ifc, "last_collection_total", api.getLastCollectionTotal());
        addInterfaceExtraField(ifc, "last_aggregation_date", api.getLastAggregationDate());
        addInterfaceExtraField(ifc, "last_aggregation_mdId", api.getLastAggregationMdid());
        addInterfaceExtraField(ifc, "last_aggregation_total", api.getLastAggregationTotal());
        addInterfaceExtraField(ifc, "last_download_date", api.getLastDownloadDate());
        addInterfaceExtraField(ifc, "last_download_objId", api.getLastDownloadObjid());
        addInterfaceExtraField(ifc, "last_download_total", api.getLastDownloadTotal());
    }

    private static void addExtraField(final Element parent, final String field, final Object value) {
        if (value != null && StringUtils.isNotBlank(value.toString())) {
            final Element f = parent.addElement("FIELD");
            f.addElement("key").setText(field);
            f.addElement("value").setText(value.toString());
        }
    }

    private static void addInterfaceExtraField(final Element parent, final String field, final Object value) {
        if (value != null && StringUtils.isNotBlank(value.toString())) {
            final Element f = parent.addElement("INTERFACE_EXTRA_FIELD");
            f.addAttribute("name", field);
            f.setText(value.toString());
        }
    }

    public static Map<String, Object> dsToMap(final Datasource<Organization<?>, Identity> ds) {
        final Map<String, Object> map = new HashMap<>();

        map.put("id", ds.getId());
        map.put("officialname", ds.getOfficialname());
        map.put("englishname", ds.getEnglishname());
        map.put("websiteurl", ds.getWebsiteurl());
        map.put("logourl", ds.getLogourl());
        map.put("contactemail", ds.getContactemail());
        map.put("latitude", ds.getLatitude());
        map.put("longitude", ds.getLongitude());
        map.put("timezone", ds.getTimezone());
        map.put("namespaceprefix", ds.getNamespaceprefix());
        map.put("languages", ds.getLanguages());
        map.put("od_contenttypes", ds.getOd_contenttypes());
        map.put("collectedfrom", ds.getCollectedfrom());
        map.put("dateofvalidation", ds.getDateofvalidation());
        map.put("typology", ds.getTypology());
        map.put("provenanceaction", ds.getProvenanceaction());
        map.put("platform", ds.getPlatform());
        map.put("activationid", ds.getActivationId());
        map.put("description", ds.getDescription());
        map.put("releasestartdate", ds.getReleasestartdate());
        map.put("releaseenddate", ds.getReleaseenddate());
        map.put("missionstatementurl", ds.getMissionstatementurl());
        map.put("dataprovider", ds.getDataprovider());
        map.put("serviceprovider", ds.getServiceprovider());
        map.put("databaseaccesstype", ds.getDatabaseaccesstype());
        map.put("datauploadtype", ds.getDatauploadtype());
        map.put("databaseaccessrestriction", ds.getDatabaseaccessrestriction());
        map.put("datauploadrestriction", ds.getDatauploadrestriction());
        map.put("versioning", ds.getVersioning());
        map.put("citationguidelineurl", ds.getCitationguidelineurl());
        map.put("qualitymanagementkind", ds.getQualitymanagementkind());
        map.put("pidsystems", ds.getPidsystems());
        map.put("certificates", ds.getCertificates());
        map.put("aggregator", ds.getAggregator());
        map.put("issn", ds.getIssn());
        map.put("eissn", ds.getEissn());
        map.put("lissn", ds.getLissn());
        map.put("registeredby", ds.getRegisteredby());
        map.put("subjects", ds.getSubjects());
        map.put("managed", ds.getManaged());

        return map;
    }

    public static Map<String, Object> orgToMap(final String dsId, final Organization<?> org) {
        final Map<String, Object> map = new HashMap<>();
        map.put("dsId", dsId);
        map.put("orgId", org.getId());
        map.put("legalname", org.getLegalname());
        map.put("legalshortname", org.getLegalshortname());
        map.put("websiteurl", org.getWebsiteurl());
        map.put("logourl", org.getLogourl());
        map.put("country", org.getCountry());
        return map;
    }

    public static Map<String, Object> apiToMap(final Api<?> api) {
        final Map<String, Object> map = new HashMap<>();
        map.put("apiId", api.getId());
        map.put("protocol", api.getProtocol());
        map.put("baseUrl", api.getBaseurl());
        map.put("dsId", api.getDatasource());
        map.put("contentDescription", api.getContentdescription());
        map.put("typology", api.getTypology());
        map.put("compatibility", api.getCompatibility());
        map.put("metadataIdentifierPath", api.getMetadataIdentifierPath());
        // The other fields are not required in the INSERT operation
        return map;
    }

    @SuppressWarnings("unchecked")
    private static <T> T castObject(final Object o, final Class<T> clazz) {
        if (o == null) {
            return null;
        }
        if (clazz.isInstance(o)) {
            return (T) o;
        }
        throw new IllegalArgumentException("Type not managed: " + clazz.getSimpleName());
    }

}
