package eu.dnetlib.openaire.exporter.datasource;

import java.util.List;

import eu.dnetlib.common.rmi.DNetRestDocumentation;

import eu.dnetlib.openaire.exporter.AbstractExporterController;
import eu.dnetlib.openaire.exporter.datasource.clients.ClientResponse;
import eu.dnetlib.openaire.exporter.datasource.clients.DatasourceDao;
import eu.dnetlib.openaire.exporter.model.datasource.BrowseTerm;
import eu.dnetlib.openaire.exporter.model.datasource.DatasourceResponse;
import eu.dnetlib.openaire.exporter.model.datasource.db.Api;
import eu.dnetlib.openaire.exporter.model.datasource.db.Datasource;
import eu.dnetlib.openaire.exporter.model.datasource.db.SearchInterfacesEntry;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.http.HttpStatus;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.web.bind.annotation.*;

@RestController
@DNetRestDocumentation
public class DatasourcesApiController extends AbstractExporterController implements DatasourcesApi {

	private static final Log log = LogFactory.getLog(DatasourcesApiController.class);

	@Autowired
	private DatasourceDao dsDao;

	@Override
	@RequestMapping(value = "/ds/list/{page}/{size}", produces = { "application/json" }, method = RequestMethod.GET)
    public List<String> listIds(@PathVariable int page, @PathVariable int size) throws ApiException {
	    return dsDao.listIds(new PageRequest(page, size));
    }

    @Override
    @RequestMapping(value = "/ds/get/{id}", produces = { "application/json" }, method = RequestMethod.GET)
    public DatasourceResponse getDs(@PathVariable final String id) {

	    if (log.isDebugEnabled()) {
		    log.debug(String.format("getDatasourceInfo(dsId = %s)", id));
	    }

	    final ClientResponse clientResponse = dsDao.getInfo(id);
    	return clientResponse.getDatasourceResponse();
    }

	@Override
	@RequestMapping(value = "/ds/search/name/{page}/{size}", produces = { "application/json" }, method = RequestMethod.GET)
	public List<DatasourceResponse> searchByName(
			@RequestParam final String name, @PathVariable final int page, @PathVariable final int size) {
		return dsDao.searchByName(name, new PageRequest(page, size));
	}

	@Override
	@RequestMapping(value = "/ds/search/email/{page}/{size}", produces = { "application/json" }, method = RequestMethod.GET)
	public List<DatasourceResponse> searchByContactemail(
			@RequestParam final String contactemail, @PathVariable final int page, @PathVariable final int size) {
		return dsDao.searchByContactemail(contactemail, new PageRequest(page, size));
	}

	@Override
	@RequestMapping(value = "/ds/search/country/{page}/{size}", produces = { "application/json" }, method = RequestMethod.GET)
	public List<DatasourceResponse> searchByCountry(
			@RequestParam final String country, @PathVariable final int page, @PathVariable final int size) {
		return dsDao.searchByCountry(country, new PageRequest(page, size));
	}

	@Override
	@RequestMapping(value = "/api/search/{field}", produces = { "application/json" }, method = RequestMethod.GET)
	public List<SearchInterfacesEntry> searchInterface(@PathVariable final String field, @RequestParam final String value) {
		return dsDao.searchInterface(field, value);
	}

	@Override
	@RequestMapping(value = "/api/browse/{field}", produces = { "application/json" }, method = RequestMethod.GET)
	public List<? extends BrowseTerm> browseField(@PathVariable final String field) throws ApiException {
		switch (field) {
		case "country":
			return dsDao.browseCountries();
		case "typology":
			return dsDao.browseTypologies();
		case "protocol":
			return dsDao.browseProtocols();
		case "compatibility":
			return dsDao.browseCompatibility();
		case "activation":
			return dsDao.browseActivation();
		default:
			throw new ApiException(HttpStatus.SC_BAD_REQUEST, String.format("unsupported browse field '%s'", field));
		}
	}

	@Override
	@RequestMapping(value = "/ds/api/{dsId}", produces = { "application/json" }, method = RequestMethod.GET)
	public List<Api> getApi(@PathVariable final String dsId) throws ApiException {
		return dsDao.getApi(dsId);
	}

	@Override
	@RequestMapping(value = "/ds/api/{dsId}", method = RequestMethod.DELETE)
	public void deleteApi(@PathVariable final String apiId) throws ApiException {
		dsDao.deleteApi(apiId);
	}

	@Override
	@RequestMapping(value = "/ds/api/add", method = RequestMethod.POST)
	public void addApi(@RequestParam final Api api) throws ApiException {
		if (StringUtils.isBlank(api.getDatasource())) {
			throw new ApiException(HttpStatus.SC_BAD_REQUEST, "missing datasource id");
		}
		dsDao.addApi(api);
	}

	@Override
	@RequestMapping(value = "/ds/manage", method = RequestMethod.POST)
	public void setManaged(@RequestParam final String id, @RequestParam final boolean managed) {
		dsDao.setManaged(id, managed);
	}

	@Override
	@RequestMapping(value = "/ds/add", method = RequestMethod.POST)
	public void saveDatasource(@RequestBody final Datasource datasource) throws ApiException {
		if (dsDao.exist(datasource)) { //TODO further check that the DS doesn't have any API
			throw new ApiException(HttpStatus.SC_CONFLICT, String.format("cannot already defined '%s'", datasource.getId()));
		}
		dsDao.save(datasource);
	}

	@Override
	@RequestMapping(value = "/ds/update/officialname", method = RequestMethod.POST)
	public void updateOfficialname(@RequestParam final String dsId, @RequestParam final String officialname) throws ApiException {
		dsDao.updateOfficialName(dsId, officialname);
	}

	@Override
	@RequestMapping(value = "/ds/update/englishname", method = RequestMethod.POST)
	public void updateEnglishname(@RequestParam final String dsId, @RequestParam final String englishname) throws ApiException {
		dsDao.updateEnglishName(dsId, englishname);
	}

	@Override
	@RequestMapping(value = "/ds/update/latitude", method = RequestMethod.POST)
	public void updateLatitude(final String dsId, final Double latitude) throws ApiException {
		dsDao.updateLatitude(dsId, latitude);
	}

	@Override
	@RequestMapping(value = "/ds/update/longitude", method = RequestMethod.POST)
	public void updateLongitude(final String dsId, final Double longitude) throws ApiException {
		dsDao.updateLongitude(dsId, longitude);
	}

}
