package eu.dnetlib.openaire.exporter.model;

import java.sql.Date;

import com.fasterxml.jackson.databind.ObjectMapper;
import eu.dnetlib.data.transform.xml.AbstractDNetXsltFunctions;
import eu.dnetlib.miscutils.datetime.DateUtils;
import eu.dnetlib.openaire.exporter.model.datasource.ApiDetails;
import eu.dnetlib.openaire.exporter.model.datasource.DatasourceDetails;
import eu.dnetlib.openaire.exporter.model.datasource.OrganizationDetails;
import eu.dnetlib.openaire.exporter.model.datasource.db.ApiDbEntry;
import eu.dnetlib.openaire.exporter.model.datasource.db.DatasourceDbEntry;
import org.apache.commons.lang3.RandomStringUtils;
import org.apache.commons.lang3.StringUtils;
import org.dom4j.DocumentHelper;
import org.dom4j.Element;

public class ConversionUtils {

	public static final String ID_SEPARATOR = "::";

	public static final String ID_PREFIX = "api_________" + ID_SEPARATOR;

	public static String createId(final ApiDetails api) {
		return ID_PREFIX + api.getDatasource() + ID_SEPARATOR + RandomStringUtils.randomAlphanumeric(8);
	}

	public static DatasourceDetails asDetails(final DatasourceDbEntry d) {
		final DatasourceDetails details = _convert(d, DatasourceDetails.class);
		return details.setOpenaireId(asOpenaireId(details.getId()));
	}

	public static ApiDetails asDetails(final ApiDbEntry d) {
		return _convert(d, ApiDetails.class);
	}

	public static ApiDbEntry asDbEntry(final ApiDetails d) {
		final ApiDbEntry apiDbEntry = _convert(d, ApiDbEntry.class);

		// Need to complete the references among objects, because you know, referencial integrity ...
		apiDbEntry.getApiParams().forEach(ap -> ap.getId().setApi(apiDbEntry));

		return apiDbEntry;
	}

	public static DatasourceDbEntry asDbEntry(DatasourceDetails d) {
		final DatasourceDbEntry dbe = _convert(d, DatasourceDbEntry.class);
		if (dbe.getOrganizations() != null) {
			dbe.getOrganizations().forEach(o -> {
				o.setId(dbe.getId() + ID_SEPARATOR + o.getLegalname());
				if (o.getDateofcollection() == null) {
					o.setDateofcollection(new Date(System.currentTimeMillis()));
				}
			});
		}
		return dbe;
	}

	public static String asRepositoryProfile(final DatasourceDetails ds) {
		final Element root = DocumentHelper.createElement("RESOURCE_PROFILE");

		final Element header = root.addElement("HEADER");
		header.addElement("RESOURCE_IDENTIFIER").addAttribute("value", "");
		header.addElement("RESOURCE_TYPE").addAttribute("value", "RepositoryServiceResourceType");
		header.addElement("RESOURCE_KIND").addAttribute("value", "RepositoryServiceResources");
		header.addElement("RESOURCE_URI").addAttribute("value", "");
		header.addElement("DATE_OF_CREATION").addAttribute("value", DateUtils.now_ISO8601());
		header.addElement("PROTOCOL");

		final Element body = root.addElement("BODY");
		final Element conf = body.addElement("CONFIGURATION");
		conf.addElement("DATASOURCE_TYPE").setText(ds.getTypology());

		final Element origId = conf.addElement("DATASOURCE_ORIGINAL_ID");
		origId.addAttribute("provenance", "D-NET");
		origId.setText(ds.getId());

		conf.addElement("DATASOURCE_AGGREGATED").setText("false");
		conf.addElement("ENVIRONMENTS").addElement("ENVIRONMENT").setText("OPENAIRE");
		conf.addElement("TYPOLOGY").setText(""+ds.getTypology());
		conf.addElement("MAX_SIZE_OF_DATASTRUCTURE").setText("0");
		conf.addElement("AVAILABLE_DISKSPACE").setText("0");
		conf.addElement("MAX_NUMBER_OF_DATASTRUCTURE").setText("0");

		final String officialName = ds.getOfficialname();
		conf.addElement("OFFICIAL_NAME").setText(officialName);
		String englishName = ds.getEnglishname();
		conf.addElement("ENGLISH_NAME").setText(StringUtils.isNotBlank(englishName) ? englishName : officialName);
		conf.addElement("ICON_URI").setText("" + ds.getLogourl());
		final OrganizationDetails org = getOrganization(ds);

		conf.addElement("COUNTRY").setText(org != null ? org.getCountry() : "");

		final Element location = conf.addElement("LOCATION");
		location.addElement("LONGITUDE").setText("" + ds.getLongitude());
		location.addElement("LATITUDE").setText("" + ds.getLatitude());
		location.addElement("TIMEZONE").setText("" + ds.getTimezone());

		conf.addElement("REPOSITORY_WEBPAGE").setText(ds.getWebsiteurl());
		getOrganization(ds);
		conf.addElement("REPOSITORY_INSTITUTION").setText(org != null ? org.getLegalname() : "");

		conf.addElement("ADMIN_INFO").setText(ds.getContactemail());

		conf.addElement("INTERFACES");

		final Element extraFields = conf.addElement("EXTRA_FIELDS");
		addExtraField(extraFields, "ACTIVATION_ID", ds.getActivationId());
		addExtraField(extraFields, "NamespacePrefix", ds.getNamespaceprefix());
		addExtraField(extraFields, "aggregatorName", ds.getAggregator());
		addExtraField(extraFields, "dateOfCollection", "" + ds.getDateofcollection());
		addExtraField(extraFields, "dateOfValidation", "" + ds.getDateofvalidation());
		conf.addElement("REGISTERED_BY").setText(ds.getRegisteredby());

		final Element status = body.addElement("STATUS");
		status.addElement("NUMBER_OF_OBJECTS").setText("0");
		status.addElement("LAST_UPDATE").addAttribute("value", DateUtils.now_ISO8601());

		final Element qos = body.addElement("QOS");
		qos.addElement("AVAILABILITY").setText("0");
		qos.addElement("CAPACITY");
		qos.addElement("THROUGHPUT").setText("0");

		body.addElement("SECURITY_PARAMETERS");
		body.addElement("BLACKBOARD");

		return root.asXML();
	}

	public static String asRepositoryInterfce(final ApiDetails api) {
		final Element iface = DocumentHelper.createElement("INTERFACE");

		iface.addAttribute("active", String.valueOf(api.getActive()))
			.addAttribute("compliance", api.getCompatibility())
			.addAttribute("contentDescription", api.getContentdescription())
			.addAttribute("id", api.getId())
			.addAttribute("label", String.format("%s (%s)", api.getTypology(), api.getCompatibility()))
			.addAttribute("removable", String.valueOf(api.getRemovable()))
			.addAttribute("typology", api.getTypology());
		iface.addElement("ACCESS_PROTOCOL").setText(api.getProtocol());
		if (api.getApiParams() != null) {
			final Element accessProtocol = (Element) iface.selectSingleNode("./ACCESS_PROTOCOL");
			api.getApiParams().forEach(ap -> {
				if ("format".equals(ap.getParam())) {
					accessProtocol.addAttribute("format", ap.getValue());
				}
				if ("set".equals(ap.getParam())) {
					accessProtocol.addAttribute("set", ap.getValue());
				}
			});
		}
		iface.addElement("BASE_URL").setText(api.getBaseurl());
		iface.addElement("INTERFACE_EXTRA_FIELD").addAttribute("name", "last_collection_date");
		iface.addElement("INTERFACE_EXTRA_FIELD").addAttribute("name", "last_collection_mdId");
		iface.addElement("INTERFACE_EXTRA_FIELD").addAttribute("name", "last_collection_total");
		iface.addElement("INTERFACE_EXTRA_FIELD").addAttribute("name", "last_aggregation_date");
		iface.addElement("INTERFACE_EXTRA_FIELD").addAttribute("name", "last_aggregation_mdId");
		iface.addElement("INTERFACE_EXTRA_FIELD").addAttribute("name", "last_aggregation_total");
		iface.addElement("INTERFACE_EXTRA_FIELD").addAttribute("name", "metadata_identifier_path");

		return iface.asXML();
	}

	// HELPERS

	private static <T> T _convert(final Object o, final Class<T> clazz) {
		final ObjectMapper mapper = new ObjectMapper();
		return mapper.convertValue(o, clazz);
	}

	private static OrganizationDetails getOrganization(final DatasourceDetails ds) {
		if (ds.getOrganizations() != null && !ds.getOrganizations().isEmpty()) {
			return ds.getOrganizations().stream().findFirst().get();
		}
		return null;
	}

	private static void addExtraField(final Element extraFields, final String field, final String value) {
		final Element f = extraFields.addElement("FIELD");
		f.addElement("key").setText(field);
		f.addElement("value").setText(value != null ? value : "");
	}

	private static String asOpenaireId(final String id) {
		final String prefix = StringUtils.substringBefore(id, ID_SEPARATOR);
		final String md5 = StringUtils.substringAfter(id, ID_SEPARATOR);

		return prefix + ID_SEPARATOR + AbstractDNetXsltFunctions.md5(md5);
	}

}
