package eu.dnetlib.openaire.community;

import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.base.Functions;
import com.google.common.base.Joiner;
import com.google.common.collect.Lists;
import eu.dnetlib.openaire.common.ISClient;
import eu.dnetlib.openaire.context.Category;
import eu.dnetlib.openaire.context.Concept;
import eu.dnetlib.openaire.context.Context;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import static eu.dnetlib.openaire.community.CommunityConstants.*;

@Component
public class CommunityApiCore {

	private static final Log log = LogFactory.getLog(CommunityApiCore.class);

	@Autowired
	private ISClient isClient;

	public List<CommunitySummary> listCommunities() throws CommunityException {
		return getContextMap().values().stream()
				.filter(context -> !communityBlackList.contains(context.getId()))
				.map(CommunityMappingUtils::asCommunitySummary)
				.collect(Collectors.toList());
	}

	public CommunityDetails getCommunity(final String id) throws CommunityException, CommunityNotFoundException {
		final Context context = getContextMap().get(id);
		if (context == null || CommunityConstants.communityBlackList.contains(id)) {
			throw new CommunityNotFoundException(String.format("community '%s' does not exist", id));
		}
		return CommunityMappingUtils.asCommunityProfile(context);
	}

	public void setCommunity(final String id, final CommunityWritableProperties details) throws CommunityException, CommunityNotFoundException {

		getCommunity(id); // ensure the community exists.

		isClient.updateContextAttribute(id, CLABEL, details.getShortName());
		isClient.updateContextParam(id, CSUMMARY_NAME, details.getName());
		isClient.updateContextParam(id, CSUMMARY_DESCRIPTION, details.getDescription());
		isClient.updateContextParam(id, CSUMMARY_LOGOURL, details.getLogoUrl());
		isClient.updateContextParam(id, CSUMMARY_MANAGER, Joiner.on(CSV_DELIMITER).join(details.getManagers()));
		isClient.updateContextParam(id, CPROFILE_SUBJECT, Joiner.on(CSV_DELIMITER).join(details.getSubjects()));
	}

	public List<CommunityProject> getCommunityProjects(final String id) throws CommunityException, CommunityNotFoundException {
		getCommunity(id); // ensure the community exists.
		return _getCommunityInfo(id, PROJECTS_ID_SUFFIX, c -> CommunityMappingUtils.asCommunityProject(id, c));
	}

	public void addCommunityProject(final String id, final CommunityProject project) throws CommunityException, CommunityNotFoundException {
		if (!StringUtils.equalsIgnoreCase(id, project.getCommunityId())) {
			throw new CommunityException("parameters 'id' and project.communityId must be coherent");
		}

		if (StringUtils.isBlank(project.getId())) {
			throw new CommunityException("parameter 'id' must be non empty");
		}

		final Map<String, CommunityProject> projects = getCommunityProjectMap(id);

		if (projects.containsKey(project.getId())) {
			throw new CommunityException(String.format("project '%s' already defined in context '%s'", project.getId(), id));
		}

		isClient.addConcept(id, id + PROJECTS_ID_SUFFIX, CommunityMappingUtils.asProjectXML(id, project));
	}

	public void removeCommunityProject(final String id, final String projectId) throws CommunityException, CommunityNotFoundException {
		final Map<String, CommunityProject> projects = getCommunityProjectMap(id);
		if (!projects.containsKey(projectId)) {
			throw new CommunityNotFoundException(String.format("project '%s' doesn't exist within context '%s'", projectId, id));
		}
		isClient.removeConcept(
				id,
				id + PROJECTS_ID_SUFFIX,
				id + PROJECTS_ID_SUFFIX + ID_SEPARATOR + projectId);
	}

	public List<CommunityContentprovider> getCommunityContentproviders(final String id) throws CommunityException, CommunityNotFoundException {
		getCommunity(id); // ensure the community exists.
		return _getCommunityInfo(id, CONTENTPROVIDERS_ID_SUFFIX, c -> CommunityMappingUtils.asCommunityDataprovider(id, c));
	}

	public void addCommunityContentprovider(final String id, final CommunityContentprovider cp) throws CommunityException, CommunityNotFoundException {
		if (!StringUtils.equalsIgnoreCase(id, cp.getCommunityId())) {
			throw new CommunityException("parameters 'id' and cp.communityId must be coherent");
		}

		if (StringUtils.isBlank(cp.getId())) {
			throw new CommunityException("parameter 'id' must be non empty");
		}

		final Map<String, CommunityContentprovider> cps = getCommunityContentproviderMap(id);

		if (cps.containsKey(cp.getId())) {
			throw new CommunityException(String.format("content provider '%s' already defined in context '%s'", cp.getId(), id));
		}
		isClient.addConcept(id, id + CONTENTPROVIDERS_ID_SUFFIX, CommunityMappingUtils.asContentProviderXML(id, cp));
	}

	public void removeCommunityContentProvider(final String id, final String contentproviderId) throws CommunityException, CommunityNotFoundException {
		final Map<String, CommunityContentprovider> providers = getCommunityContentproviderMap(id);
		if (!providers.containsKey(contentproviderId)) {
			throw new CommunityNotFoundException(String.format("content provider '%s' doesn't exist within context '%s'", contentproviderId, id));
		}
		isClient.removeConcept(
				id,
				id + CONTENTPROVIDERS_ID_SUFFIX,
				id + CONTENTPROVIDERS_ID_SUFFIX + ID_SEPARATOR + contentproviderId);
	}

	// HELPERS

	private Map<String, CommunityProject> getCommunityProjectMap(final String id) throws CommunityException, CommunityNotFoundException {
		return getCommunityProjects(id).stream()
				.collect(Collectors.toMap(
						CommunityProject::getId,
						Functions.identity(),
						(p1, p2) -> {
							log.warn("duplicate project found! " + p1.getId());
							return p2;
						}));
	}

	private Map<String, CommunityContentprovider> getCommunityContentproviderMap(final String id) throws CommunityException, CommunityNotFoundException {
		return getCommunityContentproviders(id).stream()
				.collect(Collectors.toMap(
						CommunityContentprovider::getId,
						Functions.identity(),
						(cp1, cp2) -> {
							log.warn("duplicate content provider found! " + cp1.getId());
							return cp2;
						}));
	}

	private Map<String, Context> getContextMap() throws CommunityException {
		try {
			return isClient.getCommunityContextMap();
		} catch (IOException e) {
			throw new CommunityException(e);
		}
	}

	private <R> List<R> _getCommunityInfo(final String id, final String idSuffix, final Function<Concept, R> mapping) throws CommunityException {
		final Map<String, Context> contextMap = getContextMap();
		final Context context = contextMap.get(id);
		if (context != null) {
			final Map<String, Category> categories = context.getCategories();
			final Category category = categories.get(id + idSuffix);
			if (category != null) {
				return category.getConcepts().stream()
						.map(mapping)
						.collect(Collectors.toList());
			}
		}
		return Lists.newArrayList();
	}

}
