package eu.dnetlib.openaire.dsm;

import java.util.List;
import javax.validation.Valid;

import eu.dnetlib.enabling.datasources.common.DsmException;
import eu.dnetlib.enabling.datasources.common.DsmForbiddenException;
import eu.dnetlib.enabling.datasources.common.DsmNotFoundException;
import eu.dnetlib.openaire.dsm.domain.*;
import eu.dnetlib.openaire.common.AbstractExporterController;
import eu.dnetlib.openaire.common.OperationManager;
import eu.dnetlib.openaire.dsm.domain.ApiDetailsResponse;
import eu.dnetlib.openaire.dsm.domain.DatasourceResponse;
import eu.dnetlib.openaire.dsm.domain.Response;
import eu.dnetlib.openaire.vocabularies.Country;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.StopWatch;
import org.apache.http.HttpStatus;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import static eu.dnetlib.openaire.common.ExporterConstants.*;

@RestController
@io.swagger.annotations.Api(tags = "OpenAIRE DSM API", description = "the OpenAIRE Datasource Manager API")
public class DsmApiController extends AbstractExporterController {

	@Autowired
	private DsmCore dsmCore;

	@RequestMapping(value = "/ds/countries", produces = { "application/json" }, method = RequestMethod.GET)
	@ApiOperation(
			value = "list the datasource countries",
			notes = "list the datasource countries",
			tags = { DS, R },
			response = Country[].class)
	@ApiResponses(value = {
			@ApiResponse(code = 200, message = "OK", response = Country[].class),
			@ApiResponse(code = 500, message = "unexpected error", response = ErrorMessage.class) })
	public List<Country> listCountries() throws DsmException {
		return dsmCore.listCountries();
	}

	@RequestMapping(value = "/ds/search/{page}/{size}", produces = { "application/json" }, method = RequestMethod.POST)
	@ApiOperation(
			value = "search datasources",
			notes = "Returns list of Datasource details.",
			tags = { DS, R },
			response = DatasourceResponse.class)
	@ApiResponses(value = {
			@ApiResponse(code = 200, message = "OK", response = DatasourceResponse.class),
			@ApiResponse(code = 500, message = "unexpected error", response = ErrorMessage.class) })
	public DatasourceResponse search(
			@RequestParam final RequestSort requestSortBy,
			@RequestParam final RequestSortOrder order,
			@RequestBody final RequestFilter requestFilter,
			@PathVariable final int page,
			@PathVariable final int size) throws DsmException {
		final StopWatch stop = StopWatch.createStarted();
		final DatasourceResponse rsp = dsmCore.search(requestSortBy, order, requestFilter, page, size);
		return prepareResponse(page, size, stop, rsp);
	}

	@RequestMapping(value = "/ds/api/{dsId}", produces = { "application/json" }, method = RequestMethod.GET)
	@ApiOperation(
			value = "get the list of API for a given datasource",
			notes = "Returns the list of API for a given datasource.",
			tags = { API, R },
			response = ApiDetailsResponse.class)
	@ApiResponses(value = {
			@ApiResponse(code = 200, message = "OK", response = ApiDetailsResponse.class),
			@ApiResponse(code = 500, message = "unexpected error", response = ErrorMessage.class) })
	public ApiDetailsResponse getApi(
			@PathVariable final String dsId) throws DsmException {

		final StopWatch stop = StopWatch.createStarted();
		final ApiDetailsResponse rsp = dsmCore.getApis(dsId);
		return prepareResponse(0, rsp.getApi().size(), stop, rsp);
	}

	@RequestMapping(value = "/api/baseurl/{page}/{size}", produces = { "application/json" }, method = RequestMethod.POST)
	@ApiOperation(
			value = "search for the list of base URLs of Datasource APIs managed by a user",
	        notes = "Returns the list of base URLs of Datasource APIs managed by a user",
			tags = { DS, API, R },
			response = String[].class)
	@ApiResponses(value = {
			@ApiResponse(code = 200, message = "OK", response = String[].class),
			@ApiResponse(code = 500, message = "unexpected error", response = ErrorMessage.class) })
	public List<String> searchBaseUrls(
			@RequestBody final RequestFilter requestFilter,
			@PathVariable final int page,
			@PathVariable final int size) throws DsmException {

		return dsmCore.findBaseURLs(requestFilter, page, size);
	}

	@RequestMapping(value = "/ds/api/{apiId}", method = RequestMethod.DELETE)
	@ApiOperation(
			value = "delete an API",
			notes = "delete an API, if removable",
			tags = { API, W })
	@ApiResponses(value = {
			@ApiResponse(code = 200, message = "OK"),
			@ApiResponse(code = 403, message = "Api not removable", response = ErrorMessage.class) })
	public void deleteApi(@PathVariable final String apiId) throws DsmException, DsmForbiddenException {
		dsmCore.deleteApi(apiId);
	}

	@RequestMapping(value = "/ds/manage", method = RequestMethod.POST)
	@ApiOperation(
			value = "set the managed status for a given datasource",
			notes = "set the managed status for a given datasource",
			tags = { DS, W })
	@ApiResponses(value = {
			@ApiResponse(code = 200, message = "OK"),
			@ApiResponse(code = 500, message = "unexpected error", response = ErrorMessage.class) })
	public void setManaged(
			@RequestParam final String id,
			@RequestParam final boolean managed) throws DsmException {

		dsmCore.setManaged(id, managed);
	}

	@RequestMapping(value = "/ds/managed/{dsId}", method = RequestMethod.GET)
	@ApiOperation(
			value = "get the datasource managed status",
			notes = "get the datasource managed status",
			tags = { DS, R })
	@ApiResponses(value = {
			@ApiResponse(code = 200, message = "OK"),
			@ApiResponse(code = 500, message = "unexpected error", response = ErrorMessage.class) })
	public boolean isManaged(@PathVariable final String id) throws DsmException {
		return dsmCore.isManaged(id);
	}

	@RequestMapping(value = "/ds/add", method = RequestMethod.POST)
	@ApiOperation(
			value = "add a new Datasource",
			notes = "add a new Datasource",
			tags = { DS, W })
	@ApiResponses(value = {
			@ApiResponse(code = 200, message = "OK"),
			@ApiResponse(code = 400, message = "Malformed request", response = ErrorMessage[].class),
			@ApiResponse(code = 500, message = "Unexpected error", response = ErrorMessage.class) })
	public void saveDs(@Valid @RequestBody final DatasourceDetails datasource) throws DsmException {

		if (dsmCore.exist(datasource)) { // TODO further check that the DS doesn't have any API
			throw new DsmException(HttpStatus.SC_CONFLICT, String.format("cannot register, datasource already defined '%s'", datasource.getId()));
		}
		dsmCore.save(datasource);
	}

	@RequestMapping(value = "/ds/update", method = RequestMethod.POST)
	@ApiOperation(
			value = "update Datasource details",
			notes = "update Datasource details",
			tags = { DS, W })
	@ApiResponses(value = {
			@ApiResponse(code = 200, message = "OK"),
			@ApiResponse(code = 500, message = "unexpected error", response = ErrorMessage.class) })
	public void updateDatasource(
			@RequestBody final DatasourceDetailsUpdate ds) throws DsmException, DsmNotFoundException {

		dsmCore.updateDatasource(ds);
	}

	@RequestMapping(value = "/ds/logourl", method = RequestMethod.POST)
	@ApiOperation(
			value = "update a datasource name",
			notes = "update a datasource name",
			tags = { DS, W, D })
	@ApiResponses(value = {
			@ApiResponse(code = 200, message = "OK"),
			@ApiResponse(code = 500, message = "unexpected error", response = ErrorMessage.class) })
	public void updateDatasourceLogoURL(
			@RequestParam final String dsId,
			@RequestParam final String logourl) throws DsmException {

		dsmCore.updateDatasourceLogoUrl(dsId, logourl);
	}

	@RequestMapping(value = "/ds/name", method = RequestMethod.POST)
	@ApiOperation(
			value = "update a datasource name",
			notes = "update a datasource name",
			tags = { DS, W, D })
	@ApiResponses(value = {
			@ApiResponse(code = 200, message = "OK"),
			@ApiResponse(code = 500, message = "unexpected error", response = ErrorMessage.class) })
	public void updateDatasourceName(
			@RequestParam final String dsId,
			@RequestParam final String officialname,
			@RequestParam final String englishname) throws DsmException {

		dsmCore.updateDatasourcename(dsId, officialname, englishname);
	}

	@RequestMapping(value = "/ds/coordinates", method = RequestMethod.POST)
	@ApiOperation(
			value = "update the datasource coordinates (latitude, longitude)",
			notes = "update the datasource coordinates (latitude, longitude)",
			tags = { DS, W, D })
	@ApiResponses(value = {
			@ApiResponse(code = 200, message = "OK"),
			@ApiResponse(code = 500, message = "unexpected error", response = ErrorMessage.class) })
	public void updateCoordinates(
			@RequestParam final String dsId,
			@RequestParam final Double latitude,
			@RequestParam final Double longitude) throws DsmException {

		dsmCore.updateCoordinates(dsId, latitude, longitude);
	}

	@RequestMapping(value = "/ds/timezone", method = RequestMethod.POST)
	@ApiOperation(
			value = "update a datasource timezone",
			notes = "update a datasource timezone",
			tags = { DS, W, D } )
	@ApiResponses(value = {
			@ApiResponse(code = 200, message = "OK"),
			@ApiResponse(code = 500, message = "unexpected error", response = ErrorMessage.class) })
	public void updateTimezone(
			@RequestParam final String dsId,
			@RequestParam final String timezone) throws DsmException {

		dsmCore.updateTimezone(dsId, timezone);
	}

	@RequestMapping(value = "/ds/typology", method = RequestMethod.POST)
	@ApiOperation(
			value = "update a datasource typology code",
			notes = "update a datasource typology code",
			tags = { DS, W, D } )
	@ApiResponses(value = {
			@ApiResponse(code = 200, message = "OK"),
			@ApiResponse(code = 500, message = "unexpected error", response = ErrorMessage.class) })
	public void updateTypology(
			@RequestParam final String dsId,
			@RequestParam final String typology) throws DsmException {

		dsmCore.updateDsTypology(dsId, typology);
	}

	@RequestMapping(value = "/ds/registeredby", method = RequestMethod.POST)
	@ApiOperation(
			value = "update a datasource registeredBy",
			notes = "update a datasource registeredBy",
			tags = { DS, W, D } )
	@ApiResponses(value = {
			@ApiResponse(code = 200, message = "OK"),
			@ApiResponse(code = 500, message = "unexpected error", response = ErrorMessage.class) })
	public void updateRegisteringUser(
			@RequestParam final String dsId,
			@RequestParam final String registeredBy) throws DsmException {

		dsmCore.updateDsRegisteringUser(dsId, registeredBy);
	}

	@RequestMapping(value = "/ds/platform", method = RequestMethod.POST)
	@ApiOperation(
			value = "update a datasource platform",
			notes = "update a datasource platform",
			tags = { DS, W, D } )
	@ApiResponses(value = {
			@ApiResponse(code = 200, message = "OK"),
			@ApiResponse(code = 500, message = "unexpected error", response = ErrorMessage.class) })
	public void updatePlatform(
			@RequestParam final String dsId,
			@RequestParam final String platform) throws DsmException {

		dsmCore.updateDsPlatform(dsId, platform);
	}

	@RequestMapping(value = "/ds/api/baseurl", method = RequestMethod.POST)
	@ApiOperation(
			value = "update the base URL of a datasource API",
			notes = "update the base URL of a datasource API",
			tags = { API, W })
	@ApiResponses(value = {
			@ApiResponse(code = 200, message = "OK"),
			@ApiResponse(code = 500, message = "unexpected error", response = ErrorMessage.class) })
	public void updateBaseUrl(
			@RequestParam final String dsId,
			@RequestParam final String apiId,
			@RequestParam final String baseUrl) throws DsmException {

		dsmCore.updateApiBaseurl(dsId, apiId, baseUrl);
	}

	@RequestMapping(value = "/ds/api/compliance", method = RequestMethod.POST)
	@ApiOperation(
			value = "update the compatibility of a datasource API",
			notes = "update the compatibility of a datasource API",
			tags = { API, W })
	@ApiResponses(value = {
			@ApiResponse(code = 200, message = "OK"),
			@ApiResponse(code = 500, message = "unexpected error", response = ErrorMessage.class) })
	public void updateCompliance(
			@RequestParam final String dsId,
			@RequestParam final String apiId,
			@RequestParam final String compliance,
			@RequestParam(required = false, defaultValue = "false") boolean override) throws DsmException {

		dsmCore.updateApiCompatibility(dsId, apiId, compliance, override);
	}

	@RequestMapping(value = "/ds/api/add", method = RequestMethod.POST)
	@ApiOperation(
			value = "adds a new API to one Datasource",
			notes = "adds an API to one Datasource",
			tags = { API, W })
	@ApiResponses(value = {
			@ApiResponse(code = 200, message = "OK"),
			@ApiResponse(code = 500, message = "unexpected error", response = ErrorMessage.class) })
	public void addApi(@RequestBody final ApiDetails api) throws DsmException {
		if (StringUtils.isBlank(api.getDatasource())) {
			throw new DsmException(HttpStatus.SC_BAD_REQUEST, "missing datasource id");
		}
		dsmCore.addApi(api);
	}

	// MANAGEMENT

	@Autowired
	private OperationManager operationManager;

	@RequestMapping(value = "/dsm/ops", method = RequestMethod.GET)
	@ApiOperation(
			value = "get the number of pending operations",
			notes = "get the number of pending operations",
			tags = { R, M })
	@ApiResponses(value = {
			@ApiResponse(code = 200, message = "OK"),
			@ApiResponse(code = 500, message = "unexpected error", response = ErrorMessage.class) })
	public int getOps() throws DsmException {
		return operationManager.getOpSize();
	}

	@RequestMapping(value = "/dsm/killops", method = RequestMethod.POST)
	@ApiOperation(
			value = "interrupts the pending operations",
			notes = "return the number of interrupted operations",
			tags = { W, M })
	@ApiResponses(value = {
			@ApiResponse(code = 200, message = "OK"),
			@ApiResponse(code = 500, message = "unexpected error", response = ErrorMessage.class) })
	public int killOps() throws DsmException {
		return operationManager.dropAll();
	}

	@RequestMapping(value = "/dsm/dropcache", method = RequestMethod.POST)
	@ApiOperation(
			value = "drop the caches",
			notes = "drop the internal caches",
			tags = { W, M })
	@ApiResponses(value = {
			@ApiResponse(code = 200, message = "OK"),
			@ApiResponse(code = 500, message = "unexpected error", response = ErrorMessage.class) })
	public void dropCache() throws DsmException {
		dsmCore.dropCaches();
	}


	// OLD DEPRECATED METHODS

	@ApiOperation(value = "search datasources by name", notes = "Returns list of Datasource details.", tags = { D })
	@ApiResponses(value = {
			@ApiResponse(code = 200, message = "OK", response = DatasourceResponse.class),
			@ApiResponse(code = 500, message = "unexpected error", response = ErrorMessage.class) })
	@RequestMapping(value = "/ds/search/name/{page}/{size}", produces = { "application/json" }, method = RequestMethod.GET)
	DatasourceResponse searchByName(String name, int page, int size) throws DsmException {
		final RequestSort sort = RequestSort.id;
		final RequestSortOrder order = RequestSortOrder.ASCENDING;
		final RequestFilter filter = new RequestFilter();
		filter.put(FilterName.englishname, name);

		return dsmCore.search(sort, order, filter, page, size);
	}

	@ApiOperation(value = "search datasources by contact email", notes = "Returns list of Datasource details.", tags = { D })
	@ApiResponses(value = {
			@ApiResponse(code = 200, message = "OK", response = DatasourceResponse.class),
			@ApiResponse(code = 500, message = "unexpected error", response = ErrorMessage.class) })
	@RequestMapping(value = "/ds/search/email/{page}/{size}", produces = { "application/json" }, method = RequestMethod.GET)
	DatasourceResponse searchByContactemail(String contactemail, int page, int size) throws DsmException {
		final RequestSort sort = RequestSort.id;
		final RequestSortOrder order = RequestSortOrder.ASCENDING;
		final RequestFilter filter = new RequestFilter();
		filter.put(FilterName.contactemail, contactemail);

		return dsmCore.search(sort, order, filter, page, size);
	}

	@ApiOperation(value = "search datasources by country", notes = "Returns list of Datasource details.", tags = { D })
	@ApiResponses(value = {
			@ApiResponse(code = 200, message = "OK", response = DatasourceResponse.class),
			@ApiResponse(code = 500, message = "unexpected error", response = ErrorMessage.class) })
	@RequestMapping(value = "/ds/search/country/{page}/{size}", produces = { "application/json" }, method = RequestMethod.GET)
	DatasourceResponse searchByCountry(String country, Boolean managed, int page, int size) throws DsmException {
		final RequestSort sort = RequestSort.id;
		final RequestSortOrder order = RequestSortOrder.ASCENDING;
		final RequestFilter filter = new RequestFilter();
		filter.put(FilterName.country, country);
		filter.put(FilterName.managed, managed);

		return dsmCore.search(sort, order, filter, page, size);
	}

	@ApiOperation(value = "search datasources by registering user", notes = "Returns list of Datasource details.", tags = { D })
	@ApiResponses(value = {
			@ApiResponse(code = 200, message = "OK", response = DatasourceResponse.class),
			@ApiResponse(code = 500, message = "unexpected error", response = ErrorMessage.class) })
	@RequestMapping(value = "/ds/search/registeredby/{page}/{size}", produces = { "application/json" }, method = RequestMethod.GET)
	DatasourceResponse searchByRegisteringUser(String registeredBy, int page, int size) throws DsmException {
		final RequestSort sort = RequestSort.id;
		final RequestSortOrder order = RequestSortOrder.ASCENDING;
		final RequestFilter filter = new RequestFilter();
		filter.put(FilterName.registeredby, registeredBy);

		return dsmCore.search(sort, order, filter, page, size);
	}

	// HELPERS

	private <T extends Response> T prepareResponse(int page, int size, final StopWatch stop, final Response rsp) {
		rsp.getHeader()
				.setTime(stop.getTime())
				.setPage(page)
				.setSize(size);
		return (T) rsp;
	}
}
