package eu.dnetlib.openaire.dsm.dao;

import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.collect.Lists;
import eu.dnetlib.OpenaireExporterConfig;
import eu.dnetlib.enabling.datasources.common.DsmException;
import eu.dnetlib.openaire.dsm.domain.RequestFilter;
import eu.dnetlib.openaire.dsm.domain.RequestSort;
import eu.dnetlib.openaire.dsm.domain.RequestSortOrder;
import eu.dnetlib.openaire.dsm.domain.db.ApiDbEntry;
import eu.dnetlib.openaire.dsm.domain.db.DatasourceApiDbEntry;
import eu.dnetlib.openaire.dsm.domain.db.DatasourceDbEntry;
import eu.dnetlib.openaire.vocabularies.Country;
import eu.dnetlib.openaire.vocabularies.Vocabulary;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.http.HttpStatus;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Component;

import static eu.dnetlib.openaire.dsm.dao.DatasourceSpecs.apiSpec;
import static eu.dnetlib.openaire.dsm.dao.DatasourceSpecs.dsSpec;

/**
 * Created by claudio on 20/10/2016.
 */
@Component
public class DatasourceDaoImpl implements DatasourceDao<DatasourceDbEntry, ApiDbEntry> {

	private static final Log log = LogFactory.getLog(DatasourceDao.class);

	@Autowired
	private OpenaireExporterConfig config;

	@Autowired
	private CountryTermRepository countryTermRepository;

	@Autowired
	private DatasourceDbEntryRepository dsRepository;

	@Autowired
	private ApiDbEntryRepository apiRepository;

	@Autowired
	private DatasourceApiDbEntryRepository dsApiRepository;

	@Autowired
	private VocabularyClient vocabularyClient;

	@Override
	public List<Country> listCountries() throws DsmException {
		final List<Country> countries = Lists.newArrayList();
		final Vocabulary v = vocabularyClient.getCountries();
		countries.addAll(countryTermRepository.findAll().stream()
				.filter(Objects::nonNull)
				.map(t -> new Country(t.getTerm(), v.getEnglishName(t.getTerm())))
				.collect(Collectors.toList()));
		return countries;
	}

	@Override
	public Page<DatasourceDbEntry> search(final RequestSort requestSortBy, RequestSortOrder order, RequestFilter requestFilter, final int page, final int size)
			throws DsmException {

		final Specification<DatasourceDbEntry> spec = dsSpec(requestSortBy, order, requestFilter);
		return dsRepository.findAll(spec, new PageRequest(page, size));
	}

	@Override
	public DatasourceDbEntry getDs(final String dsId) throws DsmException {
		return dsRepository.findOne(dsId);
	}

	@Override
	public void setManaged(final String id, final boolean managed) {
		log.info(String.format("setting managed = '%s' for ds '%s'", managed, id));
		dsRepository.setManaged(id, managed);
	}

	@Override
	public boolean isManaged(final String id) {
		return dsRepository.isManaged(id);
	}

	@Override
	public void updateCompliance(String dsId, String apiId, String compliance, boolean override) {
		log.info(String.format("setting compatibility = '%s' for ds '%s'", compliance, apiId));
		apiRepository.updateCompatibility(apiId, compliance);
	}

	@Override
	public List<ApiDbEntry> getApis(final String dsId) {
		return apiRepository.findByDatasource(dsId);
	}

	@Override
	public void deleteApi(final String dsId, final String apiId) throws DsmException {
		final ApiDbEntry api = apiRepository.findOne(apiId);
		if (!api.getRemovable()) {
			throw new DsmException(HttpStatus.SC_UNAUTHORIZED, "api is not removable");
		}

		apiRepository.delete(apiId);
		log.info(String.format("deleted api '%s'", apiId));
	}

	@Override
	public void addApi(final ApiDbEntry api) {
		apiRepository.save(api);
	}

	public boolean existDs(final String dsId) throws DsmException {
		return dsRepository.exists(dsId);
	}

	@Override
	public void saveDs(final DatasourceDbEntry d) {
		log.info(String.format("saving datasource '%s'", d.getId()));
		final DatasourceDbEntry datasource = dsRepository.save(d);
		log.info(String.format("saved datasource '%s'", datasource.getId()));
	}

	@Override
	public void deleteDs(final String dsId) {
		dsRepository.delete(dsId);
		log.info(String.format("deleted datasource '%s'", dsId));
	}

	@Override
	public void updateName(final String dsId, final String officialname, final String englishname) {
		//TODO what if one of the two names is null or empty?
		dsRepository.setDatasourcename(dsId, officialname, englishname);
	}

	@Override
	public void updateLogoUrl(final String dsId, final String logourl) throws DsmException {
		dsRepository.setLogoUrl(dsId, logourl);
	}

	@Override
	public void updateCoordinates(final String dsId, final Double latitude, final Double longitude) {
		dsRepository.setCoordinates(dsId, latitude, longitude);
	}

	@Override
	public void updateApiBaseUrl(final String apiId, final String baseurl) {
		apiRepository.setBaseurl(apiId, baseurl);
	}

	@Override
	public List<String> findApiBaseURLs(final RequestFilter requestFilter, final int page, final int size) throws DsmException {
		final PageRequest pageable = new PageRequest(page, size);
		final Specification<DatasourceApiDbEntry> spec = apiSpec(requestFilter);
		final Set<String> set = dsApiRepository.findAll(spec, pageable).getContent().stream()
				.map(DatasourceApiDbEntry::getBaseurl)
				.filter(StringUtils::isNotBlank)
				.collect(Collectors.toCollection(HashSet::new));
		return Lists.newArrayList(set);
	}

	@Override
	public void updateTimezone(final String dsId, final String timezone) {
		dsRepository.setTimezone(dsId, timezone);
	}

	@Override
	public void updateTypology(final String dsId, final String typology) throws DsmException {
		final Vocabulary typologies = vocabularyClient.getDatasourceTypologies();
		if (!typologies.hasCode(typology)) {
			throw new DsmException(
					HttpStatus.SC_BAD_REQUEST,
					String.format(
							"invalid datasource typology '%s', provide one according to vocabulary %s",
							typology,
							config.getVocabularies().getDatasourceTypologiesEndpoint()));
		}
		dsRepository.setTypology(dsId, typology);
	}

	@Override
	public void updateRegisteringUser(final String dsId, final String registeredBy) throws DsmException {
		dsRepository.setRegisteringUser(dsId, registeredBy);
	}

	@Override
	public void updatePlatform(final String dsId, final String platform) throws DsmException {
		dsRepository.setPlatform(dsId, platform);
	}

}
