package eu.dnetlib.openaire.community;

import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.google.common.escape.Escaper;
import com.google.common.xml.XmlEscapers;
import eu.dnetlib.openaire.context.Concept;
import eu.dnetlib.openaire.context.Context;
import eu.dnetlib.openaire.context.Param;
import org.apache.commons.lang3.StringUtils;

import static eu.dnetlib.openaire.common.Utils.escape;
import static eu.dnetlib.openaire.community.CommunityConstants.*;

public class CommunityMappingUtils {

	public static CommunitySummary asCommunitySummary(final Context c) {
		final CommunitySummary summary = new CommunitySummary();

		summary.setId(c.getId());
		summary.setShortName(c.getLabel());
		summary.setQueryId(c.getId() + PIPE_SEPARATOR + c.getLabel());
		summary.setType(c.getType());

		final Map<String, List<Param>> params = c.getParams();
		if (params.containsKey(CSUMMARY_DESCRIPTION)) {
			summary.setDescription(asCsv(params.get(CSUMMARY_DESCRIPTION)));
		}
		if (params.containsKey(CSUMMARY_LOGOURL)) {
			summary.setLogoUrl(asCsv(params.get(CSUMMARY_LOGOURL)));
		}
		if (params.containsKey(CSUMMARY_STATUS)) {
			summary.setStatus(CommunityStatus.valueOf(firstValue(params, CSUMMARY_STATUS)));
		}
		if (params.containsKey(CSUMMARY_NAME)) {
			summary.setName(asCsv(params.get(CSUMMARY_NAME)));
		}
		if (params.containsKey(CSUMMARY_MANAGER)) {
			summary.setManagers(splitValues(asValues(params.get(CSUMMARY_MANAGER)), CSV_DELIMITER));
		}

		return summary;
	}

	public static CommunityDetails asCommunityProfile(final Context c) {

		final CommunityDetails p = new CommunityDetails(asCommunitySummary(c));
		p.setCreationDate(c.getDateofcreation());
		final Map<String, List<Param>> params = c.getParams();
		if (params.containsKey(CPROFILE_SUBJECT)) {
			p.setSubjects(splitValues(asValues(params.get(CPROFILE_SUBJECT)), CSV_DELIMITER));
		}

		return p;
	}

	public static CommunityProject asCommunityProject(final String communityId, final Concept c) {

		final Map<String, List<Param>> p = c.getParams();
		final CommunityProject project = new CommunityProject();
		project.setCommunityId(communityId);
		project.setId(StringUtils.substringAfterLast(c.getId(), ID_SEPARATOR));
		project.setOpenaireId(firstValue(p, OPENAIRE_ID));
		project.setFunder(firstValue(p, CPROJECT_FUNDER));
		project.setGrantId(firstValue(p, CPROJECT_NUMBER));
		project.setName(firstValue(p, CPROJECT_FULLNAME));
		project.setAcronym(firstValue(p, CPROJECT_ACRONYM));

		return project;
	}

	public static CommunityContentprovider asCommunityDataprovider(final String communityId, final Concept c) {

		final Map<String, List<Param>> p = c.getParams();
		final CommunityContentprovider d = new CommunityContentprovider();
		d.setCommunityId(communityId);
		d.setId(StringUtils.substringAfterLast(c.getId(), ID_SEPARATOR));
		d.setOpenaireId(firstValue(p, OPENAIRE_ID));
		d.setName(firstValue(p, CCONTENTPROVIDER_NAME));
		d.setOfficialname(firstValue(p, CCONTENTPROVIDER_OFFICIALNAME));

		return d;
	}

	private static List<String> splitValues(final Stream<String> stream, final String separator) {
		return stream.map(s -> s.split(separator))
				.map(Arrays::asList)
				.flatMap(List::stream)
				.filter(StringUtils::isNotBlank)
				.map(StringUtils::trim)
				.collect(Collectors.toList());
	}

	private static String firstValue(final Map<String, List<Param>> p, final String paramName) {
		final Optional<String> first = asValues(p.get(paramName)).findFirst();
		return first.isPresent() ? first.get() : null;
	}

	private static String asCsv(final List<Param> params) {
		return asValues(params)
				.collect(Collectors.joining(CSV_DELIMITER));
	}

	private static Stream<String> asValues(final List<Param> params) {
		return params == null ? Stream.empty() : params.stream()
				.map(Param::getValue)
				.map(StringUtils::trim)
				.distinct();
	}

	public static String asProjectXML(final String contextId, final CommunityProject project) {
		final Escaper esc = XmlEscapers.xmlAttributeEscaper();
		final StringBuilder sb = new StringBuilder();
		sb.append(
				String.format(
						"<concept claim='true' id='%s%s%s%s' label='%s'>\n",
						escape(esc, contextId), PROJECTS_ID_SUFFIX, ID_SEPARATOR, escape(esc, String.valueOf(project.getId())), escape(esc, project.getAcronym())));
		sb.append(paramXML(CPROJECT_FULLNAME, project.getName()));
		sb.append(paramXML(CPROJECT_ACRONYM, project.getAcronym()));
		sb.append(paramXML(CPROJECT_NUMBER, project.getGrantId()));
		sb.append(paramXML(CPROJECT_FUNDER, project.getFunder()));
		sb.append(paramXML(OPENAIRE_ID, project.getOpenaireId()));
		sb.append("</concept>\n");
		return sb.toString();
	}

	public static String asContentProviderXML(final String contextId, final CommunityContentprovider ccp) {
		final Escaper esc = XmlEscapers.xmlAttributeEscaper();
		final StringBuilder sb = new StringBuilder();
		sb.append(
				String.format(
						"<concept claim='true' id='%s%s%s%s' label='%s'>\n",
						escape(esc, contextId), CONTENTPROVIDERS_ID_SUFFIX, ID_SEPARATOR, escape(esc, String.valueOf(ccp.getId())), escape(esc, ccp.getName())));
		sb.append(paramXML(OPENAIRE_ID, ccp.getOpenaireId()));
		sb.append(paramXML(CCONTENTPROVIDER_NAME, ccp.getName()));
		sb.append(paramXML(CCONTENTPROVIDER_OFFICIALNAME, ccp.getOfficialname()));
		sb.append("</concept>\n");
		return sb.toString();
	}

	private static String paramXML(final String paramName, final String value) {
		return String.format("<param name='%s'>%s</param>\n", paramName, escape(XmlEscapers.xmlContentEscaper(), value));
	}

}
