package eu.dnetlib.openaire.community;

import java.io.IOException;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.base.Functions;
import com.google.common.base.Joiner;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import eu.dnetlib.openaire.common.ISClient;
import eu.dnetlib.openaire.context.Category;
import eu.dnetlib.openaire.context.Concept;
import eu.dnetlib.openaire.context.Context;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.stereotype.Component;

import static eu.dnetlib.openaire.community.CommunityConstants.*;

@Component
@ConditionalOnProperty(value = "openaire.exporter.enable.community", havingValue = "true")
public class CommunityApiCore {

	private static final Log log = LogFactory.getLog(CommunityApiCore.class);

	@Autowired
	private ISClient isClient;

	public List<CommunitySummary> listCommunities() throws CommunityException {
		return getContextMap().values().stream()
				.filter(context -> !communityBlackList.contains(context.getId()))
				.map(CommunityMappingUtils::asCommunitySummary)
				.collect(Collectors.toList());
	}

	public CommunityDetails getCommunity(final String id) throws CommunityException, CommunityNotFoundException {
		final Context context = getContextMap().get(id);
		if (context == null || CommunityConstants.communityBlackList.contains(id)) {
			throw new CommunityNotFoundException(String.format("community '%s' does not exist", id));
		}
		return CommunityMappingUtils.asCommunityProfile(context);
	}

	public void setCommunity(final String id, final CommunityWritableProperties details) throws CommunityException, CommunityNotFoundException {

		getCommunity(id); // ensure the community exists.

		if(details.getShortName() != null) {
			isClient.updateContextAttribute(id, CLABEL, details.getShortName());
		}
		if (details.getName() != null){
			isClient.updateContextParam(id, CSUMMARY_NAME, details.getName());
		}
		if(details.getDescription() != null) {
			isClient.updateContextParam(id, CSUMMARY_DESCRIPTION, details.getDescription());
		}
		if(details.getLogoUrl()!=null){
		isClient.updateContextParam(id, CSUMMARY_LOGOURL, details.getLogoUrl());
		}
		if (details.getStatus() != null) {
			isClient.updateContextParam(id, CSUMMARY_STATUS, details.getStatus().name());
		}


		if (details.getManagers() != null) {
			isClient.updateContextParam(id, CSUMMARY_MANAGER, Joiner.on(CSV_DELIMITER).join(details.getManagers()));
		}
		if (details.getSubjects() != null) {
			isClient.updateContextParam(id, CPROFILE_SUBJECT, Joiner.on(CSV_DELIMITER).join(details.getSubjects()));
		}
	}

	public List<CommunityProject> getCommunityProjects(final String id) throws CommunityException, CommunityNotFoundException {
		getCommunity(id); // ensure the community exists.
		return _getCommunityInfo(id, PROJECTS_ID_SUFFIX, c -> CommunityMappingUtils.asCommunityProject(id, c));
	}

	public CommunityProject addCommunityProject(final String id, final CommunityProject project) throws CommunityException, CommunityNotFoundException {
		if (!StringUtils.equalsIgnoreCase(id, project.getCommunityId())) {
			throw new CommunityException("parameters 'id' and project.communityId must be coherent");
		}

		final TreeMap<Integer, CommunityProject> projects = getCommunityProjectMap(id);
		project.setId(nextId(projects != null ? projects.lastKey() : 0));

		isClient.addConcept(id, id + PROJECTS_ID_SUFFIX, CommunityMappingUtils.asProjectXML(id, project));

		return project;
	}

	private String nextId(final Integer id) {
		return String.valueOf(id + 1);
	}

	public void removeCommunityProject(final String id, final Integer projectId) throws CommunityException, CommunityNotFoundException {
		final Map<Integer, CommunityProject> projects = getCommunityProjectMap(id);
		if (!projects.containsKey(projectId)) {
			throw new CommunityNotFoundException(String.format("project '%s' doesn't exist within context '%s'", projectId, id));
		}
		isClient.removeConcept(
				id,
				id + PROJECTS_ID_SUFFIX,
				id + PROJECTS_ID_SUFFIX + ID_SEPARATOR + projectId);
	}

	public List<CommunityContentprovider> getCommunityContentproviders(final String id) throws CommunityException, CommunityNotFoundException {
		getCommunity(id); // ensure the community exists.
		return _getCommunityInfo(id, CONTENTPROVIDERS_ID_SUFFIX, c -> CommunityMappingUtils.asCommunityDataprovider(id, c));
	}

	public CommunityContentprovider addCommunityContentprovider(final String id, final CommunityContentprovider cp) throws CommunityException, CommunityNotFoundException {
		if (!StringUtils.equalsIgnoreCase(id, cp.getCommunityId())) {
			throw new CommunityException("parameters 'id' and cp.communityId must be coherent");
		}

		final TreeMap<Integer, CommunityContentprovider> cps = getCommunityContentproviderMap(id);
		cp.setId(nextId(MapUtils.isNotEmpty(cps) ? cps.lastKey() : 0));

		isClient.addConcept(id, id + CONTENTPROVIDERS_ID_SUFFIX, CommunityMappingUtils.asContentProviderXML(id, cp));

		return cp;
	}

	public void removeCommunityContentProvider(final String id, final Integer contentproviderId) throws CommunityException, CommunityNotFoundException {
		final Map<Integer, CommunityContentprovider> providers = getCommunityContentproviderMap(id);
		if (!providers.containsKey(contentproviderId)) {
			throw new CommunityNotFoundException(String.format("content provider '%s' doesn't exist within context '%s'", contentproviderId, id));
		}
		isClient.removeConcept(
				id,
				id + CONTENTPROVIDERS_ID_SUFFIX,
				id + CONTENTPROVIDERS_ID_SUFFIX + ID_SEPARATOR + contentproviderId);
	}


	public List<CommunityZenodoCommunity> getCommunityZenodoCommunities(final String id) throws CommunityException, CommunityNotFoundException {
		getCommunity(id); // ensure the community exists.
		return _getCommunityInfo(id, ZENODOCOMMUNITY_ID_SUFFIX, c -> CommunityMappingUtils.asCommunityZenodoCommunity(id, c));
	}


	public CommunityDetails addCommunitySubjects(final String id, final List<String> subjects) throws CommunityException, CommunityNotFoundException {

		final CommunityDetails details = getCommunity(id);

		final Set<String> current = Sets.newHashSet(details.getSubjects());

		current.addAll(subjects);

		details.setSubjects(Lists.newArrayList(current));

		setCommunity(id, CommunityWritableProperties.fromDetails(details));

		return details;
	}

	public CommunityDetails removeCommunitySubjects(final String id, final List<String> subjects) throws CommunityException, CommunityNotFoundException {

		final CommunityDetails details = getCommunity(id);

		final Set<String> current = Sets.newHashSet(details.getSubjects());

		current.removeAll(subjects);

		details.setSubjects(Lists.newArrayList(current));

		setCommunity(id, CommunityWritableProperties.fromDetails(details));

		return details;
	}

	public void removeCommunityZenodoCommunity(final String id, final Integer zenodoCommId) throws CommunityException, CommunityNotFoundException {

		final Map<Integer, CommunityZenodoCommunity> zcomms = getZenodoCommunityMap(id);
		if (!zcomms.containsKey(zenodoCommId)) {
			throw new CommunityNotFoundException(String.format("Zenodo community '%s' doesn't exist within context '%s'", zenodoCommId, id));
		}
		isClient.removeConcept(
				id,
				id + ZENODOCOMMUNITY_ID_SUFFIX,
				id + ZENODOCOMMUNITY_ID_SUFFIX + ID_SEPARATOR + zenodoCommId);
	}

	public CommunityZenodoCommunity addCommunityZenodoCommunity(final String id, final CommunityZenodoCommunity zc) throws CommunityException, CommunityNotFoundException {
		if (!StringUtils.equalsIgnoreCase(id, zc.getCommunityId())) {
			throw new CommunityException("parameters 'id' and zc.communityId must be coherent");
		}
        if(!StringUtils.isNotBlank(zc.getZenodoid())){
            throw new CommunityException("parameter zenodoid cannot be null or empty");
        }
		final TreeMap<Integer, CommunityZenodoCommunity> zcs = getZenodoCommunityMap(id);

        for(CommunityZenodoCommunity czc : zcs.values()){
        	if (czc.getZenodoid().equals(zc.getZenodoid())){
        		throw new CommunityException("Zenodo community already associated to the RCD");
			}
		}

		zc.setId(nextId(MapUtils.isNotEmpty(zcs) ? zcs.lastKey() : 0));

		isClient.addConcept(id, id + ZENODOCOMMUNITY_ID_SUFFIX, CommunityMappingUtils.asZenodoCommunityXML(id, zc));

		return zc;
	}

	// HELPERS

	private TreeMap<Integer, CommunityProject> getCommunityProjectMap(final String id) throws CommunityException, CommunityNotFoundException {
		return getCommunityProjects(id).stream()
				.collect(Collectors.toMap(
						p -> Integer.valueOf(p.getId()),
						Functions.identity(),
						(p1, p2) -> {
							log.warn(String.format("duplicate project found: '%s'", p1.getId()));
							return p2;
						},
						TreeMap::new));
	}

	private TreeMap<Integer, CommunityContentprovider> getCommunityContentproviderMap(final String id) throws CommunityException, CommunityNotFoundException {
		return getCommunityContentproviders(id).stream()
				.collect(Collectors.toMap(
						cp -> Integer.valueOf(cp.getId()),
						Functions.identity(),
						(cp1, cp2) -> {
							log.warn(String.format("duplicate content provider found: '%s'", cp1.getId()));
							return cp2;
						},
						TreeMap::new));
	}

	private TreeMap<Integer,CommunityZenodoCommunity> getZenodoCommunityMap(final String id) throws CommunityException, CommunityNotFoundException{
		return getCommunityZenodoCommunities(id).stream()
				.collect(Collectors.toMap(
						cp -> Integer.valueOf(cp.getId()),
						Functions.identity(),
						(cp1, cp2) -> {
							log.warn(String.format("duplicate Zenodo community found: '%s'", cp1.getId()));
							return cp2;
						},
						TreeMap::new));
	}

	private Map<String, Context> getContextMap() throws CommunityException {
		try {
			return isClient.getCommunityContextMap();
		} catch (IOException e) {
			throw new CommunityException(e);
		}
	}

	private <R> List<R> _getCommunityInfo(final String id, final String idSuffix, final Function<Concept, R> mapping) throws CommunityException {
		final Map<String, Context> contextMap = getContextMap();
		final Context context = contextMap.get(id);
		if (context != null) {
			final Map<String, Category> categories = context.getCategories();
			final Category category = categories.get(id + idSuffix);
			if (category != null) {
				return category.getConcepts().stream()
						.map(mapping)
						.collect(Collectors.toList());
			}
		}
		return Lists.newArrayList();
	}

}
