package eu.dnetlib.openaire.community;

import com.google.common.collect.Lists;
import eu.dnetlib.openaire.common.ISClient;
import eu.dnetlib.openaire.context.Category;
import eu.dnetlib.openaire.context.Concept;
import eu.dnetlib.openaire.context.Context;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

import static eu.dnetlib.openaire.community.CommunityConstants.ZENODOCOMMUNITY_ID_SUFFIX;
import static eu.dnetlib.openaire.community.CommunityConstants.communityBlackList;

@Component
public class CommunityCommon {

    @Autowired
    private ISClient isClient;

    public Map<String, Context> getContextMap() throws CommunityException {
        try {
            return isClient.getCommunityContextMap();
        } catch (IOException e) {
            throw new CommunityException(e);
        }
    }

    public List<CommunitySummary> listCommunities()  throws CommunityException {
        return getContextMap().values().stream()
                .filter(context -> !communityBlackList.contains(context.getId()))
                .map(CommunityMappingUtils::asCommunitySummary)
                .collect(Collectors.toList());
    }


    public <R> List<R> getCommunityInfo(final String id, final String idSuffix, final Function<Concept, R> mapping) throws CommunityException {
        final Map<String, Context> contextMap = getContextMap();
        final Context context = contextMap.get(id);
        if (context != null) {
            final Map<String, Category> categories = context.getCategories();
            final Category category = categories.get(id + idSuffix);
            if (category != null) {
                return category.getConcepts().stream()
                        .map(mapping)
                        .collect(Collectors.toList());
            }
        }
        return Lists.newArrayList();
    }

    public CommunityDetails getCommunity(final String id) throws CommunityException, CommunityNotFoundException {
        final Context context = getContextMap().get(id);
        if (context == null || CommunityConstants.communityBlackList.contains(id)) {
            throw new CommunityNotFoundException(String.format("community '%s' does not exist", id));
        }
        return CommunityMappingUtils.asCommunityProfile(context);
    }

    public List<CommunityZenodoCommunity> getCommunityZenodoCommunities(final String id) throws CommunityException, CommunityNotFoundException {
        getCommunity(id); // ensure the community exists.
        return getCommunityInfo(id, ZENODOCOMMUNITY_ID_SUFFIX, c -> CommunityMappingUtils.asCommunityZenodoCommunity(id, c));
    }
}
