package eu.dnetlib.openaire.project;

import java.io.BufferedOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.sql.SQLException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Map;
import java.util.zip.ZipOutputStream;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import com.google.common.xml.XmlEscapers;
import eu.dnetlib.OpenaireExporterConfig;
import eu.dnetlib.OpenaireExporterConfig.Project;
import eu.dnetlib.openaire.common.AbstractExporterController;
import eu.dnetlib.openaire.common.ExporterConstants;
import eu.dnetlib.openaire.project.domain.db.ProjectTsv;
import eu.dnetlib.openaire.project.domain.db.ProjectDetails;
import eu.dnetlib.openaire.project.dao.JdbcApiDao;
import eu.dnetlib.openaire.project.dao.ValueCleaner;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import org.antlr.stringtemplate.StringTemplate;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.core.io.Resource;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.CrossOrigin;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;

@Controller
@CrossOrigin(origins = { "*" })
@ConditionalOnProperty(value = "openaire.exporter.enable.project", havingValue = "true")
@io.swagger.annotations.Api(tags = "OpenAIRE projects API", description = "the OpenAIRE projects API")
public class ProjectsController extends AbstractExporterController {

	private static final Log log = LogFactory.getLog(ProjectsController.class); // NOPMD by marko on 11/24/08 5:02 PM

	public final static String UTF8 = "UTF-8";

	@Autowired
	private OpenaireExporterConfig config;

	@Autowired
	private JdbcApiDao dao;

	@Autowired
	private ProjectQueryParamsFactory projectQueryParamsFactory;

	@RequestMapping(value = "/export/**/project/dspace.do", method = RequestMethod.GET)
	@ApiOperation(
			value = "DSpace",
			notes = "return project information in compatible with the OpenAIRE plugin for DSpace",
			tags = { ExporterConstants.DSPACE },
			response = String.class)
	@ApiResponses(value = {
			@ApiResponse(code = 200, message = "OK", response = String.class),
			@ApiResponse(code = 500, message = "unexpected error", response = ErrorMessage.class) })
	public void processDspace(final HttpServletRequest request, final ServletResponse response,
			@RequestParam(value = "startFrom", required = false) final String startFrom,
			@RequestParam(value = "startUntil", required = false) final String startUntil,
			@RequestParam(value = "endFrom", required = false) final String endFrom,
			@RequestParam(value = "endUntil", required = false) final String endUntil) throws Exception {

		final Project conf = config.getProject();

		final ProjectQueryParams params = projectQueryParamsFactory.generateParams(request, startFrom, startUntil, endFrom, endUntil);
		final StringTemplate headSt = new StringTemplate(IOUtils.toString(conf.getDspaceHeadTemplate().getInputStream(), UTF8));

		headSt.setAttribute("fundingProgramme", params.getFundingProgramme());

		final StringTemplate tailSt = new StringTemplate(IOUtils.toString(conf.getDspaceTailTemplate().getInputStream(), UTF8));

		response.setContentType("text/xml");
		doProcess(response, params, headSt.toString(), conf.getDspaceTemplate(), tailSt.toString(), s -> XmlEscapers.xmlContentEscaper().escape(oneLiner(s)));
	}

	@RequestMapping(value = "/export/**/project/eprints.do", method = RequestMethod.GET)
	@ApiOperation(
			value = "EPrints",
			notes = "return project information in compatible with the OpenAIRE plugin for Eprints",
			tags = { ExporterConstants.EPRINT },
			response = String.class)
	@ApiResponses(value = {
			@ApiResponse(code = 200, message = "OK", response = String.class),
			@ApiResponse(code = 500, message = "unexpected error", response = ErrorMessage.class) })
	public void processEprints(final HttpServletRequest request, final ServletResponse response,
			@RequestParam(value = "startFrom", required = false) final String startFrom,
			@RequestParam(value = "startUntil", required = false) final String startUntil,
			@RequestParam(value = "endFrom", required = false) final String endFrom,
			@RequestParam(value = "endUntil", required = false) final String endUntil) throws Exception {

		final ProjectQueryParams params = projectQueryParamsFactory.generateParams(request, startFrom, startUntil, endFrom, endUntil);
		response.setContentType("text/html");
		doProcess(response, params, null, config.getProject().getEprintsTemplate(), null, this::oneLiner);
	}

	private String oneLiner(final String s) {
		return StringUtils.isNotBlank(s) ? s.replaceAll("\\n", " ").trim() : "";
	}

	private void doProcess(
			final ServletResponse response,
			final ProjectQueryParams params,
			final String head, final Resource projectTemplate, final String tail,
			final ValueCleaner cleaner) throws IOException, SQLException {

		final StringTemplate st = new StringTemplate(IOUtils.toString(projectTemplate.getInputStream(), UTF8));
		try(final OutputStream out = new BufferedOutputStream(response.getOutputStream())) {
			dao.streamProjects(obtainQuery(params), out, head, st, tail, cleaner);
		}
	}

	@RequestMapping(value = "/export/project2tsv.do", method = RequestMethod.GET)
	@ApiOperation(
			value = "TSV",
			notes = "download project information in TSV format",
			tags = { ExporterConstants.TSV },
			response = ProjectTsv[].class)
	@ApiResponses(value = {
			@ApiResponse(code = 200, message = "OK", response = ProjectTsv[].class),
			@ApiResponse(code = 500, message = "unexpected error", response = ErrorMessage.class) })
	public void processTsv(final HttpServletResponse response,
			@RequestParam(value = "funding", required = true) final String funding,
			@RequestParam(value = "article293", required = false) final Boolean article293) throws Exception {

		final String fundingPrefix = getFundingPrefix(funding, null);

		final String date = new SimpleDateFormat("yyyyMMdd").format(new Date());
		final String filename = "projects_" + funding + "_" + date + ".tsv";
		response.setContentType("text/tab-separated-values");
		response.setHeader("Content-Disposition", "attachment; filename=\"" + filename + ".zip\"");
		try(final ZipOutputStream out = new ZipOutputStream(new BufferedOutputStream(response.getOutputStream()))) {
			dao.processTsvRequest(out, article293, fundingPrefix, filename);
		} catch (Throwable e) {
			throw new RuntimeException("Error processing the request", e);
		}
	}

	@RequestMapping(value = "/export/streamProjectDetails.do", method = RequestMethod.GET)
	@ApiOperation(
			value = "Stream projects",
			notes = "stream project information",
			tags = { ExporterConstants.STREAMING },
			response = ProjectDetails[].class)
	@ApiResponses(value = {
			@ApiResponse(code = 200, message = "OK", response = ProjectDetails[].class),
			@ApiResponse(code = 500, message = "unexpected error", response = ErrorMessage.class) })
	public void streamProjectDetails(final HttpServletResponse response,
			@RequestParam(value = "format", required = true) final String format,
			@RequestParam(value = "compress", required = false) final Boolean compress) throws IOException, SQLException {

		if (compress != null && compress) {
			response.setHeader("Content-Encoding", "gzip");
		}
		switch (format) {
			case "csv":
				response.setContentType("text/csv");
				break;
			case "json":
				response.setContentType("text/plain");
				break;
			default: throw new IllegalArgumentException("unsupported format: " + format);
		}

		dao.processProjectDetails(response.getOutputStream(), format, compress);
	}

	/**
	 * Creates the query on the fundingProgramme specified in the given parameters.
	 *
	 * @param params
	 *            request parameters
	 * @return the query string
	 * @throws IllegalArgumentException
	 *             if the funding program is not recognized
	 * @throws IOException
	 *             if there are problem loading the query temlate
	 * @throws IllegalArgumentException
	 *             if the funding program is not recognized
	 */
	protected String obtainQuery(final ProjectQueryParams params) throws IllegalArgumentException, IOException {
		String funding = params.getFundingProgramme();
		String suffix = params.getFundingPath();

		final StringTemplate st = new StringTemplate(IOUtils.toString(config.getProject().getProjectsFundingQueryTemplate().getInputStream(), UTF8));
		st.setAttribute("fundingprefix", getFundingPrefix(funding, suffix));
		String theQuery = setDateParameters(st.toString(), params);
		log.debug("Generated query: " + theQuery);
		return theQuery;
	}

	private String getFundingPrefix(final String funding, final String suffix) {
		final Map<String, String> fundingIds = dao.readFundingpathIds();
		if (!fundingIds.containsKey(funding.toUpperCase())) {
			throw new IllegalArgumentException("invalid funding " + funding);
		}
		String fundingPrefix = fundingIds.get(funding.toUpperCase());
		return StringUtils.isBlank(suffix) ? fundingPrefix : fundingPrefix + "::" + suffix.toUpperCase();
	}

	private String setDateParameters(final String query, final ProjectQueryParams params) {
		String queryWithDates = query;
		if (params.getStartFrom() != null) {
			queryWithDates += " AND startdate >= '" + params.getStartFrom() + "'";
		}
		if (params.getStartUntil() != null) {
			queryWithDates += " AND startdate <= '" + params.getStartUntil() + "'";
		}
		if (params.getEndFrom() != null) {
			queryWithDates += " AND enddate >= '" + params.getEndFrom() + "'";
		}
		if (params.getEndUntil() != null) {
			queryWithDates += " AND enddate <= '" + params.getEndUntil() + "'";
		}
		return queryWithDates;
	}

}
