package eu.dnetlib.openaire.info;


import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import eu.dnetlib.openaire.common.AbstractExporterController;
import eu.dnetlib.openaire.common.ExporterConstants;
import io.swagger.annotations.*;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.format.annotation.DateTimeFormat;
import org.springframework.web.bind.annotation.*;
import java.time.LocalDate;
import java.util.List;
import java.util.Map;


@RestController
@CrossOrigin(origins = { "*" })
@ConditionalOnProperty(value = "openaire.exporter.enable.info", havingValue = "true")
@io.swagger.annotations.Api(tags = "OpenAIRE Info API", description = "the OpenAIRE info API")
public class InfoController extends AbstractExporterController {

	private static final Log log = LogFactory.getLog(InfoController.class); // NOPMD by marko on 11/24/08 5:02 PM

	public final static String UTF8 = "UTF-8";

	@Autowired
	private JdbcInfoDao jdbcInfoDao;

	@RequestMapping(value = "/info/{infoKey}", produces = { "application/json" }, method = RequestMethod.GET)
	@ApiOperation(value = "get info date", notes = "get info date", tags = { ExporterConstants.R }, response = LocalDate.class)
	@ApiResponses(value = {
			@ApiResponse(code = 200, message = "OK", response = LocalDate.class),
			@ApiResponse(code = 500, message = "unexpected error", response = ErrorMessage.class) })
	public LocalDate getDate(@PathVariable  final String infoKey){
		JdbcInfoDao.DATE_INFO info = JdbcInfoDao.DATE_INFO.valueOf(infoKey);
		if(info == null) throw new RuntimeException(infoKey + " not recognized");
		return jdbcInfoDao.getDate(info);
	}

	@RequestMapping(value = "/info", produces = { "application/json" }, method = RequestMethod.GET)
	@ApiOperation(value = "get all the info date", notes = "get all the info date", tags = { ExporterConstants.R }, response = Map.class)
	@ApiResponses(value = {
			@ApiResponse(code = 200, message = "OK", response = LocalDate.class),
			@ApiResponse(code = 500, message = "unexpected error", response = ErrorMessage.class) })
	public Map<String, LocalDate> listInfo(){
		Map<String, LocalDate> map = Maps.newHashMap();
		map.put(JdbcInfoDao.DATE_INFO.claim_load_date.name(), jdbcInfoDao.getDate(JdbcInfoDao.DATE_INFO.claim_load_date));
		map.put(JdbcInfoDao.DATE_INFO.inference_date.name(),jdbcInfoDao.getDate(JdbcInfoDao.DATE_INFO.inference_date));
		map.put(JdbcInfoDao.DATE_INFO.oaf_load_date.name(), jdbcInfoDao.getDate(JdbcInfoDao.DATE_INFO.oaf_load_date));
		map.put(JdbcInfoDao.DATE_INFO.odf_load_date.name(), jdbcInfoDao.getDate(JdbcInfoDao.DATE_INFO.odf_load_date));
		return map;
	}

	@RequestMapping(value = "/info/keys", produces = { "application/json" }, method = RequestMethod.GET)
	@ApiOperation(value = "get the available keys", notes = "get the available keys", tags = { ExporterConstants.R },
			response = String.class, responseContainer = "List")
	@ApiResponses(value = {
			@ApiResponse(code = 200, message = "OK", response = LocalDate.class),
			@ApiResponse(code = 500, message = "unexpected error", response = ErrorMessage.class) })
	public List<String> listInfoKeys(){
		List<String> keys = Lists.newArrayList(
				JdbcInfoDao.DATE_INFO.claim_load_date.name(),
				JdbcInfoDao.DATE_INFO.inference_date.name(),
				JdbcInfoDao.DATE_INFO.oaf_load_date.name(),
				JdbcInfoDao.DATE_INFO.odf_load_date.name()
		);
		return keys;
	}


	@RequestMapping(value = "/info/dropCache", produces = { "application/json" }, method = RequestMethod.GET)
	@ApiOperation(value = "Drops the info cache", notes = "Drops the info cache", tags = { ExporterConstants.R })
	public void dropCache(){
		jdbcInfoDao.dropCache();
	}



}
