package eu.dnetlib.openaire.context;

import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import com.google.common.base.Splitter;
import eu.dnetlib.openaire.common.ISClient;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.stereotype.Component;

@Component
@ConditionalOnProperty(value = "openaire.exporter.enable.context", havingValue = "true")
public class ContextApiCore {

	private static final Log log = LogFactory.getLog(ContextApiCore.class);
	private static final String SEPARATOR = "::";

	@Autowired
	private ISClient isClient;

	public List<ContextSummary> listContexts() throws ContextException {
		return getContextMap().values().stream()
				.map(c -> new ContextSummary()
						.setId(c.getId())
						.setType(c.getType())
						.setLabel(c.getLabel())
						.setStatus(c.getParams().containsKey("status") ? c.getParams().get("status").get(0).getValue() : ""))
				.collect(Collectors.toList());
	}

	public List<CategorySummary> listCategories(final String contextId) throws ContextException {
		return getContextMap().get(contextId).getCategories().values().stream()
				.filter(Category::isClaim)
				.map(c -> new CategorySummary()
						.setId(c.getId())
						.setLabel(c.getLabel())
						.setHasConcept(c.hasConcepts()))
				.collect(Collectors.toList());
	}

	public List<ConceptSummary> listConcepts(final String categoryId) throws ContextException {
		final String contextId = StringUtils.substringBefore(categoryId, SEPARATOR);
		return getContextMap().get(contextId).getCategories().get(categoryId).getConcepts().stream()
				.filter(Concept::isClaim)
				.map(c -> new ConceptSummary()
						.setId(c.getId())
						.setLabel(c.getLabel())
						.setHasSubConcept(c.hasSubConcepts()))
				.collect(Collectors.toList());

	}

	public List<ConceptSummary> listSubConcepts(final String conceptId) throws ContextException {
		final List<String> ids = Splitter.on(SEPARATOR).splitToList(conceptId);
		if (ids.size() < 3) {
			throw new ContextException("");
		}

		final String contextId = ids.get(0);
		final String categoryId = contextId + SEPARATOR + ids.get(1);

		final List<Concept> concepts = getContextMap().get(contextId).getCategories().get(categoryId).getConcepts()
				.stream()
				.filter(c -> conceptId.equals(c.getId()))
				.filter(Concept::isClaim)
				.collect(Collectors.toList());

		return mapConcepts(concepts);
	}

	private List<ConceptSummary> mapConcepts(final List<Concept> concepts) {
		if (concepts == null || concepts.isEmpty()) {
			return null;
		}
		return concepts.stream()
				.map(c -> new ConceptSummary()
						.setId(c.getId())
						.setLabel(c.getLabel())
						.setHasSubConcept(c.hasSubConcepts())
						.setConcept(mapConcepts(c.getConcepts())))
				.collect(Collectors.toList());
	}

	private Map<String, Context> getContextMap() throws ContextException {
		try {
			return isClient.getContextMap();
		} catch (IOException e) {
			throw new ContextException(e);
		}
	}


}
