package eu.dnetlib;

import javax.sql.DataSource;

import com.google.common.collect.Lists;
import com.mongodb.MongoClient;
import com.mongodb.MongoClientOptions;
import com.mongodb.ServerAddress;
import eu.dnetlib.OpenaireExporterConfig.Jdbc;
import eu.dnetlib.data.objectstore.rmi.ObjectStoreService;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpService;
import eu.dnetlib.enabling.is.registry.rmi.ISRegistryService;
import io.micrometer.core.instrument.ImmutableTag;
import io.micrometer.core.instrument.Metrics;
import io.micrometer.core.instrument.Tag;
import io.micrometer.prometheus.PrometheusMeterRegistry;
import io.prometheus.client.Gauge;
import org.apache.commons.dbcp2.BasicDataSource;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.cxf.jaxws.JaxWsProxyFactoryBean;
import org.apache.maven.model.Model;
import org.apache.maven.model.io.xpp3.MavenXpp3Reader;
import org.codehaus.plexus.util.xml.pull.XmlPullParserException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.actuate.autoconfigure.metrics.MeterRegistryCustomizer;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;

/**
 * Created by claudio on 07/07/2017.
 */
@Configuration
public class DNetOpenaireExporterBeanFactory {

	private static final Log log = LogFactory.getLog(DNetOpenaireExporterBeanFactory.class);

	private static final String[] POM_PATHS = {
			"/META-INF/maven/eu.dnetlib/dnet-openaire-exporter/pom.xml",
			"/maven/eu.dnetlib/dnet-openaire-exporter/pom.xml",
			"/eu.dnetlib/dnet-openaire-exporter/pom.xml",
			"/dnet-openaire-exporter/pom.xml",
			"/pom.xml",
			"META-INF/maven/eu.dnetlib/dnet-openaire-exporter/pom.xml",
			"maven/eu.dnetlib/dnet-openaire-exporter/pom.xml",
			"eu.dnetlib/dnet-openaire-exporter/pom.xml",
			"dnet-openaire-exporter/pom.xml",
			"pom.xml"
	};

	static {
		final MavenXpp3Reader reader = new MavenXpp3Reader();
		FileReader f = null;
		for(final String path : POM_PATHS) {
			try {
				f = new FileReader(path);
				log.info("found pom file in " + path);
			} catch (FileNotFoundException e) {
				log.warn("cannot find pom file in " + path);
			}
		}

		if (f == null) {
			throw new IllegalStateException("unable to get pom information");
		}
		try {
			final Model model = reader.read(f);

			log.info(String.format("registering metric for %s", model.getArtifactId()));
			Metrics.gauge("micrometer_info", Lists.newArrayList(
					new ImmutableTag("component", model.getGroupId()+":"+model.getArtifactId()),
					new ImmutableTag("version", model.getVersion()),
					new ImmutableTag("scmtag", model.getScm().getTag())), 1);
		} catch (IOException | XmlPullParserException e) {
			log.error(e);
		}
	}

	@Autowired
	private OpenaireExporterConfig config;

	@Bean
	public ISLookUpService getLookUpService() {
		return getServiceStub(ISLookUpService.class, config.getIsLookupUrl());
	}

	@Bean
	public ObjectStoreService getObjectStoreService() {
		return getServiceStub(ObjectStoreService.class, config.getObjectStoreServiceUrl());
	}

	@Bean
	public ISRegistryService getRegistryService() {
		return getServiceStub(ISRegistryService.class, config.getIsRegistryServiceUrl());
	}

	@SuppressWarnings("unchecked")
	private <T> T getServiceStub(final Class<T> clazz, final String endpoint) {
		log.info(String.format("Initializing service stub %s, endpoint %s", clazz.toString(),endpoint));
		final JaxWsProxyFactoryBean jaxWsProxyFactory = new JaxWsProxyFactoryBean();
		jaxWsProxyFactory.setServiceClass(clazz);
		jaxWsProxyFactory.setAddress(endpoint);
		return (T) jaxWsProxyFactory.create();
	}

	@Bean
	public DataSource getSqlDataSource() {
		final Jdbc jdbc = config.getJdbc();
		return getDatasource(
				jdbc.getDriverClassName(),
				jdbc.getUrl(),
				jdbc.getUser(),
				jdbc.getPwd(),
				jdbc.getMinIdle(),
				jdbc.getMaxRows());
	}

	private BasicDataSource getDatasource(String driverClassName, String jdbcUrl, String jdbcUser, String jdbcPwd, int jdbcMinIdle, int jdbcMaxIdle) {
		final BasicDataSource d = new BasicDataSource();
		d.setDriverClassName(driverClassName);
		d.setUrl(jdbcUrl);
		d.setUsername(jdbcUser);
		d.setPassword(jdbcPwd);
		d.setMinIdle(jdbcMinIdle);
		d.setMaxIdle(jdbcMaxIdle);
		return d;
	}

	@Bean
	public MongoClient getMongoClient() {
		return new MongoClient(
				new ServerAddress(config.getDatasource().getMongoHost(), config.getDatasource().getMongoPort()),
				MongoClientOptions.builder().connectionsPerHost(config.getDatasource().getMongoConnectionsPerHost()).build());
	}

}
