package eu.dnetlib.openaire.community;

import java.util.List;

import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.web.bind.annotation.*;

import static eu.dnetlib.openaire.common.ExporterConstants.*;

@RestController
@CrossOrigin(origins = { "*" })
@ConditionalOnProperty(value = "openaire.exporter.enable.community", havingValue = "true")
@io.swagger.annotations.Api(tags = "OpenAIRE Communities API", description = "the OpenAIRE Community API")
public class CommunityApiController {

	@Autowired
	private CommunityApiCore communityApiCore;


	@RequestMapping(value = "/community/communities", produces = { "application/json" }, method = RequestMethod.GET)
	@ApiOperation(
			value = "get all community profiles",
			notes = "get all community profiles",
			tags = { C, R },
			response = CommunitySummary[].class)
	@ApiResponses(value = {
			@ApiResponse(code = 200, message = "OK", response = CommunitySummary[].class),
			@ApiResponse(code = 500, message = "unexpected error", response = CommunityException.class) })
	public List<CommunitySummary> listCommunities() throws CommunityException {
		return communityApiCore.listCommunities();
	}

	@RequestMapping(value = "/community/{id}", produces = { "application/json" }, method = RequestMethod.GET)
	@ApiOperation(
			value = "get community profile",
			notes = "get community profile",
			tags = { C, R },
			response = CommunityDetails.class)
	@ApiResponses(value = {
			@ApiResponse(code = 200, message = "OK", response = CommunityDetails.class),
			@ApiResponse(code = 404, message = "not found", response = CommunityNotFoundException.class),
			@ApiResponse(code = 500, message = "unexpected error", response = CommunityException.class) })
	public CommunityDetails getCommunity(@PathVariable final String id) throws CommunityException, CommunityNotFoundException {
		return communityApiCore.getCommunity(id);
	}

	@RequestMapping(value = "/community/{id}", produces = { "application/json" }, method = RequestMethod.POST)
	@ApiOperation(
			value = "update community details",
			notes = "update community details",
			tags = { C, R })
	@ApiResponses(value = {
			@ApiResponse(code = 200, message = "OK"),
			@ApiResponse(code = 404, message = "not found", response = CommunityNotFoundException.class),
			@ApiResponse(code = 500, message = "unexpected error", response = CommunityException.class) })
	public void setCommunity(
			@PathVariable final String id,
			@RequestBody CommunityWritableProperties properties) throws CommunityException, CommunityNotFoundException {

		communityApiCore.setCommunity(id, properties);
	}

	@RequestMapping(value = "/community/{id}/projects", produces = { "application/json" }, method = RequestMethod.GET)
	@ApiOperation(
			value = "get community projects",
			notes = "get community projects",
			tags = { C_PJ, R },
			response = CommunityProject[].class)
	@ApiResponses(value = {
			@ApiResponse(code = 200, message = "OK", response = CommunityProject[].class),
			@ApiResponse(code = 404, message = "not found", response = CommunityNotFoundException.class),
			@ApiResponse(code = 500, message = "unexpected error", response = CommunityException.class) })
	public List<CommunityProject> getCommunityProjects(@PathVariable final String id) throws CommunityException, CommunityNotFoundException {
		return communityApiCore.getCommunityProjects(id);
	}

	@RequestMapping(value = "/community/{id}/projects", produces = { "application/json" }, method = RequestMethod.POST)
	@ApiOperation(
			value = "associate a project to the community",
			notes = "associate a project to the community",
			tags = { C_PJ, W })
	@ApiResponses(value = {
			@ApiResponse(code = 200, message = "OK"),
			@ApiResponse(code = 404, message = "not found", response = CommunityNotFoundException.class),
			@ApiResponse(code = 500, message = "unexpected error", response = CommunityException.class) })
	public CommunityProject addCommunityProject(
			@PathVariable final String id,
			@RequestBody final CommunityProject project) throws CommunityException, CommunityNotFoundException {

		return communityApiCore.addCommunityProject(id, project);
	}

	@RequestMapping(value = "/community/{id}/projects", produces = { "application/json" }, method = RequestMethod.DELETE)
	@ApiOperation(
			value = "remove a project from the community",
			notes = "remove a project from the community",
			tags = { C_PJ, W })
	@ApiResponses(value = {
			@ApiResponse(code = 200, message = "OK"),
			@ApiResponse(code = 404, message = "not found", response = CommunityNotFoundException.class),
			@ApiResponse(code = 500, message = "unexpected error", response = CommunityException.class) })
	public void deleteCommunityProject(
			@PathVariable final String id,
			@RequestBody final Integer projectId) throws CommunityException, CommunityNotFoundException {

		communityApiCore.removeCommunityProject(id, projectId);
	}

	@RequestMapping(value = "/community/{id}/contentproviders", produces = { "application/json" }, method = RequestMethod.GET)
	@ApiOperation(
			value = "get the list of content providers associated to a given community",
			notes = "get the list of content providers associated to a given community",
			tags = { C_CP, R },
			response = CommunityContentprovider[].class)
	@ApiResponses(value = {
			@ApiResponse(code = 200, message = "OK", response = CommunityContentprovider[].class),
			@ApiResponse(code = 404, message = "not found", response = CommunityNotFoundException.class),
			@ApiResponse(code = 500, message = "unexpected error", response = CommunityException.class) })
	public List<CommunityContentprovider> getCommunityContentproviders(@PathVariable final String id) throws CommunityException, CommunityNotFoundException {
		return communityApiCore.getCommunityContentproviders(id);
	}

	@RequestMapping(value = "/community/{id}/contentproviders", produces = { "application/json" }, method = RequestMethod.POST)
	@ApiOperation(
			value = "associate a content provider to the community",
			notes = "associate a content provider to the community",
			tags = { C_CP, W })
	@ApiResponses(value = {
			@ApiResponse(code = 200, message = "OK"),
			@ApiResponse(code = 404, message = "not found", response = CommunityNotFoundException.class),
			@ApiResponse(code = 500, message = "unexpected error", response = CommunityException.class) })
	public CommunityContentprovider addCommunityContentprovider(
			@PathVariable final String id,
			@RequestBody final CommunityContentprovider contentprovider) throws CommunityException, CommunityNotFoundException {

		return communityApiCore.addCommunityContentprovider(id, contentprovider);
	}

	@RequestMapping(value = "/community/{id}/contentproviders", produces = { "application/json" }, method = RequestMethod.DELETE)
	@ApiOperation(
			value = "remove the association between a content provider and the community",
			notes = "remove the association between a content provider and the community",
			tags = { C_CP, W })
	@ApiResponses(value = {
			@ApiResponse(code = 200, message = "OK"),
			@ApiResponse(code = 404, message = "not found", response = CommunityNotFoundException.class),
			@ApiResponse(code = 500, message = "unexpected error", response = CommunityException.class) })
	public void removeCommunityContentprovider(
			@PathVariable final String id,
			@RequestBody final Integer contentproviderId) throws CommunityException, CommunityNotFoundException {

		communityApiCore.removeCommunityContentProvider(id, contentproviderId);
	}

	//ADDING CODE FOR COMMUNITY ORGANIZATIONS

	@RequestMapping(value = "/community/{id}/organizations", produces = { "application/json" }, method = RequestMethod.GET)
	@ApiOperation(
			value = "get the list of organizations for a given community",
			notes = "get the list of organizations for a given community",
			tags = { C_O, R },
			response = CommunityOrganization[].class)
	@ApiResponses(value = {
			@ApiResponse(code = 200, message = "OK", response = CommunityContentprovider[].class),
			@ApiResponse(code = 404, message = "not found", response = CommunityNotFoundException.class),
			@ApiResponse(code = 500, message = "unexpected error", response = CommunityException.class) })
	public List<CommunityOrganization> getCommunityOrganizations(@PathVariable final String id) throws CommunityException, CommunityNotFoundException {
		return communityApiCore.getCommunityOrganizations(id);
	}

	@RequestMapping(value = "/community/{id}/organizations", produces = { "application/json" }, method = RequestMethod.POST)
	@ApiOperation(
			value = "associate an organization to the community",
			notes = "associate an organization to the community",
			tags = { C_O, W })
	@ApiResponses(value = {
			@ApiResponse(code = 200, message = "OK"),
			@ApiResponse(code = 404, message = "not found", response = CommunityNotFoundException.class),
			@ApiResponse(code = 500, message = "unexpected error", response = CommunityException.class) })
	public CommunityOrganization addCommunityOrganization(
			@PathVariable final String id,
			@RequestBody final CommunityOrganization organization) throws CommunityException, CommunityNotFoundException {

		return communityApiCore.addCommunityOrganization(id, organization);
	}

	@RequestMapping(value = "/community/{id}/organizations", produces = { "application/json" }, method = RequestMethod.DELETE)
	@ApiOperation(
			value = "remove the association between an organization and the community",
			notes = "remove the association between an organization and the community",
			tags = { C_O, W })
	@ApiResponses(value = {
			@ApiResponse(code = 200, message = "OK"),
			@ApiResponse(code = 404, message = "not found", response = CommunityNotFoundException.class),
			@ApiResponse(code = 500, message = "unexpected error", response = CommunityException.class) })
	public void removeCommunityOrganization(
			@PathVariable final String id,
			@RequestBody final Integer organizationId) throws CommunityException, CommunityNotFoundException {

		communityApiCore.removeCommunityOrganization(id, organizationId);
	}
	//**********************

	@RequestMapping(value = "/community/{id}/subjects", produces = { "application/json" }, method = RequestMethod.POST)
	@ApiOperation(
			value = "associate a subject to the community",
			notes = "associate a subject to the community",
			tags = { C, W })
	@ApiResponses(value = {
			@ApiResponse(code = 200, message = "OK"),
			@ApiResponse(code = 404, message = "not found", response = CommunityNotFoundException.class),
			@ApiResponse(code = 500, message = "unexpected error", response = CommunityException.class) })
	public CommunityDetails addCommunitySubjects(
			@PathVariable final String id,
			@RequestBody final List<String> subjects) throws CommunityException, CommunityNotFoundException {

		return communityApiCore.addCommunitySubjects(id, subjects);
	}

	@RequestMapping(value = "/community/{id}/subjects", produces = { "application/json" }, method = RequestMethod.DELETE)
	@ApiOperation(
			value = "remove subjects from a community",
			notes = "remove subjects from a community",
			tags = { C, W })
	@ApiResponses(value = {
			@ApiResponse(code = 200, message = "OK"),
			@ApiResponse(code = 404, message = "not found", response = CommunityNotFoundException.class),
			@ApiResponse(code = 500, message = "unexpected error", response = CommunityException.class) })
	public CommunityDetails removeCommunitySubjects(
			@PathVariable final String id,
			@RequestBody final List<String> subjects) throws CommunityException, CommunityNotFoundException {

		return communityApiCore.removeCommunitySubjects(id, subjects);
	}

	@RequestMapping(value = "/community/{id}/zenodocommunities", produces = { "application/json" }, method = RequestMethod.GET)
	@ApiOperation(
			value = "get the list of Zenodo communities associated to a given community",
			notes = "get the list of Zenodo communities associated to a given community",
			tags = { C_ZC, R },
			response = CommunityZenodoCommunity[].class)
	@ApiResponses(value = {
			@ApiResponse(code = 200, message = "OK", response = CommunityZenodoCommunity[].class),
			@ApiResponse(code = 404, message = "not found", response = CommunityNotFoundException.class),
			@ApiResponse(code = 500, message = "unexpected error", response = CommunityException.class) })
	public List<CommunityZenodoCommunity> getCommunityZenodoCommunities(@PathVariable final String id) throws CommunityException, CommunityNotFoundException {
		return communityApiCore.getCommunityZenodoCommunities(id);
	}

	@RequestMapping(value = "/community/{id}/zenodocommunities", produces = { "application/json" }, method = RequestMethod.POST)
	@ApiOperation(
			value = "associate a Zenodo community to the community",
			notes = "associate a Zenodo community to the community",
			tags = { C_ZC, W })
	@ApiResponses(value = {
			@ApiResponse(code = 200, message = "OK"),
			@ApiResponse(code = 404, message = "not found", response = CommunityNotFoundException.class),
			@ApiResponse(code = 500, message = "unexpected error", response = CommunityException.class) })
	public CommunityZenodoCommunity addCommunityZenodoCommunity(
			@PathVariable final String id,
			@RequestBody final CommunityZenodoCommunity zenodocommunity) throws CommunityException, CommunityNotFoundException {

		return communityApiCore.addCommunityZenodoCommunity(id, zenodocommunity);

	}

	@RequestMapping(value = "/community/{id}/zenodocommunities", produces = { "application/json" }, method = RequestMethod.DELETE)
	@ApiOperation(
			value = "remove a Zenodo community from a community",
			notes = "remove a Zenodo community from a community",
			tags = { C_ZC, W })
	@ApiResponses(value = {
			@ApiResponse(code = 200, message = "OK"),
			@ApiResponse(code = 404, message = "not found", response = CommunityNotFoundException.class),
			@ApiResponse(code = 500, message = "unexpected error", response = CommunityException.class) })
	public void removeCommunityZenodoCommunity(
			@PathVariable final String id,
			@RequestBody final Integer  zenodoCommId) throws CommunityException, CommunityNotFoundException {

		communityApiCore.removeCommunityZenodoCommunity(id, zenodoCommId);

	}


	@RequestMapping(value = "/community/{zenodoId}/openairecommunities", produces = { "application/json" }, method = RequestMethod.GET)
	@ApiOperation(
			value = "get the list of OpenAIRE communities associated to a given Zenodo community",
			notes = "get the list of OpenAIRE communities associated to a given Zenodo community",
			tags = { C_ZC, R })
	@ApiResponses(value = {
			@ApiResponse(code = 200, message = "OK"),
			@ApiResponse(code = 404, message = "not found", response = CommunityNotFoundException.class),
			@ApiResponse(code = 500, message = "unexpected error", response = CommunityException.class) })
	public CommunityOpenAIRECommunities getOpenAireCommunities(
			@PathVariable final String zenodoId) throws CommunityException, CommunityNotFoundException {

		return communityApiCore.getOpenAIRECommunities(zenodoId);

	}
}
