package eu.dnetlib.data.transform.xml.vtd;

import java.nio.charset.Charset;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.google.common.collect.Maps;
import com.google.common.collect.Streams;
import com.google.protobuf.Descriptors;
import com.google.protobuf.Descriptors.Descriptor;
import com.google.protobuf.InvalidProtocolBufferException;
import com.google.protobuf.Message;
import com.ximpleware.AutoPilot;
import com.ximpleware.VTDGen;
import com.ximpleware.VTDNav;
import eu.dnetlib.data.proto.FieldTypeProtos;
import eu.dnetlib.data.proto.FieldTypeProtos.*;
import eu.dnetlib.data.proto.FieldTypeProtos.OAIProvenance.OriginDescription;
import eu.dnetlib.data.proto.KindProtos.Kind;
import eu.dnetlib.data.proto.OafProtos.Oaf;
import eu.dnetlib.data.proto.OafProtos.OafEntity;
import eu.dnetlib.data.proto.OafProtos.OafRel;
import eu.dnetlib.data.proto.RelTypeProtos.RelType;
import eu.dnetlib.data.proto.RelTypeProtos.SubRelType;
import eu.dnetlib.data.proto.ResultProtos.Result;
import eu.dnetlib.data.proto.ResultProtos.Result.*;
import eu.dnetlib.data.proto.TypeProtos.Type;
import eu.dnetlib.data.transform.xml.AbstractDNetXsltFunctions;
import eu.dnetlib.miscutils.collections.Pair;
import eu.dnetlib.pace.model.Person;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import static eu.dnetlib.data.transform.xml.AbstractDNetXsltFunctions.oafSimpleId;
import static eu.dnetlib.data.transform.xml.AbstractDNetXsltFunctions.oafSplitId;
import static eu.dnetlib.data.transform.xml.vtd.VtdUtilityParser.*;
import static java.lang.String.format;

public abstract class AbstractResultVtdParser implements Function<String, Oaf> {

	private static final Log log = LogFactory.getLog(AbstractResultVtdParser.class);

	public static final String URL_REGEX = "^(http|https|ftp)\\://.*";

	public static final String ID_SEPARATOR = "::";

	public static final String TITLE_TYPE = "titleType";
	public static final String DATE_TYPE = "dateType";
	public static final String KEYWORD = "keyword";

	public static final String DNET_EXT_REF_TYPOLOGIES = "dnet:externalReference_typologies";
	public static final String DNET_TITLE_TYPOLOGIES = "dnet:dataCite_title";
	public static final String DNET_SUBJECT_TYPOLOGIES = "dnet:subject_classification_typologies";
	public static final String DNET_RESULT_TYPOLOGIES = "dnet:result_typologies";
	public static final String DNET_PUBLICATION_RESOURCE = "dnet:publication_resource";
	public static final String DNET_DATA_CITE_RESOURCE = "dnet:dataCite_resource";
	public static final String DNET_ACCESS_MODES = "dnet:access_modes";
	public static final String DNET_LANGUAGES = "dnet:languages";
	public static final String DNET_PID_TYPES = "dnet:pid_types";

	public static final String IDENTIFIER_TYPE = "identifierType";
	public static final String ALTERNATE_IDENTIFIER_TYPE = "alternateIdentifierType";
	public static final String DNET_PROVENANCE_ACTIONS = "dnet:provenanceActions";

	public static final String CLASSID = "classid";
	public static final String CLASSNAME = "classname";
	public static final String SCHEMEID = "schemeid";
	public static final String SCHEMENAME = "schemename";

	public static final String RELATION_TYPE = "relationType";
	public static final String RELATED_IDENTIFIER_TYPE = "relatedIdentifierType";
	public static final String RIGHTS_URI = "rightsURI";

	public static final String UTF_8 = "UTF-8";

	// publication
	public static final String PROJECTID = "projectid";
	public static final String RELATED_DATASET = "relateddataset";
	public static final String RELATED_PUBLICATION = "relatedpublication";
	public static final String RELATED_IDENTIFIER = "relatedidentifier";

	protected static Map<String, String> mappingAccess = Maps.newHashMap();

	static {
		mappingAccess.put("info:eu-repo/semantics/openAccess", "OPEN");
		mappingAccess.put("info:eu-repo/semantics/closedAccess", "CLOSED");
		mappingAccess.put("info:eu-repo/semantics/restrictedAccess", "RESTRICTED");
		mappingAccess.put("info:eu-repo/semantics/embargoedAccess", "EMBARGO");

		// Transformator now maps the access rights into proper values, not sure if it does for all datasets.
		mappingAccess.put("OPEN", "OPEN");
		mappingAccess.put("CLOSED", "CLOSED");
		mappingAccess.put("RESTRICTED", "RESTRICTED");
		mappingAccess.put("EMBARGO", "EMBARGO");
	}

	protected boolean invisible = false;
	protected String provenance = "";
	protected String trust = "0.9";

	public AbstractResultVtdParser() {}

	public AbstractResultVtdParser(final boolean invisible, final String provenance, final String trust) {
		this.invisible = invisible;
		this.provenance = provenance;
		this.trust = trust;
	}

	@Override
	public Oaf apply(final String xml) {
		try {
			final VTDGen vg = parseXml(xml);
			final VTDNav vn = vg.getNav();
			final AutoPilot ap = new AutoPilot(vn);

			final boolean skiprecord = Boolean.valueOf(getFirstValue(ap, vn, xpath("record", "header", "skipRecord")));
			int metadata = countNodes(ap, vn, format("count(%s)", xpath("record", "metadata")));

			if (metadata == 0 || skiprecord) {
				return null;
			}

			final String objIdentifier = oafSimpleId(Type.result.name(), getFirstValue(ap, vn, xpath("record", "header", "objIdentifier")));
			if (StringUtils.isBlank(objIdentifier)) {
				return null;
			}

			return transform(ap, vn, objIdentifier, getFields());
		} catch (Throwable e) {
			log.error(e.getMessage());
			log.error(ExceptionUtils.getStackTrace(e));
			return null;
		}
	}

	protected Oaf transform(final AutoPilot ap, final VTDNav vn, final String objIdentifier, final Map<String, String> fields) throws VtdException {

		final SpecificationMap specs = new SpecificationMap();
		specs.put(Result.getDescriptor(), SpecificationDescriptor.newInstance())
				.setBuilder(Result.newBuilder())
				.put("externalReference", fields.get("externalReference"), nodes -> nodes.stream()
					.map(node -> {
						final ExternalReference.Builder extref = Result.ExternalReference.newBuilder();
						if (StringUtils.isNotBlank(node.getTextValue())) {
							extref.setUrl(node.getTextValue());
						}
						final Map<String, String> a = node.getAttributes();
						final String source = a.get("source");
						if (StringUtils.isNotBlank(source)) {
							extref.setSitename(source);
						}
						final String identifier = a.get("identifier");
						if (StringUtils.isNotBlank(identifier)) {
							extref.setRefidentifier(identifier);
						}
						final String title = a.get("title");
						if (StringUtils.isNotBlank(title)) {
							extref.setLabel(title);
						}
						final String query = a.get("query");
						if (StringUtils.isNotBlank(query)) {
							extref.setQuery(query);
						}
						final String type = a.get("type");
						if (StringUtils.isNotBlank(type)) {
							extref.setQualifier(getSimpleQualifier(type, DNET_EXT_REF_TYPOLOGIES));
						}
						return extref.build();
					}));

		specs.put(Instance.getDescriptor(), SpecificationDescriptor.newInstance())
				.setBuilder(Instance.newBuilder())
				.put("license", fields.get("license"), nodes -> nodes.stream()
						.filter(node -> {
							final Map<String, String> a = node.getAttributes();
							switch (node.getName()) {
							case "rights":
								return a.containsKey(RIGHTS_URI) && a.get(RIGHTS_URI).matches(URL_REGEX);
							case "license":
								return true;
							default:
								return false;
							}
						})
						.map(Node::getTextValue))
				.put("accessright", fields.get("accessright"), nodes -> nodes.stream()
						.map(Node::getTextValue)
						.map(rights -> mappingAccess.containsKey(rights) ? mappingAccess.get(rights) : "UNKNOWN")
						.map(code -> getQualifier(code, getClassName(code), DNET_ACCESS_MODES, DNET_ACCESS_MODES)))
				.put("instancetype", fields.get("instancetype"), nodes -> nodes.stream()
						.map(Node::getTextValue)
						.map(code -> getQualifier(code, getClassName(code), DNET_PUBLICATION_RESOURCE, DNET_PUBLICATION_RESOURCE)))
				.put("hostedby", fields.get("hostedby"), nodes -> nodes.stream()
						.map(node -> getKV(oafSplitId("datasource", node.getAttributes().get("id")), node.getAttributes().get("name"))))
				.put("url", fields.get("url"), nodes -> nodes.stream()
						.map(Node::getTextValue)
						.filter(s -> s.trim().matches(URL_REGEX)))
				.put("dateofacceptance", fields.get("dateofacceptance"), nodes -> nodes.stream()
						.map(Node::getTextValue));

		specs.put(Metadata.getDescriptor(), SpecificationDescriptor.newInstance())
				.setBuilder(Metadata.newBuilder())
				.put("title", fields.get("title"), nodes -> nodes.stream()
						.map(node -> {
							final Qualifier.Builder q = Qualifier.newBuilder().setSchemeid(DNET_TITLE_TYPOLOGIES).setSchemename(DNET_TITLE_TYPOLOGIES);
							switch (node.getAttributes().get(TITLE_TYPE) + "") {
							case "AlternativeTitle":
								q.setClassid("alternative title").setClassname("alternative title");
								break;
							case "Subtitle":
								q.setClassid("subtitle").setClassname("subtitle");
								break;
							case "TranslatedTitle":
								q.setClassid("translated title").setClassname("translated title");
								break;
							default:
								q.setClassid("main title").setClassname("main title");
								break;
							}
							return StructuredProperty.newBuilder().setValue(node.getTextValue()).setQualifier(q).build();
						}))
				.put("description", fields.get("description"), nodes -> nodes.stream()
						.map(Node::getTextValue))
				.put("storagedate", fields.get("storagedate"), nodes -> nodes.stream()
						.map(Node::getTextValue))
				.put("lastmetadataupdate", fields.get("lastmetadataupdate"), nodes -> nodes.stream()
						.map(Node::getTextValue))
				.put("embargoenddate", fields.get("embargoenddate"), nodes -> nodes.stream()
						.map(Node::getTextValue))
				.put("dateofacceptance", fields.get("dateofacceptance"), nodes -> nodes.stream()
						.map(Node::getTextValue))
				.put("author", fields.get("author"), nodes -> Streams.mapWithIndex(
						nodes.stream()
								.map(Node::getTextValue),
						(creator, i) -> new Pair<>(i, creator))
						.map(pair -> {
							final Author.Builder author = Author.newBuilder();
							author.setFullname(pair.getValue());
							author.setRank(pair.getKey().intValue() + 1);
							final Person p = new Person(pair.getValue(), false);
							if (p.isAccurate()) {
								author.setName(p.getNormalisedFirstName());
								author.setSurname(p.getNormalisedSurname());
							}
							return author.build();
						}))
				.put("contributor", fields.get("contributor"), nodes -> nodes.stream()
						.map(Node::getTextValue))
				.put("subject", fields.get("subject"), nodes -> nodes.stream()
						.map(node -> {
							final Map<String, String> a = node.getAttributes();
							final String classId = StringUtils.isNotBlank(a.get(CLASSID)) ? a.get(CLASSID) : KEYWORD;
							final String className = StringUtils.isNotBlank(a.get(CLASSNAME)) ? a.get(CLASSNAME) : KEYWORD;
							final String schemeId = StringUtils.isNotBlank(a.get(SCHEMEID)) ? a.get(SCHEMEID) : DNET_SUBJECT_TYPOLOGIES;
							final String schemeName = StringUtils.isNotBlank(a.get(SCHEMENAME)) ? a.get(SCHEMENAME) : DNET_SUBJECT_TYPOLOGIES;
							return getStructuredProperty(node.getTextValue(), classId, className, schemeId, schemeName);
						}))
				.put("format", fields.get("format"), nodes -> nodes.stream()
						.map(Node::getTextValue))
				.put("source", fields.get("source"), nodes -> nodes.stream()
						.map(Node::getTextValue))
				.put("size", fields.get("size"), nodes -> nodes.stream()
						.map(Node::getTextValue))
				.put("version", fields.get("version"), nodes -> nodes.stream()
						.map(Node::getTextValue))
				.put("publisher", fields.get("publisher"), nodes -> nodes.stream()
						.map(Node::getTextValue))
				.put("language", fields.get("language"), nodes -> nodes.stream()
						.map(Node::getTextValue)
						.map(code -> getQualifier(code, getClassName(code), DNET_LANGUAGES, DNET_LANGUAGES)))
				.put("resourcetype", fields.get("resourcetype"), nodes -> nodes.stream()
						.map(node -> node.getAttributes().get("resourceTypeGeneral"))
						.map(resourceType -> getSimpleQualifier(resourceType, DNET_DATA_CITE_RESOURCE)))
				.put("resulttype", fields.get("resulttype"), nodes -> nodes.stream()
						.map(Node::getTextValue)
						.map(cobjcategory -> getSimpleQualifier(getResulttype(cobjcategory), DNET_RESULT_TYPOLOGIES)))
				.put("concept", fields.get("concept"), nodes -> nodes.stream()
					.filter(node -> node.getAttributes() != null && StringUtils.isNotBlank(node.getAttributes().get("id")))
					.map(node -> Context.newBuilder().setId(node.getAttributes().get("id"))))
				.put("journal", fields.get("journal"), nodes -> nodes.stream()
					.map(node -> {
						final Journal.Builder journal = Journal.newBuilder();
						if (StringUtils.isNotBlank(node.getTextValue())) {
							journal.setName(node.getTextValue());
						}
						if (node.getAttributes() != null) {
							final Map<String, String> a = node.getAttributes();
							if (StringUtils.isNotBlank(a.get("issn"))) {
								journal.setIssnPrinted(a.get("issn"));
							}
							if (StringUtils.isNotBlank(a.get("eissn"))) {
								journal.setIssnOnline(a.get("eissn"));
							}
							if (StringUtils.isNotBlank(a.get("lissn"))) {
								journal.setIssnLinking(a.get("lissn"));
							}
							if (StringUtils.isNotBlank(a.get("sp"))) {
								journal.setSp(a.get("sp"));
							}
							if (StringUtils.isNotBlank(a.get("ep"))) {
								journal.setEp(a.get("ep"));
							}
							if (StringUtils.isNotBlank(a.get("iss"))) {
								journal.setIss(a.get("iss"));
							}
							if (StringUtils.isNotBlank(a.get("vol"))) {
								journal.setVol(a.get("vol"));
							}
						}
						return journal;
					}));

		specs.put(OafEntity.getDescriptor(), SpecificationDescriptor.newInstance())
				.setBuilder(OafEntity.newBuilder().setType(Type.result).setId(objIdentifier))
				.put("originalId", fields.get("originalId"), nodes -> nodes.stream()
						.map(Node::getTextValue)
						.map(s -> StringUtils.contains(s, ID_SEPARATOR) ? StringUtils.substringAfter(s, ID_SEPARATOR) : s)
						.filter(s -> !s.trim().matches(URL_REGEX)))
				.put("collectedfrom", fields.get("collectedfrom"), nodes -> nodes.stream()
						.map(node -> getKV(
								oafSplitId(Type.datasource.name(), node.getAttributes().get("id")),
								node.getAttributes().get("name"))))
				.put("pid", fields.get("pid"), nodes -> nodes.stream()
						.filter(pid -> {
							final Map<String, String> a = pid.getAttributes();
							return a.containsKey(IDENTIFIER_TYPE) || a.containsKey(ALTERNATE_IDENTIFIER_TYPE);
						})
						.filter(pid -> {
							final Map<String, String> a = pid.getAttributes();
							return !"url".equalsIgnoreCase(a.get(IDENTIFIER_TYPE)) && !"url".equalsIgnoreCase(a.get(ALTERNATE_IDENTIFIER_TYPE));
						})
						.map(pid -> {
							final Map<String, String> a = pid.getAttributes();
							final String identifierType = a.get(IDENTIFIER_TYPE);
							final String altIdentifierType = a.get(ALTERNATE_IDENTIFIER_TYPE);
							return StructuredProperty.newBuilder()
									.setValue(pid.getTextValue())
									.setQualifier(getSimpleQualifier(
											StringUtils.isNotBlank(identifierType) ?
													identifierType : altIdentifierType, DNET_PID_TYPES))
									.build();
						}))
				.put("dateofcollection", fields.get("dateofcollection"), nodes -> nodes.stream()
						.map(Node::getTextValue))
				.put("dateoftransformation", fields.get("dateoftransformation"), nodes -> nodes.stream()
						.map(Node::getTextValue))
				.put("cachedRel", fields.get("cachedRel"), nodes -> nodes.stream()
						.map(node -> getOafRel(objIdentifier, node,
								OafRel.newBuilder()
									.setSource(objIdentifier)
									.setChild(false)))
						.filter(Objects::nonNull)
						.map(oafRel -> oafRel.build()));

		for(final Entry<Descriptor, SpecificationDescriptor> spec : specs.entrySet()) {
			final Descriptor d = spec.getKey();
			final SpecificationDescriptor md = spec.getValue();

			for(Entry<String, Pair<String, Function<List<Node>, Object>>> entry : md.getFields().entrySet()) {
				final String fieldName = entry.getKey();
				final Pair<String, Function<List<Node>, Object>> pair = entry.getValue();
				final String xpath = pair.getKey();
				final Function<List<Node>, Object> function = pair.getValue();
				addField(md.getBuilder(), d.findFieldByName(fieldName), function.apply(getNodes(ap, vn, xpath)));
			}
		}

		return Oaf.newBuilder()
				.setKind(Kind.entity)
				.setDataInfo(ensureDataInfo(ap, vn, DataInfo.newBuilder()))
				.setEntity(((OafEntity.Builder) specs.get(OafEntity.getDescriptor())
						.getBuilder()
						.setField(
								OafEntity.getDescriptor().findFieldByName(Type.result.name()),
								((Result.Builder) specs.get(Result.getDescriptor()).getBuilder())
										.setMetadata((Metadata) specs.get(Metadata.getDescriptor()).getBuilder().build())
										.addInstance((Instance) specs.get(Instance.getDescriptor()).getBuilder().build())
										.build()))
						.setOaiprovenance(getOaiProvenance(ap, vn))
						.build())
				.build();
	}

	private OafRel.Builder getOafRel(final String objIdentifier, final Node node, final OafRel.Builder oafRel) {
		final Map<String, String> a = node.getAttributes();

		switch (node.getName().toLowerCase()) {
		case PROJECTID:
			return oafRel
					.setTarget(oafSplitId(Type.project.name(), StringUtils.trim(node.getTextValue())))
					.setRelType(RelType.resultProject)
					.setSubRelType(SubRelType.outcome)
					.setRelClass("isProducedBy");

		case RELATED_PUBLICATION:
		case RELATED_DATASET:
			return oafRel
					.setTarget(oafSimpleId(Type.result.name(), StringUtils.trim(a.get("id"))))
					.setRelType(RelType.resultResult)
					.setSubRelType(SubRelType.publicationDataset)
					.setRelClass("isRelatedTo");

		case RELATED_IDENTIFIER:
			return oafRel
					.setTarget(node.getTextValue())
					.setRelType(RelType.resultResult)
					.setSubRelType(SubRelType.relationship)
					.setRelClass(a.get(RELATION_TYPE))
					.setCachedTarget(
							OafEntity.newBuilder()
									.setType(Type.result)
									.setId(objIdentifier)
									.addPid(
											StructuredProperty.newBuilder()
													.setValue(node.getTextValue())
													.setQualifier(getSimpleQualifier(a.get(RELATED_IDENTIFIER_TYPE), DNET_PID_TYPES))
													.build()));
		default:
			return null;
		}
	}


	protected abstract String getResulttype(final String cobjcategory);

	protected abstract Map<String, String> getFields();

	protected String metadataXpath(final String otherValues) {
		return xpath("record", "metadata", otherValues);
	}

	private OriginDescription getOriginDescription(final AutoPilot ap, final VTDNav vn, final String basePath) throws VtdException {
		final OriginDescription.Builder od = OriginDescription.newBuilder();
		if (getNodes(ap, vn, basePath).isEmpty()) {
			return od.build();
		}
		final Map<String, String> odAttr = getNode(ap, vn, basePath).getAttributes();

		final String harvestDate = odAttr.get("harvestDate");
		if (StringUtils.isNotBlank(harvestDate)) {
			od.setHarvestDate(harvestDate);
		}
		final String altered = odAttr.get("altered");
		if (StringUtils.isNotBlank(altered)) {
			od.setAltered(Boolean.valueOf(altered));
		}
		final String baseUrl = getFirstValue(ap, vn, basePath + xpath("baseURL"));
		if (StringUtils.isNotBlank(basePath)) {
			od.setBaseURL(baseUrl);
		}
		final String identifier = getFirstValue(ap, vn, basePath + xpath("identifier"));
		if (StringUtils.isNotBlank(identifier)) {
			od.setIdentifier(identifier);
		}
		final String datestamp = getFirstValue(ap, vn, basePath + xpath("datestamp"));
		if (StringUtils.isNotBlank(datestamp)) {
			od.setDatestamp(datestamp);
		}
		final String metadataNamespace = getFirstValue(ap, vn, basePath + xpath("metadataNamespace"));
		if (StringUtils.isNotBlank(metadataNamespace)) {
			od.setMetadataNamespace(metadataNamespace);
		}
		final OriginDescription originDescription = getOriginDescription(ap, vn, basePath + xpath("originDescription"));
		if (originDescription.hasHarvestDate()) {
			od.setOriginDescription(originDescription);
		}

		return od.build();
	}

    private void addField(final Message.Builder builder, final Descriptors.FieldDescriptor descriptor, final Object value) {

        if (value == null) return;

        if (value instanceof Stream) {
        	addField(builder, descriptor, ((Stream) value).collect(Collectors.toList()));
        } else if (value instanceof Collection<?>) {
            for (final Object o : (Collection<Object>) value) {
                addField(builder, descriptor, o);
            }
        } else {
            Object v = value;
            switch (descriptor.getType()) {
                case BOOL:
                    v = Boolean.valueOf(value.toString());
                    break;
                case BYTES:
                    v = value.toString().getBytes(Charset.forName(UTF_8));
                    break;
                case DOUBLE:
                    v = Double.valueOf(value.toString());
                    break;
                case FLOAT:
                    v = Float.valueOf(value.toString());
                    break;
                case INT32:
                case INT64:
                case SINT32:
                case SINT64:
                    v = Integer.valueOf(value.toString());
                    break;
                case MESSAGE:
                    final Message.Builder q = builder.newBuilderForField(descriptor);

                    if (value instanceof Message.Builder) {
	                    v = ((Message.Builder) value).build();
                        final byte[] b = ((Message) v).toByteArray();
                        try {
                            q.mergeFrom(b);
                        } catch (final InvalidProtocolBufferException e) {
                            throw new IllegalArgumentException("Unable to merge value: " + v + " with builder: " + q.getDescriptorForType().getName());
                        }
                    } else if (Qualifier.getDescriptor().getName().equals(q.getDescriptorForType().getName())) {
                        if (value instanceof Qualifier) {
                            q.mergeFrom((Qualifier) v);
                        }
                    } else if (StructuredProperty.getDescriptor().getName().equals(q.getDescriptorForType().getName())) {
                        if (value instanceof StructuredProperty) {
                            q.mergeFrom((StructuredProperty) v);
                        }
                    } else if (KeyValue.getDescriptor().getName().equals(q.getDescriptorForType().getName())) {
                        if (value instanceof KeyValue) {
                            q.mergeFrom((KeyValue) v);
                        }
                    } else if (Journal.getDescriptor().getName().equals(q.getDescriptorForType().getName())) {
	                    if (value instanceof Journal) {
		                    q.mergeFrom((Journal) v);
	                    }
                    } else if (Context.getDescriptor().getName().equals(q.getDescriptorForType().getName())) {
	                    if (value instanceof Context) {
		                    q.mergeFrom((Context) v);
	                    }
                    } else if (Author.getDescriptor().getName().equals(q.getDescriptorForType().getName())) {
                    	if (value instanceof Author) {
                    		q.mergeFrom((Author) v);
	                    }
                    } else if (ExternalReference.getDescriptor().getName().equals(q.getDescriptorForType().getName())) {
                        if (value instanceof ExternalReference) {
                        	q.mergeFrom((ExternalReference) v);
                        }
                    } else if (OafRel.getDescriptor().getName().equals(q.getDescriptorForType().getName())) {
                    	if (value instanceof OafRel) {
                    	    q.mergeFrom((OafRel) v);
	                    }
                    } else if (StringField.getDescriptor().getName().equals(q.getDescriptorForType().getName())) {
                        if (value instanceof StringField) {
                            q.mergeFrom((StringField) v);
                        } else {
                            q.setField(StringField.getDescriptor().findFieldByName("value"), v);
                        }
                    } else if (BoolField.getDescriptor().getName().equals(q.getDescriptorForType().getName())) {
                        if (value instanceof BoolField) {
                            q.mergeFrom((BoolField) v);
                        } else if (value instanceof String) {
                            q.setField(BoolField.getDescriptor().findFieldByName("value"), Boolean.valueOf((String) v));
                        } else {
                            q.setField(BoolField.getDescriptor().findFieldByName("value"), v);
                        }
                    } else if (IntField.getDescriptor().getName().equals(q.getDescriptorForType().getName())) {
                        if (value instanceof IntField) {
                            q.mergeFrom((IntField) v);
                        } else if (value instanceof String) {
                            q.setField(IntField.getDescriptor().findFieldByName("value"), NumberUtils.toInt((String) v));
                        } else {
                            q.setField(IntField.getDescriptor().findFieldByName("value"), v);
                        }
                    }

                    v = q.buildPartial();
                    break;
                default:
                    break;
            }

            doAddField(builder, descriptor, v);
        }
    }

	private void doAddField(final Message.Builder builder, final Descriptors.FieldDescriptor fd, final Object value) {
        if (value != null) {
            if (fd.isRepeated()) {
                builder.addRepeatedField(fd, value);
            } else if (fd.isOptional() || fd.isRequired()) {
                builder.setField(fd, value);
            }
        }
    }

	private OAIProvenance getOaiProvenance(final AutoPilot ap, final VTDNav vn) throws VtdException {
		return OAIProvenance.newBuilder()
				.setOriginDescription(getOriginDescription(ap, vn, xpath("record", "about", "provenance", "originDescription")))
				.build();
	}

	private FieldTypeProtos.DataInfo.Builder ensureDataInfo(
    		final AutoPilot ap, final VTDNav vn,
            final DataInfo.Builder info) throws VtdException {

        if (info.isInitialized()) return info;
        return buildDataInfo( ap, vn, invisible, provenance, trust, false, false);
    }

	private FieldTypeProtos.DataInfo.Builder buildDataInfo(
            final AutoPilot ap,
            final VTDNav vn,
            final boolean invisible,
            final String defaultProvenanceaction,
            final String defaultTrust,
            final boolean defaultDeletedbyinference,
            final boolean defaultInferred) throws VtdException {

		final DataInfo.Builder dataInfoBuilder = FieldTypeProtos.DataInfo.newBuilder()
            .setInvisible(invisible)
			.setInferred(defaultInferred)
            .setDeletedbyinference(defaultDeletedbyinference)
            .setTrust(defaultTrust)
	        .setProvenanceaction(getSimpleQualifier(defaultProvenanceaction, DNET_PROVENANCE_ACTIONS));

        // checking instanceof because when receiving an empty <oaf:datainfo> we don't want to parse it.

	    final String xpath = xpath("record", "about", "datainfo");
	    if (getNodes(ap, vn, xpath).size() > 0) {
		    final Map<String, String> provAction = getNode(ap, vn, xpath + xpath("provenanceaction")).getAttributes();
		    dataInfoBuilder
				    .setInvisible(Boolean.valueOf(getValue(getNode(ap, vn, xpath + xpath("invisible")), String.valueOf(invisible))))
				    .setInferred(Boolean.valueOf(getValue(getNode(ap, vn, xpath + xpath("inferred")), String.valueOf(defaultInferred))))
				    .setDeletedbyinference(Boolean.valueOf(
						    getValue(getNode(ap, vn, xpath + xpath("deletedbyinference")), String.valueOf(defaultDeletedbyinference))))
				    .setTrust(getValue(getNode(ap, vn, xpath + xpath("trust")), defaultTrust))
				    .setInferenceprovenance(getValue(getNode(ap, vn, xpath + xpath("inferenceprovenance")), ""))
				    .setProvenanceaction(getSimpleQualifier(
						    getValue(provAction.get(CLASSID), defaultProvenanceaction),
						    DNET_PROVENANCE_ACTIONS));
	    }

	    return dataInfoBuilder;
    }

	private static String getValue(final Node node, final String defaultValue) {
		return (node != null && StringUtils.isNotBlank(node.getTextValue())) ? node.getTextValue() : defaultValue;
	}

	private static String getValue(final String value, final String defaultValue) {
		return StringUtils.isNotBlank(value) ? value : defaultValue;
	}

	private KeyValue getKV(final String id, final String name) {
		return KeyValue.newBuilder().setKey(id).setValue(name).build();
	}

	private Qualifier getSimpleQualifier(final String classname, final String schemename) {
		return getQualifier(classname, classname, schemename, schemename);
	}

	private Qualifier getQualifier(final String classid, final String classname, final String schemeid, final String schemename) {
		return Qualifier.newBuilder().setClassid(classid).setClassname(classname).setSchemeid(schemeid).setSchemename(schemename).build();
	}

	private StructuredProperty getStructuredProperty(final String value,
			final String classid,
			final String classname,
			final String schemeid,
			final String schemename) {
		if ((value == null) || value.isEmpty()) return null;
		return StructuredProperty.newBuilder().setValue(value).setQualifier(getQualifier(classid, classname, schemeid, schemename)).build();
	}

	/**
	 * Gets the classname of the given class code
	 *
	 * @param code class code.
	 * @return the class name, if the code is a key of the map. The code itself otherwise.
	 */
	private String getClassName(final String code) {
		final String classname = AbstractDNetXsltFunctions.code2name.get(code);
		if (StringUtils.isBlank(classname)) return code;
		return classname;
	}
}
