package eu.dnetlib.data.transform.xml2;

import java.nio.charset.Charset;
import java.util.Collection;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.google.common.collect.Maps;
import com.google.protobuf.Descriptors;
import com.google.protobuf.InvalidProtocolBufferException;
import com.google.protobuf.Message;
import eu.dnetlib.data.proto.FieldTypeProtos.*;
import eu.dnetlib.data.proto.OafProtos.OafRel;
import eu.dnetlib.data.proto.ResultProtos.Result.Context;
import eu.dnetlib.data.proto.ResultProtos.Result.ExternalReference;
import eu.dnetlib.data.proto.ResultProtos.Result.Journal;
import eu.dnetlib.data.transform.xml.AbstractDNetXsltFunctions;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;

import static eu.dnetlib.data.transform.xml2.VtdUtilityParser.xpath;

public class Utils {

	public static final String URL_REGEX = "^(http|https|ftp)\\://.*";

	public static final String ID_SEPARATOR = "::";

	public static final String TITLE_TYPE = "titleType";
	public static final String DATE_TYPE = "dateType";
	public static final String KEYWORD = "keyword";

	public static final String DNET_EXT_REF_TYPOLOGIES = "dnet:externalReference_typologies";
	public static final String DNET_TITLE_TYPOLOGIES = "dnet:dataCite_title";
	public static final String DNET_SUBJECT_TYPOLOGIES = "dnet:subject_classification_typologies";
	public static final String DNET_RESULT_TYPOLOGIES = "dnet:result_typologies";
	public static final String DNET_PUBLICATION_RESOURCE = "dnet:publication_resource";
	public static final String DNET_DATA_CITE_RESOURCE = "dnet:dataCite_resource";
	public static final String DNET_ACCESS_MODES = "dnet:access_modes";
	public static final String DNET_LANGUAGES = "dnet:languages";
	public static final String DNET_PID_TYPES = "dnet:pid_types";

	public static final String IDENTIFIER_TYPE = "identifierType";
	public static final String ALTERNATE_IDENTIFIER_TYPE = "alternateIdentifierType";
	public static final String DNET_PROVENANCE_ACTIONS = "dnet:provenanceActions";

	public static final String CLASSID = "classid";
	public static final String CLASSNAME = "classname";
	public static final String SCHEMEID = "schemeid";
	public static final String SCHEMENAME = "schemename";

	public static final String RELATION_TYPE = "relationType";
	public static final String RELATED_IDENTIFIER_TYPE = "relatedIdentifierType";
	public static final String RIGHTS_URI = "rightsURI";

	public static final String UTF_8 = "UTF-8";

	// publication
	public static final String PROJECTID = "projectid";
	public static final String RELATED_DATASET = "relatedDataSet";
	public static final String RELATED_PUBLICATION = "relatedPublication";
	public static final String RELATED_IDENTIFIER = "relatedIdentifier";

	protected static Map<String, String> mappingAccess = Maps.newHashMap();

	static {
		mappingAccess.put("info:eu-repo/semantics/openAccess", "OPEN");
		mappingAccess.put("info:eu-repo/semantics/closedAccess", "CLOSED");
		mappingAccess.put("info:eu-repo/semantics/restrictedAccess", "RESTRICTED");
		mappingAccess.put("info:eu-repo/semantics/embargoedAccess", "EMBARGO");

		// Transformator now maps the access rights into proper values, not sure if it does for all datasets.
		mappingAccess.put("OPEN", "OPEN");
		mappingAccess.put("CLOSED", "CLOSED");
		mappingAccess.put("RESTRICTED", "RESTRICTED");
		mappingAccess.put("EMBARGO", "EMBARGO");
	}

	public static String getValue(final Node node, final String defaultValue) {
		return (node != null && StringUtils.isNotBlank(node.getTextValue())) ? node.getTextValue() : defaultValue;
	}

	public static String getValue(final String value, final String defaultValue) {
		return StringUtils.isNotBlank(value) ? value : defaultValue;
	}

	public static KeyValue getKV(final String id, final String name) {
		return KeyValue.newBuilder().setKey(id).setValue(name).build();
	}

	public static Qualifier getSimpleQualifier(final String classname, final String schemename) {
		return getQualifier(classname, classname, schemename, schemename);
	}

	public static Qualifier getQualifier(final String classid, final String classname, final String schemeid, final String schemename) {
		return Qualifier.newBuilder().setClassid(classid).setClassname(classname).setSchemeid(schemeid).setSchemename(schemename).build();
	}

	public static StructuredProperty getStructuredProperty(final String value,
			final String classid,
			final String classname,
			final String schemeid,
			final String schemename) {
		if ((value == null) || value.isEmpty()) return null;
		return StructuredProperty.newBuilder().setValue(value).setQualifier(getQualifier(classid, classname, schemeid, schemename)).build();
	}

	/**
	 * Gets the classname of the given class code
	 *
	 * @param code class code.
	 * @return the class name, if the code is a key of the map. The code itself otherwise.
	 */
	public static String getClassName(final String code) {
		final String classname = AbstractDNetXsltFunctions.code2name.get(code);
		if (StringUtils.isBlank(classname)) return code;
		return classname;
	}

	public static String metadataXpath(final String otherValues) {
		return xpath("record", "metadata", otherValues);
	}

	public static void addField(final Message.Builder builder, final Descriptors.FieldDescriptor descriptor, final Object value) {

		if (value == null) return;

		if (value instanceof Stream) {
			addField(builder, descriptor, ((Stream) value).collect(Collectors.toList()));
		} else if (value instanceof Collection<?>) {
			for (final Object o : (Collection<Object>) value) {
				addField(builder, descriptor, o);
			}
		} else {
			Object v = value;
			switch (descriptor.getType()) {
			case BOOL:
				v = Boolean.valueOf(value.toString());
				break;
			case BYTES:
				v = value.toString().getBytes(Charset.forName(UTF_8));
				break;
			case DOUBLE:
				v = Double.valueOf(value.toString());
				break;
			case FLOAT:
				v = Float.valueOf(value.toString());
				break;
			case INT32:
			case INT64:
			case SINT32:
			case SINT64:
				v = Integer.valueOf(value.toString());
				break;
			case MESSAGE:
				final Message.Builder q = builder.newBuilderForField(descriptor);

				if (value instanceof Message.Builder) {
					v = ((Message.Builder) value).build();
					final byte[] b = ((Message) v).toByteArray();
					try {
						q.mergeFrom(b);
					} catch (final InvalidProtocolBufferException e) {
						throw new IllegalArgumentException("Unable to merge value: " + v + " with builder: " + q.getDescriptorForType().getName());
					}
				} else if (Qualifier.getDescriptor().getName().equals(q.getDescriptorForType().getName())) {
					if (value instanceof Qualifier) {
						q.mergeFrom((Qualifier) v);
					}
				} else if (StructuredProperty.getDescriptor().getName().equals(q.getDescriptorForType().getName())) {
					if (value instanceof StructuredProperty) {
						q.mergeFrom((StructuredProperty) v);
					}
				} else if (KeyValue.getDescriptor().getName().equals(q.getDescriptorForType().getName())) {
					if (value instanceof KeyValue) {
						q.mergeFrom((KeyValue) v);
					}
				} else if (Journal.getDescriptor().getName().equals(q.getDescriptorForType().getName())) {
					if (value instanceof Journal) {
						q.mergeFrom((Journal) v);
					}
				} else if (Context.getDescriptor().getName().equals(q.getDescriptorForType().getName())) {
					if (value instanceof Context) {
						q.mergeFrom((Context) v);
					}
				} else if (Author.getDescriptor().getName().equals(q.getDescriptorForType().getName())) {
					if (value instanceof Author) {
						q.mergeFrom((Author) v);
					}
				} else if (ExternalReference.getDescriptor().getName().equals(q.getDescriptorForType().getName())) {
					if (value instanceof ExternalReference) {
						q.mergeFrom((ExternalReference) v);
					}
				} else if (OafRel.getDescriptor().getName().equals(q.getDescriptorForType().getName())) {
					if (value instanceof OafRel) {
						q.mergeFrom((OafRel) v);
					}
				} else if (StringField.getDescriptor().getName().equals(q.getDescriptorForType().getName())) {
					if (value instanceof StringField) {
						q.mergeFrom((StringField) v);
					} else {
						q.setField(StringField.getDescriptor().findFieldByName("value"), v);
					}
				} else if (BoolField.getDescriptor().getName().equals(q.getDescriptorForType().getName())) {
					if (value instanceof BoolField) {
						q.mergeFrom((BoolField) v);
					} else if (value instanceof String) {
						q.setField(BoolField.getDescriptor().findFieldByName("value"), Boolean.valueOf((String) v));
					} else {
						q.setField(BoolField.getDescriptor().findFieldByName("value"), v);
					}
				} else if (IntField.getDescriptor().getName().equals(q.getDescriptorForType().getName())) {
					if (value instanceof IntField) {
						q.mergeFrom((IntField) v);
					} else if (value instanceof String) {
						q.setField(IntField.getDescriptor().findFieldByName("value"), NumberUtils.toInt((String) v));
					} else {
						q.setField(IntField.getDescriptor().findFieldByName("value"), v);
					}
				}

				v = q.buildPartial();
				break;
			default:
				break;
			}

			doAddField(builder, descriptor, v);
		}
	}

	private static void doAddField(final Message.Builder builder, final Descriptors.FieldDescriptor fd, final Object value) {
		if (value != null) {
			if (fd.isRepeated()) {
				builder.addRepeatedField(fd, value);
			} else if (fd.isOptional() || fd.isRequired()) {
				builder.setField(fd, value);
			}
		}
	}

}
