package eu.dnetlib.data.transform.xml;

import java.util.HashMap;
import java.util.List;

import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.protobuf.Descriptors.Descriptor;
import eu.dnetlib.data.mapreduce.util.OafRowKeyDecoder;
import eu.dnetlib.data.proto.FieldTypeProtos.Author;
import eu.dnetlib.data.proto.FieldTypeProtos.KeyValue;
import eu.dnetlib.data.proto.FieldTypeProtos.StructuredProperty;
import eu.dnetlib.data.proto.OafProtos.Oaf;
import eu.dnetlib.data.proto.OafProtos.OafEntity;
import eu.dnetlib.data.proto.ResultProtos.Result;
import eu.dnetlib.data.proto.ResultProtos.Result.Context;
import eu.dnetlib.data.proto.ResultProtos.Result.ExternalReference;
import eu.dnetlib.data.proto.ResultProtos.Result.Instance;
import eu.dnetlib.data.proto.TypeProtos.Type;
import org.apache.commons.lang3.StringUtils;
import org.w3c.dom.NodeList;

public class OafToHbaseXsltFunctions extends CommonDNetXsltFunctions {

	public static String oafResult(
			final String resultId,
			final boolean invisible,
			final String provenance,
			final String trust,
			final NodeList about,
			final String originalId,
			final String dateOfCollection,
			final String dateOfTransformation,
			final NodeList metadata) {

		ValueMap values = null;
		try {
			final String entityId = OafRowKeyDecoder.decode(resultId).getKey();
			values = ValueMap.parseNodeList(metadata);
			final Descriptor mDesc = Result.Metadata.getDescriptor();

			final List<KeyValue> collectedFrom = getKeyValues(values, "collectedfrom", Type.datasource);
			final List<KeyValue> hostedBy = getKeyValues(values, "hostedby", Type.datasource);

			final Result.Metadata.Builder metadataBuilder = buildMetadata(values, mDesc);
			final Result.Builder result = buildResult(metadataBuilder, values, collectedFrom, hostedBy);
			final OafEntity.Builder entity = buildOafEntity(result, entityId, metadata, collectedFrom, originalId);
			entity.setDateofcollection(dateOfCollection)
					.setDateoftransformation(dateOfTransformation).setOaiprovenance(getOAIProvenance(about));

			final Oaf oaf = getOaf(entity, getDataInfo(invisible, about, provenance, trust, false, false));
			return base64(oaf.toByteArray());
		} catch (final Throwable e) {
			handleException(e, resultId, values);
		}
		return null;
	}

	public static String oafResultUpdate(final String resultId,
			final String provenance,
			final String trust,
			final NodeList nodelist) {
		ValueMap values = null;
		try {
			final String entityId = OafRowKeyDecoder.decode(resultId).getKey();
			values = ValueMap.parseNodeList(nodelist);
			final List<KeyValue> hostedBy = getKeyValues(values, "hostedby", Type.datasource);

			final Descriptor mDesc = Result.Metadata.getDescriptor();

			final Result.Metadata.Builder metadata = buildMetadata(values, mDesc);
			final Result.Builder result = buildResult(metadata, values, null, hostedBy);

			final OafEntity.Builder entity = buildOafEntity(result, entityId, nodelist, null, null);
			final Oaf oaf = getOaf(entity, null);
			return base64(oaf.toByteArray());
		} catch (final Throwable e) {
			handleException(e, resultId, values);
		}
		return null;
	}

	private static OafEntity.Builder buildOafEntity(
			final Result.Builder result,
			final String entityId,
			final NodeList nodelist,
			final List<KeyValue> collectedFrom,
			final String originalId) {

		final List<StructuredProperty> pids = Lists.newArrayList();
		pids.addAll(parsePids(nodelist));

		final OafEntity.Builder entity =
				getEntity(Type.result, entityId, collectedFrom, StringUtils.isBlank(originalId) ? null : Lists.newArrayList(originalId), null, null, pids)
						.setResult(result);
		return entity;
	}

	private static Result.Metadata.Builder buildMetadata(final ValueMap values, final Descriptor mDesc) {
		final Result.Metadata.Builder metadata = Result.Metadata.newBuilder();

		if (values.get("creator") != null) {
			for (final Element e : values.get("creator")) {

				final Author.Builder author = Author.newBuilder();

				final String fullname = e.getText();
				author.setFullname(fullname);
				author.setRank(Integer.valueOf(e.getAttributeValue(ValueMap.IDX_ATTRIBUTE)));

				final String nameIdentifier = e.getAttributeValue("nameIdentifier");
				final String nameIdentifierScheme = e.getAttributeValue("nameIdentifierScheme");

				if (StringUtils.isNotBlank(nameIdentifier) && StringUtils.isNotBlank(nameIdentifierScheme)) {
					author.addPid(getKV(nameIdentifierScheme, nameIdentifier));
				}

				final eu.dnetlib.pace.model.Person p = new eu.dnetlib.pace.model.Person(fullname, false);
				if (p.isAccurate()) {
					author.setName(p.getNormalisedFirstName());
					author.setSurname(p.getNormalisedSurname());
				}
				metadata.addAuthor(author);
			}
		}

		addStructuredProps(metadata, mDesc.findFieldByName("subject"), values.get("subject"), "keyword", "dnet:subject_classification_typologies");
		addStructuredProps(metadata, mDesc.findFieldByName("title"), values.get("title"), "main title", "dnet:dataCite_title");
		for (final String fieldname : Lists.newArrayList("description", "source", "contributor", "refereed")) {
			if (values.get(fieldname) != null) {
				for (final String s : values.get(fieldname).listValues()) {
					addField(metadata, mDesc.findFieldByName(fieldname), s);
				}
			}
		}
		addField(metadata, mDesc.findFieldByName("language"), setQualifier(getDefaultQualifier("dnet:languages"), values.get("language").listValues()));
		addField(metadata, mDesc.findFieldByName("dateofacceptance"), values.get("dateaccepted").listValues());
		addField(metadata, mDesc.findFieldByName("publisher"), values.get("publisher").listValues());
		addField(metadata, mDesc.findFieldByName("embargoenddate"), values.get("embargoenddate").listValues());
		addField(metadata, mDesc.findFieldByName("storagedate"), values.get("storagedate").listValues());

		String resulttype = getResultType(values);
		addField(metadata, mDesc.findFieldByName("resulttype"), getSimpleQualifier(resulttype, "dnet:result_typologies"));

		addField(metadata, mDesc.findFieldByName("fulltext"), values.get("fulltext").listValues());
		addField(metadata, mDesc.findFieldByName("format"), values.get("format").listValues());
		if (values.get("concept") != null) {
			for (final Element e : values.get("concept")) {
				final String id = e.getAttributes().get("id");
				if (StringUtils.isNotBlank(id)) {
					metadata.addContext(Context.newBuilder().setId(id));
				}
			}
		}
		if (values.get("journal") != null) {
			for (final Element e : values.get("journal")) {
				addJournal(metadata, e);
			}
		}
		return metadata;
	}

	private static String getResultType(final ValueMap values) {
			
		final Element cobjcategory = values.get("cobjcategory").stream()
				.map(e -> StringUtils.isNotBlank(e.getText()) ? e : new Element("0000", e.getAttributes()))
				.findFirst()
				.orElse(new Element("0000", new HashMap<>()));

		final String resulttype = cobjcategory.getAttributeValue("type");
		if (StringUtils.isNotBlank(resulttype)) {
			return resulttype;
		}

		return getDefaultResulttype(cobjcategory);
	}

	private static Result.Builder buildResult(final Result.Metadata.Builder metadata,
			final ValueMap values,
			final List<KeyValue> collectedFrom,
			final List<KeyValue> hostedBy) {
		final Result.Builder result = Result.newBuilder();

		final Instance.Builder instance = Instance.newBuilder();

		addField(instance, Instance.getDescriptor().findFieldByName("license"), values.get("license").listValues());

		addField(instance, Instance.getDescriptor().findFieldByName("accessright"),
				setQualifier(getDefaultQualifier("dnet:access_modes"), values.get("accessrights").listValues()));

		addField(instance, Instance.getDescriptor().findFieldByName("instancetype"),
				setQualifier(getDefaultQualifier("dnet:publication_resource"), values.get("cobjcategory").listValues()));

		addField(instance, Instance.getDescriptor().findFieldByName("hostedby"), hostedBy);
		addField(instance, Instance.getDescriptor().findFieldByName("collectedfrom"), collectedFrom);
		addField(instance, Instance.getDescriptor().findFieldByName("dateofacceptance"), values.get("dateaccepted").listValues());

		if (values.get("identifier") != null) {
			addField(instance, Instance.getDescriptor().findFieldByName("url"),
					Lists.newArrayList(Iterables.filter(values.get("identifier").listValues(), urlFilter)));
		}

		result.addInstance(instance);

		final List<Element> extrefs = values.get("reference");
		if (!extrefs.isEmpty()) {
			final Descriptor extDesc = ExternalReference.getDescriptor();
			for (final Element element : extrefs) {
				final ExternalReference.Builder extref = ExternalReference.newBuilder();
				addField(extref, extDesc.findFieldByName("url"), element.getText());
				addField(extref, extDesc.findFieldByName("sitename"), element.getAttributes().get("source"));
				addField(extref, extDesc.findFieldByName("refidentifier"), element.getAttributes().get("identifier"));
				addField(extref, extDesc.findFieldByName("label"), element.getAttributes().get("title"));
				addField(extref, extDesc.findFieldByName("query"), element.getAttributes().get("query"));
				addField(extref, extDesc.findFieldByName("qualifier"),
						setQualifier(getDefaultQualifier("dnet:externalReference_typologies"), Lists.newArrayList(element.getAttributes().get("type")))
								.build());

				result.addExternalReference(extref);
			}
		}

		return result.setMetadata(metadata);
	}

	private static void handleException(Throwable e, final String resultId, final ValueMap values) {
		System.err.println("resultId: " + resultId);
		if (values != null) {
			System.err.println("values: " + values);
		}
		e.printStackTrace();
		throw new RuntimeException(e);
	}
}
