package eu.dnetlib.msro.openaireplus.api.objects;

import java.text.SimpleDateFormat;
import java.util.*;
import java.util.concurrent.TimeUnit;

import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpException;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpService;
import eu.dnetlib.miscutils.datetime.DateUtils;
import eu.dnetlib.miscutils.functional.hash.Hashing;
import eu.dnetlib.msro.openaireplus.api.OpenAIRESubmitterUtils;
import eu.dnetlib.msro.rmi.MSROException;
import io.swagger.annotations.ApiModelProperty;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.velocity.app.VelocityEngine;
import org.apache.velocity.tools.generic.EscapeTool;
import org.springframework.ui.velocity.VelocityEngineUtils;

/**
 * Created by michele on 02/12/15.
 */
public class ResultEntry {

	private String openaireId;
	private String originalId;
	private String title;
	private List<String> authors = new ArrayList<String>();
	private String publisher;
	private String description;
	private String language;
	private List<PidEntry> pids = new ArrayList<PidEntry>();
	private String licenseCode;
	private String embargoEndDate;
	private String type = "publication";
	private String resourceType;
	private String url;
	private String collectedFromId;
	private String hostedById;

	// String according to openaire guidelines:
	// info:eu-repo/grantAgreement/Funder/FundingProgram/ProjectID/[Jurisdiction]/[ProjectName]/[ProjectAcronym]
	private List<String> contexts = new ArrayList<String>();

	// String according to the EGI context profile, example: egi::classification::natsc::math
	private List<String> linksToProjects = new ArrayList<String>();

	private static long last_cache_update = 0;
	private static final Map<String, Map<String, String>> cached_vocabularies = new HashMap<String, Map<String, String>>();
	private static final Map<String, DatasourceEntry> cached_datasources = new HashMap<String, DatasourceEntry>();
	private static final Map<String, String> cached_contexts = new HashMap<String, String>();

	private static final Log log = LogFactory.getLog(ResultEntry.class);

	public ResultEntry() {
	}

	public String getOpenaireId() {
		return openaireId;
	}

	public void setOpenaireId(String openaireId) {
		this.openaireId = openaireId;
	}

	public String getOriginalId() {
		return originalId;
	}

	public void setOriginalId(final String originalId) {
		this.originalId = originalId;
	}

	@ApiModelProperty(required = true)
	public String getTitle() {
		return title;
	}

	public void setTitle(final String title) {
		this.title = title;
	}

	public List<String> getAuthors() {
		return authors;
	}

	public void setAuthors(final List<String> authors) {
		this.authors = authors;
	}

	public String getPublisher() {
		return publisher;
	}

	public void setPublisher(final String publisher) {
		this.publisher = publisher;
	}

	public String getDescription() {
		return description;
	}

	public void setDescription(final String description) {
		this.description = description;
	}

	@ApiModelProperty(value = "ISO Alpha-3 code. E.g. 'eng', 'ita'")
	public String getLanguage() {
		return language;
	}

	public void setLanguage(final String language) {
		this.language = language;
	}

	public List<PidEntry> getPids() {
		return pids;
	}

	public void setPids(final List<PidEntry> pids) {
		this.pids = pids;
	}

	@ApiModelProperty(required = true, allowableValues = "OPEN, CLOSED, RESTRICTED, EMBARGO, UNKNOWN, OTHER")
	public String getLicenseCode() {
		return licenseCode;
	}

	public void setLicenseCode(final String licenseCode) {
		this.licenseCode = licenseCode;
	}

	@ApiModelProperty(required = true, value = "Use 001 for articles, 021 for datasets. See: http://api.openaire.eu/vocabularies/dnet:publication_resource.")
	public String getResourceType() {
		return resourceType;
	}

	public void setResourceType(final String resourceType) {
		this.resourceType = resourceType;
	}

	@ApiModelProperty(required = true)
	public String getUrl() {
		return url;
	}

	public void setUrl(final String url) {
		this.url = url;
	}

	@ApiModelProperty(required = true, value = "Use opendoar___::2659 for Zenodo Publications; re3data_____::r3d100010468 for Zenodo datasets; infrastruct::openaire for OpenAIRE portal.")
	public String getCollectedFromId() {
		return collectedFromId;
	}

	public void setCollectedFromId(final String collectedFromId) {
		this.collectedFromId = collectedFromId;
	}

	public String getHostedById() {
		return hostedById;
	}

	public void setHostedById(final String hostedById) {
		this.hostedById = hostedById;
	}

	@ApiModelProperty(value = "E.g. fet, egi::classification::natsc::math::pure, egi::projects::EMI")
	public List<String> getContexts() {
		return contexts;
	}

	public void setContexts(final List<String> contexts) {
		this.contexts = contexts;
	}

	@ApiModelProperty(value = "E.g. info:eu-repo/grantAgreement/EC/FP7/283595/EU//OpenAIREplus")
	public List<String> getLinksToProjects() {
		return linksToProjects;
	}

	public void setLinksToProjects(final List<String> linksToProjects) {
		this.linksToProjects = linksToProjects;
	}

	@ApiModelProperty(allowableValues = "publication, dataset")
	public String getType() {
		return type;
	}

	public void setType(final String type) {
		this.type = type;
	}

	public String getEmbargoEndDate() {
		return embargoEndDate;
	}

	public void setEmbargoEndDate(final String embargoEndDate) {
		this.embargoEndDate = embargoEndDate;
	}

	public String asOafRecord(final VelocityEngine ve,
			final ISLookUpService lookupService,
			final String oafSchemaLocation) throws Exception {

		if (StringUtils.isBlank(getOriginalId()) && StringUtils.isBlank(getOpenaireId())) {
			throw new MSROException("One of the following fields is required: originalId or openaireId");
		}
		if (StringUtils.isBlank(getTitle())) { throw new MSROException("A required field is missing: title"); }
		if (StringUtils.isBlank(getUrl())) { throw new MSROException("A required field is missing: url"); }
		if (StringUtils.isBlank(getLicenseCode())) { throw new MSROException("A required field is missing: licenseCode"); }
		if (StringUtils.isBlank(getResourceType())) { throw new MSROException("A required field is missing: resourceType"); }
		if (StringUtils.isBlank(getCollectedFromId())) { throw new MSROException("A required field is missing: collectedFromId"); }
		if (StringUtils.isBlank(getType())) { throw new MSROException("A required field is missing: type"); }

		final DatasourceEntry collectedFromEntry = getDatasourceInfo(collectedFromId, lookupService);
		final DatasourceEntry hostedByEntry = getDatasourceInfo(hostedById, lookupService);

		if (StringUtils.isBlank(openaireId)) {
			setOpenaireId(calculateOpenaireId(originalId, collectedFromEntry));
		}

		if (!openaireId.matches("^\\w{12}::\\w{32}$")) {
			throw new MSROException("Invalid openaireId: " + openaireId + " - regex ^\\w{12}::\\w{32}$ not matched");
		}

		final Map<String, Object> model = new HashMap<String, Object>();
		model.put("esc", new EscapeTool());
		model.put("util", new OpenAIRESubmitterUtils());
		model.put("pub", this);
		model.put("objIdentifier", getOpenaireId());
		model.put("oafSchemaLocation", oafSchemaLocation);
		model.put("resultTypes", getVocabulary("dnet:result_typologies", lookupService));
		model.put("licenses", getVocabulary("dnet:access_modes", lookupService));
		model.put("resourceTypes", getVocabulary("dnet:publication_resource", lookupService));
		model.put("pidTypes", getVocabulary("dnet:pid_types", lookupService));
		model.put("languages", getVocabulary("dnet:languages", lookupService));
		model.put("contexts", getContexts(lookupService));
		model.put("dateOfCollection", (new SimpleDateFormat("yyyy-MM-dd\'T\'hh:mm:ss\'Z\'")).format(new Date()));
		model.put("collectedFrom", collectedFromEntry);
		model.put("hostedBy", hostedByEntry);

		return VelocityEngineUtils.mergeTemplateIntoString(ve, "/eu/dnetlib/msro/openaireplus/api/indexRecord.xml.vm", "UTF-8", model);
	}

	private static String calculateOpenaireId(final String originalId, final DatasourceEntry collectedFromEntry) {
		return collectedFromEntry.getPrefix() + "::" + Hashing.md5(originalId);
	}

	public static String calculateOpenaireId(final String originalId, final String collectedFromId, final ISLookUpService lookupService)
			throws ISLookUpException {
		return calculateOpenaireId(originalId, getDatasourceInfo(collectedFromId, lookupService));
	}

	private synchronized static DatasourceEntry getDatasourceInfo(final String dsId, final ISLookUpService lookupService) throws ISLookUpException {
		if (StringUtils
				.isBlank(dsId)) { return new DatasourceEntry("openaire____::1256f046-bf1f-4afc-8b47-d0b147148b18", "Unknown Repository", "unknown_____"); }

		if (!cached_datasources.containsKey(dsId)) {
			final String query =
					"collection('/db/DRIVER/RepositoryServiceResources/RepositoryServiceResourceType')//CONFIGURATION[./DATASOURCE_ORIGINAL_ID='" + dsId
							+ "']/concat(./OFFICIAL_NAME, ' @@@ ', .//FIELD/value[../key='NamespacePrefix'])";
			final String s = lookupService.getResourceProfileByQuery(query);
			final String[] arr = s.split("@@@");

			final DatasourceEntry ds = new DatasourceEntry(dsId, arr[0].trim(), arr[1].trim());

			if (StringUtils.isBlank(ds.getName()) || StringUtils.isBlank(ds.getPrefix())) {
				log.error("Invalid datasource id: " + dsId);
				throw new ISLookUpException("Invalid datasource id: " + dsId);
			} else {
				cached_datasources.put(dsId, ds);
			}
		}

		return cached_datasources.get(dsId);

	}

	private synchronized static Map<String, String> getVocabulary(final String voc, final ISLookUpService lookupService) throws ISLookUpException {

		if (((DateUtils.now() - last_cache_update) < TimeUnit.MINUTES.toMillis(15)) && cached_vocabularies.containsKey(voc)) {
			return cached_vocabularies.get(voc);
		} else {
			final String query = "collection('/db/DRIVER/VocabularyDSResources/VocabularyDSResourceType')[.//VOCABULARY_NAME/@code='" + voc
					+ "']//TERM/concat(@code, ' @@@ ', @english_name)";

			final Map<String, String> map = new HashMap<String, String>();
			for (final String s : lookupService.quickSearchProfile(query)) {
				final String[] arr = s.split("@@@");
				map.put(arr[0].trim(), arr[1].trim());
			}

			cached_vocabularies.put(voc, map);

			last_cache_update = DateUtils.now();

			return map;
		}
	}

	private synchronized static Map<String, String> getContexts(final ISLookUpService lookupService) throws ISLookUpException {
		if (((DateUtils.now() - last_cache_update) > TimeUnit.MINUTES.toMillis(15)) || cached_contexts.isEmpty()) {
			final String query =
					"collection('/db/DRIVER/ContextDSResources/ContextDSResourceType')[.//context/@type='community']//*[name()='context' or name()='category' or name()='concept']/concat(@id, ' @@@ ', @label)";

			cached_contexts.clear();
			for (final String s : lookupService.quickSearchProfile(query)) {
				final String[] arr = s.split("@@@");
				cached_contexts.put(arr[0].trim(), arr[1].trim());
			}
			last_cache_update = DateUtils.now();
		}
		return cached_contexts;
	}

	@Override
	public String toString() {
		return StringUtils.isNotBlank(openaireId) ? openaireId : originalId;
	}
}