package eu.dnetlib.msro.openaireplus.workflows.nodes.index;

import java.io.StringReader;
import java.util.ArrayList;
import java.util.List;
import javax.annotation.Resource;

import com.googlecode.sarasvati.Arc;
import com.googlecode.sarasvati.NodeToken;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpService;
import eu.dnetlib.enabling.locators.UniqueServiceLocator;
import eu.dnetlib.functionality.index.client.solr.SolrIndexClient;
import eu.dnetlib.functionality.index.client.solr.SolrIndexClientFactory;
import eu.dnetlib.miscutils.functional.xml.ApplyXslt;
import eu.dnetlib.msro.openaireplus.api.RecentResultsQueue;
import eu.dnetlib.msro.openaireplus.utils.OafToIndexRecordFactory;
import eu.dnetlib.msro.rmi.MSROException;
import eu.dnetlib.msro.workflows.nodes.AsyncJobNode;
import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.solr.common.SolrInputDocument;
import org.dom4j.io.SAXReader;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.ClassPathResource;

/**
 * Created by michele on 15/12/15.
 */
public class FeedMissingClaimsJobNode extends AsyncJobNode {

	private static final Log log = LogFactory.getLog(FeedMissingClaimsJobNode.class);
	private RecentResultsQueue queue;
	private OafToIndexRecordFactory oafToIndexRecordFactory;

	@Resource
	private UniqueServiceLocator serviceLocator;

	@Value(value = "${openaireplus.msro.api.findSolrIndexUrl.xquery}")
	private ClassPathResource findSolrIndexUrl;

	@Autowired
	private SolrIndexClientFactory indexClientFactory;

	@Override
	protected String execute(final NodeToken nodeToken) throws Exception {

		final String format =
				nodeToken.getEnv().hasAttribute("format") ? nodeToken.getEnv().getAttribute("format") : nodeToken.getFullEnv().getAttribute("format");
		final String coll = format + "-index-openaire";
		final String indexDsId = nodeToken.getEnv().getAttribute("index_id");

		try(SolrIndexClient indexClient = (SolrIndexClient) indexClientFactory.getClient(coll)) {
			final List<SolrInputDocument> toFeed = new ArrayList<SolrInputDocument>();
			final List<String> toDeleteFromCache = new ArrayList<String>();

			final SAXReader reader = new SAXReader();
			final ApplyXslt xslt = oafToIndexRecordFactory.newTransformer(format);

			for (String record : queue) {
				final String id = reader.read(new StringReader(record)).valueOf("//*[local-name() = 'objIdentifier']");
				if (indexClient.isRecordIndexed(id)) {
					toDeleteFromCache.add(id);
				} else {
					toFeed.add(indexClient.prepareSolrDocument(record, indexDsId, xslt));
				}
			}

			indexClient.feed(toFeed, null);
			queue.remove(toDeleteFromCache);

		} catch (Throwable e) {
			log.error("Error feeding missing claims", e);
			throw new MSROException("Error feeding missing claims: " + e.getMessage(), e);
		}

		return Arc.DEFAULT_ARC;
	}

	public RecentResultsQueue getQueue() {
		return queue;
	}

	@Required
	public void setQueue(final RecentResultsQueue queue) {
		this.queue = queue;
	}

	public OafToIndexRecordFactory getOafToIndexRecordFactory() {
		return oafToIndexRecordFactory;
	}

	@Required
	public void setOafToIndexRecordFactory(final OafToIndexRecordFactory oafToIndexRecordFactory) {
		this.oafToIndexRecordFactory = oafToIndexRecordFactory;
	}

	private String calculateIndexBaseUrl() throws Exception {
		final String query = IOUtils.toString(findSolrIndexUrl.getInputStream());
		return serviceLocator.getService(ISLookUpService.class).getResourceProfileByQuery(query);
	}
}
