package eu.dnetlib.organizations.controller;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.security.core.Authentication;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import eu.dnetlib.organizations.model.OpenaireDuplicate;
import eu.dnetlib.organizations.model.Relationship;
import eu.dnetlib.organizations.model.utils.BrowseEntry;
import eu.dnetlib.organizations.model.utils.OrganizationConflict;
import eu.dnetlib.organizations.model.utils.OrganizationConflictImpl;
import eu.dnetlib.organizations.model.view.ConflictGroupView;
import eu.dnetlib.organizations.model.view.DuplicateGroupView;
import eu.dnetlib.organizations.model.view.OrganizationInfoView;
import eu.dnetlib.organizations.model.view.OrganizationSimpleView;
import eu.dnetlib.organizations.model.view.OrganizationView;
import eu.dnetlib.organizations.repository.OpenaireConflictRepository;
import eu.dnetlib.organizations.repository.OpenaireDuplicateRepository;
import eu.dnetlib.organizations.repository.UserCountryRepository;
import eu.dnetlib.organizations.repository.readonly.ConflictGroupViewRepository;
import eu.dnetlib.organizations.repository.readonly.DuplicateGroupViewRepository;
import eu.dnetlib.organizations.repository.readonly.OrganizationInfoViewRepository;
import eu.dnetlib.organizations.repository.readonly.OrganizationSimpleViewRepository;
import eu.dnetlib.organizations.repository.readonly.OrganizationViewRepository;
import eu.dnetlib.organizations.repository.readonly.SuggestionInfoViewByCountryRepository;
import eu.dnetlib.organizations.utils.DatabaseUtils;
import eu.dnetlib.organizations.utils.RelationType;

@RestController
@RequestMapping("/api/organizations")
public class OrganizationController {

	@Autowired
	private OrganizationViewRepository organizationViewRepository;
	@Autowired
	private OrganizationInfoViewRepository organizationInfoViewRepository;
	@Autowired
	private OrganizationSimpleViewRepository organizationSimpleViewRepository;
	@Autowired
	private OpenaireDuplicateRepository openaireDuplicateRepository;
	@Autowired
	private OpenaireConflictRepository openaireConflictRepository;
	@Autowired
	private ConflictGroupViewRepository conflictGroupViewRepository;
	@Autowired
	private SuggestionInfoViewByCountryRepository suggestionInfoViewByCountryRepository;
	@Autowired
	private UserCountryRepository userCountryRepository;
	@Autowired
	private DuplicateGroupViewRepository duplicateGroupViewRepository;
	@Autowired
	private DatabaseUtils databaseUtils;

	@RequestMapping(value = "/save", method = RequestMethod.POST)
	public List<String> save(@RequestBody final OrganizationView org, final Authentication authentication) {
		if (StringUtils.isBlank(org.getName())) {
			throw new RuntimeException("Missing field: name");
		} else if (StringUtils.isBlank(org.getCountry())) {
			throw new RuntimeException("Missing field: country");
		} else if (StringUtils.isBlank(org.getType())) {
			throw new RuntimeException("Missing field: type");
		} else if (UserInfo.isSuperAdmin(authentication) || userCountryRepository.verifyAuthorizationForCountry(org.getCountry(), authentication.getName())) {
			final String orgId = databaseUtils.insertOrUpdateOrganization(org, authentication.getName(), StringUtils.isNotBlank(org.getId()));
			return Arrays.asList(orgId);
		} else {
			throw new RuntimeException("User not authorized");
		}
	}

	@RequestMapping(value = "/info", method = RequestMethod.GET)
	public OrganizationInfoView infoById(@RequestParam final String id, final Authentication authentication) {
		return organizationInfoViewRepository.findById(id).get();
	}

	@RequestMapping(value = "/suggestionsInfo", method = RequestMethod.GET)
	public SuggestionInfo suggestionsInfo(final Authentication authentication) {
		final SuggestionInfo info = new SuggestionInfo();

		if (UserInfo.isSuperAdmin(authentication)) {
			suggestionInfoViewByCountryRepository.findAll().forEach(info::add);
		} else if (UserInfo.isSimpleUser(authentication) || UserInfo.isNationalAdmin(authentication)) {
			userCountryRepository.getCountriesForUser(authentication.getName())
					.stream()
					.map(suggestionInfoViewByCountryRepository::findById)
					.filter(Optional::isPresent)
					.map(Optional::get)
					.forEach(info::add);
		}
		return info;
	}

	@RequestMapping(value = "/get", method = RequestMethod.GET)
	public OrganizationView findById(@RequestParam final String id, final Authentication authentication) {
		final OrganizationView org = organizationViewRepository.findById(id).get();

		if (UserInfo.isSuperAdmin(authentication) || userCountryRepository.verifyAuthorizationForCountry(org.getCountry(), authentication.getName())) {
			return org;
		} else {
			throw new RuntimeException("User not authorized");
		}
	}

	@RequestMapping(value = "/conflicts", method = RequestMethod.GET)
	public List<OrganizationConflict> conflicts(@RequestParam final String id, final Authentication authentication) {
		if (UserInfo.isSuperAdmin(authentication) || userCountryRepository.verifyAuthorizationForId(id, authentication.getName())) {
			return openaireConflictRepository.getConflictsForId(id);
		} else {
			throw new RuntimeException("User not authorized");
		}
	}

	@RequestMapping(value = "/duplicates", method = RequestMethod.GET)
	public List<OpenaireDuplicate> duplicates(@RequestParam final String id, final Authentication authentication) {
		if (UserInfo.isSuperAdmin(authentication) || userCountryRepository.verifyAuthorizationForId(id, authentication.getName())) {
			return openaireDuplicateRepository.findByLocalId(id);
		} else {
			throw new RuntimeException("User not authorized");
		}
	}

	@RequestMapping(value = "/conflicts/byCountry/{country}", method = RequestMethod.GET)
	public Collection<Set<OrganizationConflictImpl>> findConflictsByCountry(@PathVariable final String country, final Authentication authentication) {

		databaseUtils.verifyConflictGroups(false);

		if (UserInfo.isSuperAdmin(authentication)) {
			return groupConflicts(conflictGroupViewRepository.findByCountry1OrCountry2(country, country).stream());
		} else if (UserInfo.isSimpleUser(authentication) || UserInfo.isNationalAdmin(authentication)) {
			final Stream<ConflictGroupView> list = userCountryRepository.getCountriesForUser(authentication.getName())
					.stream()
					.filter(country::equalsIgnoreCase)
					.map(c -> conflictGroupViewRepository.findByCountry1OrCountry2(c, c).stream())
					.findFirst()
					.orElse(Stream.empty());
			return groupConflicts(list);
		} else {
			throw new RuntimeException("User not authorized");
		}

	}

	@RequestMapping(value = "/duplicates/byCountry/{country}", method = RequestMethod.GET)
	public Iterable<DuplicateGroupView> findDuplicatesByCountry(@PathVariable final String country, final Authentication authentication) {

		if (UserInfo.isSuperAdmin(authentication)) {
			return duplicateGroupViewRepository.findByCountry(country);
		} else if (UserInfo.isSimpleUser(authentication) || UserInfo.isNationalAdmin(authentication)) {
			return userCountryRepository.getCountriesForUser(authentication.getName())
					.stream()
					.filter(country::equalsIgnoreCase)
					.map(duplicateGroupViewRepository::findByCountry)
					.findFirst()
					.orElse(new ArrayList<DuplicateGroupView>());
		} else {
			throw new RuntimeException("User not authorized");
		}

	}

	private Collection<Set<OrganizationConflictImpl>> groupConflicts(final Stream<ConflictGroupView> stream) {
		final Map<String, Set<OrganizationConflictImpl>> map = new TreeMap<>();
		stream.forEach(c -> {
			if (!map.containsKey(c.getGroup())) {
				map.put(c.getGroup(), new TreeSet<>());
			}
			map.get(c.getGroup()).add(new OrganizationConflictImpl(c.getId1(), c.getName1(), c.getType1(), c.getCity1(), c.getCountry1()));
			map.get(c.getGroup()).add(new OrganizationConflictImpl(c.getId2(), c.getName2(), c.getType2(), c.getCity2(), c.getCountry2()));
		});
		return map.values();
	}

	@RequestMapping(value = "/duplicates", method = RequestMethod.POST)
	public List<OpenaireDuplicate> duplicates(@RequestBody final List<OpenaireDuplicate> simrels, final Authentication authentication) {

		final boolean b = UserInfo.isSuperAdmin(authentication)
				|| simrels.stream()
						.map(OpenaireDuplicate::getLocalId)
						.distinct()
						.allMatch(id -> userCountryRepository.verifyAuthorizationForId(id, authentication.getName()));

		if (b) {
			return openaireDuplicateRepository.saveAll(simrels);
		} else {
			throw new RuntimeException("User not authorized");
		}
	}

	@RequestMapping(value = "/search/{page}/{size}", method = RequestMethod.GET)
	public Page<OrganizationSimpleView> search(@PathVariable final int page,
			@PathVariable final int size,
			@RequestParam final String q,
			final Authentication authentication) {
		return UserInfo.isSuperAdmin(authentication)
				? organizationSimpleViewRepository.findByNameContainingIgnoreCase(q, PageRequest.of(page, size))
				: organizationSimpleViewRepository.findByNameForUser(q, authentication.getName(), PageRequest.of(page, size));
	}

	@RequestMapping(value = "/byCountry/{code}/{page}/{size}", method = RequestMethod.GET)
	public Page<OrganizationSimpleView> findByCountry(@PathVariable final String code,
			@PathVariable final int page,
			@PathVariable final int size,
			final Authentication authentication) {
		if (UserInfo.isSuperAdmin(authentication) || userCountryRepository.verifyAuthorizationForCountry(code, authentication.getName())) {
			return organizationSimpleViewRepository.findByCountry(code, PageRequest.of(page, size));
		} else {
			throw new RuntimeException("User not authorized");
		}
	}

	@RequestMapping(value = "/byType/{type}/{page}/{size}", method = RequestMethod.GET)
	public Page<OrganizationSimpleView> findByType(@PathVariable final String type,
			@PathVariable final int page,
			@PathVariable final int size,
			final Authentication authentication) {
		return UserInfo.isSuperAdmin(authentication)
				? organizationSimpleViewRepository.findByType(type, PageRequest.of(page, size))
				: organizationSimpleViewRepository.findByTypeForUser(type, authentication.getName(), PageRequest.of(page, size));
	}

	@RequestMapping(value = "/browse/countries", method = RequestMethod.GET)
	public List<BrowseEntry> browseCountries(final Authentication authentication) {
		return UserInfo.isSuperAdmin(authentication)
				? organizationSimpleViewRepository.browseCountries()
				: organizationSimpleViewRepository.browseCountriesForUser(authentication.getName());
	}

	@RequestMapping(value = "/browse/types", method = RequestMethod.GET)
	public List<BrowseEntry> browseOrganizationTypes(final Authentication authentication) {
		return UserInfo.isSuperAdmin(authentication)
				? organizationSimpleViewRepository.browseTypes()
				: organizationSimpleViewRepository.browseTypesForUser(authentication.getName());
	}

	@RequestMapping(value = "/conflicts/fix/{masterId}", method = RequestMethod.POST)
	public List<Relationship> fixConflicts(final Authentication authentication, @PathVariable final String masterId, @RequestBody final List<String> otherIds) {
		if (UserInfo.isSuperAdmin(authentication) || userCountryRepository.verifyAuthorizationForId(masterId, authentication.getName())) {

			return otherIds.stream()
					.filter(id -> UserInfo.isSuperAdmin(authentication) || userCountryRepository.verifyAuthorizationForId(id, authentication.getName()))
					.map(id -> databaseUtils.makeRelation(masterId, id, RelationType.Merges))
					.flatMap(List::stream)
					.collect(Collectors.toList());

		} else {
			return new ArrayList<>();
		}

	}

}
