package eu.dnetlib.data.objectstore.s3;

import java.io.*;
import java.net.URLEncoder;
import java.nio.file.Files;
import java.nio.file.Path;
import javax.servlet.http.HttpServletResponse;

import com.amazonaws.ClientConfiguration;
import com.amazonaws.Protocol;
import com.amazonaws.auth.AWSCredentials;
import com.amazonaws.auth.AWSStaticCredentialsProvider;
import com.amazonaws.auth.BasicAWSCredentials;
import com.amazonaws.client.builder.AwsClientBuilder;
import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.AmazonS3ClientBuilder;
import com.amazonaws.services.s3.model.S3Object;
import eu.dnetlib.data.objectstore.rmi.ObjectStoreServiceException;
import eu.dnetlib.miscutils.datetime.HumanTime;
import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;

/**
 * The Class ModularObjectStoreRESTService implement the controller REST of the object Store.
 */
@Controller
public class ModularObjectStoreRESTService {

    private static final Log log = LogFactory.getLog(ModularObjectStoreRESTService.class); // NOPMD by marko on 11/24/08 5:02 PM



    @Value("${dnet.objectStore.s3.accessKey}")
    private String accessKey;


    @Value("${dnet.objectStore.s3.secretKey}")
    private String secretKey;

    @Value("${dnet.objectStore.s3.endPoint}")
    private String s3EndPoint;

    @Value("${dnet.objectStore.s3.objectStoreBucket}")
    private String objectStoreBucket;

    private AmazonS3 client;

    private static final String S3_REGION = "eu-west-3";


    private AmazonS3 initializeClient() throws ObjectStoreServiceException {
        try {
            final AWSCredentials credentials = new BasicAWSCredentials(this.accessKey, this.secretKey);
            final ClientConfiguration cfg = new ClientConfiguration().withProtocol(Protocol.HTTPS);
            final AmazonS3 s3 = AmazonS3ClientBuilder.standard().withCredentials(new AWSStaticCredentialsProvider(credentials))
                    .withClientConfiguration(cfg)
                    .withEndpointConfiguration(new AwsClientBuilder.EndpointConfiguration(this.s3EndPoint, S3_REGION))
                    .build();
            if (s3 == null)
                throw new ObjectStoreServiceException("Cannot initialize s3 client because is NULL");
            return s3;
        } catch (Throwable e) {
            log.error("An Error happen on initialize client ", e);
            throw new ObjectStoreServiceException("Cannot initialize s3 client", e);
        }
    }


    public static String retrieveURL(final String baseURI, final String basePath, final String objectStoreId, final String objectId)
            throws UnsupportedEncodingException {
        final StringBuilder sb = new StringBuilder(baseURI)
                .append("?objectStore=" + encode(objectStoreId))
                .append("&objectId=" + encode(objectId))
                .append("&basePath=" + encode(basePath));
        return sb.toString();
    }

    private static String encode(final String s) throws UnsupportedEncodingException {
        return URLEncoder.encode(s, "UTF-8");
    }

    /**
     *
     * @param res
     * @param objectStoreId
     * @param objectId
     * @throws IOException
     * @throws ObjectStoreServiceException
     */
    @RequestMapping(value = "/**/objectStore/retrieve.do")
    public void retrieve(final HttpServletResponse res,
                         @RequestParam(value = "objectStore", required = true) final String objectStoreId,
                         @RequestParam(value = "objectId", required = true) final String objectId) throws IOException, ObjectStoreServiceException {

        final long start = System.currentTimeMillis();
        if (client == null)
            client = initializeClient();
        final S3Object object = this.client.getObject(objectStoreBucket, objectStoreId + "/" + objectId);



        if (object == null) {
            final String msg = String.format("Object with identifier: %s not found the in %s", objectId, objectStoreId);
            res.sendError(HttpServletResponse.SC_NOT_FOUND, msg);
            log.warn(msg);
        } else {
            try ( final InputStream is =  object.getObjectContent()) {

                res.setHeader("Content-Length", String.valueOf(object.getObjectMetadata().getContentLength()));
                IOUtils.copyLarge(is, res.getOutputStream());
                if (log.isDebugEnabled()) {
                    log.debug(String.format("retrive.do completed in %s, objId: %s", HumanTime.exactly(System.currentTimeMillis() - start), objectId));
                }
            } catch (IOException e) {
                final String msg = "unable to close input Stream";
                res.sendError(HttpServletResponse.SC_INTERNAL_SERVER_ERROR, msg);
                log.error(msg, e);
            }
        }
    }
}