package eu.dnetlib.data.objectstore.s3;

import com.mongodb.BasicDBObject;
import com.mongodb.client.MongoCollection;
import com.mongodb.client.MongoDatabase;
import com.mongodb.client.model.Filters;
import com.mongodb.client.model.IndexModel;
import com.mongodb.client.result.DeleteResult;
import com.mongodb.client.result.UpdateResult;
import eu.dnetlib.data.objectstore.modular.connector.ObjectStore;
import eu.dnetlib.data.objectstore.modular.connector.ObjectStoreDao;
import eu.dnetlib.data.objectstore.rmi.ObjectStoreFileNotFoundException;
import eu.dnetlib.data.objectstore.rmi.ObjectStoreServiceException;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.bson.Document;
import org.bson.conversions.Bson;
import org.springframework.beans.factory.annotation.Value;

import javax.annotation.Resource;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;

public class S3ObjectStoreDao implements ObjectStoreDao {
    public static final String INTERPRETATION_FIELD = "interpretation";
    public final static String OBJECTSTORE_METADATA_NAME_FIELD = "metadataObjectStore";
    public final static String OBJECTSTORE_ID_FIELD = "obsId";
    private static final String OBJECTSTORE_PROFILE_SUFFIX = "_T2JqZWN0U3RvcmVEU1Jlc291cmNlcy9PYmplY3RTdG9yZURTUmVzb3VyY2VUeXBl";
    private static final Log log = LogFactory.getLog(S3ObjectStoreDao.class);

    private static final List<IndexModel> metadataIndexes = Arrays.asList(new IndexModel(new Document().append("id",1)), new IndexModel(new Document().append("timestamp",1)));

    @Resource(name="objectstoreMongoDB")
	private MongoDatabase db;

    @Value("${dnet.objectStore.s3.accessKey}")
    private String accessKey;


    @Value("${dnet.objectStore.s3.secretKey}")
    private String secretKey;

    @Value("${dnet.objectStore.s3.endPoint}")
    private String s3EndPoint;

    @Value("${dnet.objectStore.s3.objectStoreBucket}")
    private String objectStoreBucket;

    @Value("${dnet.objectStore.s3.basePath}")
    private String objectStoreBasePath;


	private Document getObjectStoreMetadata(final String objectStoreId) {
        String find_id = objectStoreId;
        if (objectStoreId.length() == 36) {
            find_id += OBJECTSTORE_PROFILE_SUFFIX;
        }
        final MongoCollection<Document> metadataObjectStore = getDb().getCollection(OBJECTSTORE_METADATA_NAME_FIELD);
        final Bson query = Filters.eq(OBJECTSTORE_ID_FIELD, find_id);
        log.debug("QUERY :" + query.toString());
        final Document resultQuery = metadataObjectStore.find(query).first();
        log.debug("result " + resultQuery);
        return resultQuery;
    }

    @Override
    public ObjectStore getObjectStore(final String objectStoreId) throws ObjectStoreServiceException {
        if (StringUtils.isBlank(objectStoreId)) throw new ObjectStoreServiceException("Error on getting ObjectStore, id is Blank");
        final Document resultQuery = getObjectStoreMetadata(objectStoreId);
        if ((resultQuery == null)) throw new ObjectStoreFileNotFoundException("the objectStore with identifier: "+objectStoreId+" was not found");
        final MongoCollection<Document> collection = getDb().getCollection(objectStoreId.substring(0,36));
        collection.createIndexes(metadataIndexes);
        return new S3ObjectStore(resultQuery.getString(OBJECTSTORE_ID_FIELD),resultQuery.getString(INTERPRETATION_FIELD),accessKey, secretKey, s3EndPoint, objectStoreBucket, collection, objectStoreBasePath);
    }

    @Override
    public List<String> listObjectStores() {
        MongoCollection<Document> metadata = getDb().getCollection(OBJECTSTORE_METADATA_NAME_FIELD);
        Iterable<Document> tmp = () -> metadata.find().iterator();
        return StreamSupport.stream(tmp.spliterator(), false).map(it-> it.getString(OBJECTSTORE_ID_FIELD)).collect(Collectors.toList());
    }


    @Override
    public boolean createObjectStore(final String obsId, final String interpretation, final String basePath) throws ObjectStoreServiceException {
        if (getObjectStoreMetadata(obsId)!= null)
            throw new ObjectStoreServiceException("Error unable to create an ObjectStore that already exists in mongo");
        final MongoCollection<Document> metadata = getDb().getCollection(OBJECTSTORE_METADATA_NAME_FIELD);
        final Document item = new Document()
                .append(OBJECTSTORE_ID_FIELD, obsId)
                .append(INTERPRETATION_FIELD, interpretation);
        metadata.insertOne(item);
        MongoCollection<Document> objectStore = getDb().getCollection(obsId.substring(0, 36));
        objectStore.createIndex(new BasicDBObject("id", 1));
        objectStore.createIndex(new BasicDBObject("timestamp", 1));
        return true;
    }

    @Override
    public boolean updateObjectStore(final String obsId, final String interpretation) {
        MongoCollection<Document> coll = getDb().getCollection(OBJECTSTORE_METADATA_NAME_FIELD);
        final Document update = new Document().append("$set", new Document(INTERPRETATION_FIELD, interpretation));
        final UpdateResult updateResult = coll.updateOne(Filters.eq(OBJECTSTORE_ID_FIELD, obsId), update);
        if (updateResult.isModifiedCountAvailable()) {
            log.debug("Matched / Modified " + updateResult.getMatchedCount() + " / " + updateResult.getModifiedCount());
        }
        return true;
    }

    @Override
    public boolean deleteObjectStore(String obsId) throws ObjectStoreServiceException {
        final Document objectStoreMetadata = getObjectStoreMetadata(obsId);
        if (objectStoreMetadata== null)
            throw new ObjectStoreServiceException("ObjectStore not found with Identifier "+obsId);
        log.debug("Start to deleting all the object on the ObjectStore in teh bucket");
        getObjectStore(obsId.substring(0,36)).dropContent();
        log.debug("All object Deleted");
        log.debug("Deleting mongo collection");
        MongoCollection<Document> objectStoreCollection = db.getCollection(objectStoreMetadata.getString(OBJECTSTORE_ID_FIELD));
        objectStoreCollection.drop();
        log.debug("Deleting item on mongo metadata Collection");
        final MongoCollection<Document> metadata = getDb().getCollection(OBJECTSTORE_METADATA_NAME_FIELD);
        DeleteResult deleteResult = metadata.deleteOne(Filters.eq(OBJECTSTORE_ID_FIELD, obsId));
        if (deleteResult.getDeletedCount() != 1)
            throw new ObjectStoreServiceException("Unexpected number of Deleting object on ObjectStoreMetadata, should be 1 instead of"+deleteResult.getDeletedCount());
        return true;
    }

    @Override
    public boolean dropContent(String obsId) throws ObjectStoreServiceException {
        return getObjectStore(obsId).dropContent();
    }

    public MongoDatabase getDb() {
        return db;
    }

    public void setDb(MongoDatabase db) {
        this.db = db;
    }
}
