package eu.dnetlib.lbs.controllers;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;

import org.apache.lucene.queryparser.flexible.standard.QueryParserUtil;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import com.google.common.collect.Lists;

import eu.dnetlib.lbs.LiteratureBrokerServiceConfiguration;
import eu.dnetlib.lbs.cron.ScheduledActions;
import eu.dnetlib.lbs.elasticsearch.Event;
import eu.dnetlib.lbs.elasticsearch.EventRepository;
import eu.dnetlib.lbs.elasticsearch.EventStatsManager;
import eu.dnetlib.lbs.elasticsearch.EventStatsManager.BrowseEntry;
import eu.dnetlib.lbs.subscriptions.Subscription;
import eu.dnetlib.lbs.subscriptions.SubscriptionRepository;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;

@RestController
@RequestMapping("/api/events")
@Api(tags = LiteratureBrokerServiceConfiguration.TAG_EVENTS)
public class EventsController extends AbstractLbsController {

	@Autowired
	private EventRepository eventRepository;

	@Autowired
	private SubscriptionRepository subscriptionRepo;

	@Autowired
	private EventStatsManager eventStatsManager;

	@Autowired
	private ScheduledActions scheduledActions;

	@ApiOperation("Return an event by ID")
	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	public Event getEvent(@PathVariable final String id) {
		return this.eventRepository.findById(id).get();
	}

	@ApiOperation("Delete an event by ID")
	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	public void deleteEvent(@PathVariable final String id) {
		this.eventRepository.deleteById(id);
	}

	@ApiOperation("Save an event by ID")
	@RequestMapping(value = "/{id}", method = RequestMethod.POST)
	public Event saveEvent(@RequestBody final Event event) {
		return this.eventRepository.save(event);
	}

	@ApiOperation("Return a page of events")
	@RequestMapping(value = "/list/{page}/{pageSize}", method = RequestMethod.GET)
	public List<Event> events(
			@PathVariable final int page,
			@PathVariable final int pageSize) {
		return Lists.newArrayList(this.eventRepository.findAll(PageRequest.of(page, pageSize)));
	}

	@ApiOperation("Return a page of events by topic")
	@RequestMapping(value = "/byTopic/{page}/{pageSize}", method = RequestMethod.GET)
	public List<Event> eventsByTopic(
			@PathVariable final int page,
			@PathVariable final int pageSize,
			@RequestParam final String topic) {
		return Lists.newArrayList(this.eventRepository.findByTopic(QueryParserUtil.escape(topic), PageRequest.of(page, pageSize)));
	}

	@ApiOperation("Delete all the events")
	@RequestMapping(value = "/all", method = RequestMethod.DELETE)
	public Map<String, Object> clearEvents() {
		this.eventRepository.deleteAll();
		final Map<String, Object> res = new HashMap<>();
		res.put("deleted", true);
		return res;
	}

	@ApiOperation("Delete the expired events")
	@RequestMapping(value = "/expired", method = RequestMethod.DELETE)
	public Map<String, Long> deleteExpiredEvents() {
		final Map<String, Long> res = new HashMap<>();
		res.put("deleted", this.scheduledActions.deleteExpiredEvents());
		return res;
	}

	@ApiOperation("Return the topics of the indexed events (all)")
	@RequestMapping(value = "/topics/all", method = RequestMethod.GET)
	public List<BrowseEntry> browseTopics() {
		return this.eventStatsManager.browseTopics();
	}

	@ApiOperation("Return the topics of the indexed events (only with subscriptions)")
	@RequestMapping(value = "/topics/withSubscriptions", method = RequestMethod.GET)
	public List<BrowseEntry> browseTopicsWithSubscriptions() {

		final Iterable<Subscription> iter = this.subscriptionRepo.findAll();

		final Set<String> validTopics = StreamSupport.stream(iter.spliterator(), false)
				.map(Subscription::getTopic)
				.collect(Collectors.toSet());

		return this.browseTopics()
				.stream()
				.filter(e -> validTopics.contains(e.getValue()))
				.collect(Collectors.toList());
	}

}
