package eu.dnetlib.lbs.events.output;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Date;
import java.util.Properties;
import java.util.stream.Collectors;

import javax.mail.Authenticator;
import javax.mail.Message;
import javax.mail.PasswordAuthentication;
import javax.mail.Session;
import javax.mail.Transport;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeMessage;

import org.antlr.stringtemplate.StringTemplate;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.Resource;
import org.springframework.stereotype.Component;

import com.google.common.base.Splitter;

import eu.dnetlib.lbs.elasticsearch.Event;
import eu.dnetlib.lbs.properties.EmailProperties;
import eu.dnetlib.lbs.subscriptions.NotificationMode;
import eu.dnetlib.lbs.subscriptions.Subscription;

@Component
public class EmailDispatcher extends AbstractNotificationDispatcher<Message> {

	@Autowired
	private EmailProperties props;

	@Value("${lbs.mail.message.template}")
	private Resource emailTemplate;

	private static final int MAX_NUMEBER_OF_EVENTS = 20;

	private static final Log log = LogFactory.getLog(EmailDispatcher.class);

	@Override
	protected Message prepareAction(final Subscription subscription, final Event... events) throws Exception {

		if (subscription == null || StringUtils.isBlank(subscription.getSubscriber())) {
			log.warn("Invalid subscription");
			throw new IllegalArgumentException("Invalid subscription");
		}
		if (events.length == 0) {
			log.warn("Event list is empty");
			throw new IllegalArgumentException("Event list is empty");
		}

		final String topics = Arrays.stream(events).map(e -> e.getTopic()).distinct().collect(Collectors.joining(", "));

		final Session session = Session.getInstance(obtainProperties(), obtainAuthenticator());

		final MimeMessage mimeMessage = new MimeMessage(session);
		mimeMessage.setFrom(new InternetAddress(this.props.getFrom(), this.props.getFromName()));
		mimeMessage.setSubject("Notification for topic(s): " + topics);
		mimeMessage.setContent(generateMailContent(subscription, events), "text/html; charset=utf-8");
		mimeMessage.setSentDate(new Date());

		mimeMessage.addRecipient(Message.RecipientType.TO, new InternetAddress(subscription.getSubscriber()));

		if (StringUtils.isNotBlank(this.props.getCc())) {
			for (final String aCC : Splitter.on(",").omitEmptyStrings().trimResults().split(this.props.getCc())) {
				mimeMessage.addRecipient(Message.RecipientType.CC, new InternetAddress(aCC));
			}
		}

		log.info("Created mail, to: " + subscription.getSubscriber());

		return mimeMessage;
	}

	private String generateMailContent(final Subscription subscription, final Event... events) throws IOException {
		final StringTemplate st = new StringTemplate(IOUtils.toString(this.emailTemplate.getInputStream(), StandardCharsets.UTF_8));
		st.setAttribute("sub", subscription);

		st.setAttribute("total", events.length);
		st.setAttribute("max", MAX_NUMEBER_OF_EVENTS);

		if (events.length > MAX_NUMEBER_OF_EVENTS) {
			st.setAttribute("events", Arrays.copyOfRange(events, 0, MAX_NUMEBER_OF_EVENTS));
		} else {
			st.setAttribute("events", events);
		}
		return st.toString();
	}

	@Override
	protected void performAction(final Message message) throws Exception {
		log.info("Sending mail to " + Arrays.toString(message.getAllRecipients()) + "...");
		Transport.send(message);
		log.info("...sent");
	}

	private Properties obtainProperties() {
		final Properties p = new Properties();
		p.put("mail.transport.protocol", "smtp");
		p.put("mail.smtp.host", this.props.getSmtpHost());
		p.put("mail.smtp.port", this.props.getSmtpPort());
		p.put("mail.smtp.auth", Boolean.toString(StringUtils.isNotBlank(this.props.getSmtpUser())));
		return p;
	}

	private Authenticator obtainAuthenticator() {
		if (StringUtils.isBlank(this.props.getSmtpUser())) { return null; }

		return new Authenticator() {

			private final PasswordAuthentication authentication =
					new PasswordAuthentication(EmailDispatcher.this.props.getSmtpUser(), EmailDispatcher.this.props.getSmtpPassword());

			@Override
			protected PasswordAuthentication getPasswordAuthentication() {
				return this.authentication;
			}

		};
	}

	@Override
	public NotificationMode getMode() {
		return NotificationMode.EMAIL;
	}

}
